{****************************************************************************

  Name      : ZIP File System
  Version   : 1.0b3
  Part Of   : DIABOLO Game Engine
  Compiler  : FPC v0.99.x (Go32v2, Win32)
  Tested on : FPC v0.99.11 (Go32v2, Win32) - P133 32Mb, Dos6.22, Win'98
  Date      : 24.FEB.99
  Author    : Ivn Montes Velencoso <drslump@axis.org>
  Purpose   : Handling of files from disk or from a ZIP.
  Updates   : My homepage : http://members.tripod.com/imontes/
              Free Pascal home page (only releases, not bugfixes)

  LICENCE   : Take a look at the documentation

  BUGS      : * The reset function isn't 100% compatible with the RTL,
              with Typed Files it needs the size of a componnet as second
              parameter unlike pascal RTL.

  History   : 24-FEB-99 : Speed-up on the unzipping routines (FLUSH)
              12-FEB-99 : Fixed a bug in the BlockRead function that not
              returned the bytes read for compressed files
              12-DEC-98 : Fixed the IOResult function
              09-DEC-98 : Fixed the EOF function
              05-DEC-98 : Some bug fixes on BlockRead
              02-DEC-98 : Changed some LongInt by DWord and some string by
              pchar. Also changed the Boolean by LongBool.
              Time and Date fields now supported correctly.
              30-NOV-98 : Optimized the sequencial access to big files.
              20-NOV-98 : Read function supported
              17-NOV-98 : Long File Name support.
              13-NOV-98 : Now it supports all methods by the use of an
              implementation of the InfoZIP unzip sources. The system is
              completly rebuild and a 140% faster and more flexible and bug
              free than ever.
              07-NOV-98 : User defined unzipping program for unsuported
              methods.
              03-AUG-98 : DirMode added (see documentation)
              02-AUG-98 : Unzipping code ripped into external files for easy
              management and optimization.
              01-AUG-98 : Lots of changes, now is more flexible and useable.
              Crc codes no longer supported. RecSize now works OK.
              26-JUL-98 : Corrected a bug with the Win32 version
              11-JUL-98 : Full working Win32 version :)
              10-JUL-98 : Replaced the unportable MEM arrays by MOVEs
              03-JUL-98 : TP and BP no longer supported.
              Now it uses the FILE type instead of the ZFSfile !.
              25-JUN-98 : Nothing, becouse is my birthday and I've been
              taking soft drugs for all the day ;-)
              16-JUN-98 : Fixed a bug in the BlockRead function that didn't
              check if the file position was out of boundaries.
              15-JUN-98 : Clean Up the code and improved TP/BP support.
              11-JUN-98 : Fixed a minor bug in the BlockRead function
              10-JUN-98 : UpLoaded to my homepage!!
              09-JUN-98 : Full featured Blockread
              06-JUN-98 : Now works with compressed ZIP files (methods 0 & 1)
              instead of PACK files. Not Huffman's method (8) yet.
              24-MAY-98 : Support for Quake PACK files


  Future    : -Better error support
              -Text files support
              -Implement more systems (Os2, linux ...)

  Credits   : InfoZIP, Christian Ghisler, Mark Adler, Dr. Abimbola Olowofoyeku
              and Peter Vreman for the uncompression routines.

  Greetings : to the whole FearMinds team
              to the FPC mailing lists users
              to the FPC developers
              to everyone that thinks that the good if it's free twice good.


 A brief explanation :

    With this unit you can access to files from the disk or files packed
    inside ZIP files with a total transparency to the programmer.
    For do that this unit uses a set of modificated routines based on the
    pascal standard file handling functions : ASSIGN, CLOSE, BLOCKREAD,
    SEEK, EOF ...


 Notes:
   When a file is inside a ZIP file its value "filerec.mode" is fmZipped
   The pointer to the extra data needed by the packed files is stored in
   the first four bytes of "filerec.UserData". I think that in Linux this
   array is used for something like "pipes" but I don't know what it is.
   BlockWrite returns the error 101 if we try to write in a zipped file.


IOResult Values
---------------

0      : All went O.K.

1..255 : The same as IOResult of the Pascal Run Time Library

300    : The Zip file pointer is NIL and can't be erased
301    : Unable to create a new node on the zipped files linked list
302    : Unable to create a new node on ZIP linked list
303    : The file record isn't initialized (assign+reset)
304    : Error seeking a packed file : Out of boundaries
305    : --------------UNUSED ERROR--------------
306    : Missing or invalid local file header in Zip file
307    : Filename of a compressed file exceeds 255 chars!
308    : Extra field of a compressed file exceeds 255 chars!
309    : Write error while decompressing
310    : Read error while decompressing
311    : Error in Zip file
312    : Compression method not supported
65535  : The reset function for typed files doesn't work in the same way that
         in the RTL. See the documentation.


*****************************************************************************}

{$I DEFINES.INC}

Unit ZFS;

{$MODE OBJFPC}  // -s2  (Delphi 2 extensions ON)

INTERFACE

type
  tFileZIP = record {Packed Files info}
    Nom   :pchar;       {file name}
    Kind  :byte;        {0:Store, 1:LZW, 2:..., 8:HUFFMAN}
    Pos   :DWord;       {file position in the ZIP}
    Size  :DWord;       {file size}
    USize :DWord;       {Uncompressed size}
    time  :DWord;     {Packed date+time struct}
  end; {tfilepack}

  pMEMDir = ^tMEMDir;
  tMEMDir = record
    Obj  :tFileZIP;            {Packed file information}
    next :pMEMDir;             {Next packed file}
  end; {tMEMDir}


  pZIP = ^tZIP;
  tZIP = record
    Active      :LongBool;     {Is this active?}
    Name        :String;       {file name .ZIP}
    zfile       :file;         {handle for the zip files}
    DirMode     :LongBool;     {Normalmode or Dirmode}
    NumFiles    :DWord;        {number of files zipped}
    FirstMEMDir :pMEMDir;      {Index of the files linked list}
    Next        :pZIP;         {Next ZIP file (NIL if none)}
  end; {tpack}


  pZFSfile= ^tZFSfile;
  tZFSfile = record
    ZIP     :pZIP;             {ZIP where it's stored}
    comp    :longint;          {Type of compression}
    pos     :DWord;            {position in case it is on a ZIP}
    size    :DWord;            {size in case it is on a ZIP}
    usize   :DWord;            {size of the uncompressed archive}
    curr    :DWord;            {Current position in the file}
  end; {24 bytes}



const
  FirstZIP  :pZIP = NIL;    {Index of the ZIP archives linked list}

  ZFSResult :DWord = 0;     {Variable with the result of a function (0=O.K.)}

  fmZipped  =$D7B4; {fmClosed=$D7B0
                     fmInput =$D7B1
                     fmOutput=$D7B2
                     fmInOut =$D7B3 -> Those are standard in Pascal}

  NormalMode =FALSE;        {opens a ZIP in normal mode}
  DirMode    =TRUE;         {opens a ZIP in directory mode}


function  FormatFileName(s:string) : string;
{Converts a case sensitive UNIX file name to a DOS/Win32 upcase one. '/'->'\'}

Function  OpenZIP (ZIPName :string; Mode:LongBool ):pZIP;
{Opens a ZIP}
Procedure CloseZIP(zipf:pZip);
{Closes a ZIP}
Procedure KillZIPs;
{Closes all the oppened ZIPs}


function  IOResult :DWord;
{Returns the status of the last I/O operation performed}

function  FileSize(var f :file):DWord;
{Returns the size of the file 'f' in number of components}

function  FileExist( sExp :STRING ):LongBool;
{Returns TRUE if the file or directory 'sExp' exists}

function  FilePos( var f :file ):DWord;
{Returns the current file possition of 'f'}

Procedure Assign(var f:file; fname:string);
{Assigns the name of an external file to a file variable}
Procedure Reset(var f:file; l:DWord); {recsize=l}
Procedure Reset(var f:file); {recsize=128}
Procedure Reset(var f:typedfile); {BUGGY!!}
{Opens an existing file}

Procedure Close(var f:file);
{Closes an open file}
procedure seek(var f:file; N:DWord);
{Changes the current possition of a file to a specified possition}
Function  EOF(f:file):LongBool;
{Returns true if the file have reached the end}
procedure BlockRead(var F :file; var Buf; Count :DWord; var readed:DWord);
procedure BlockRead(var F :file; var Buf; Count :DWord);
{Read from a file, returns the bytes readed in result}

procedure Read (var f :typedfile; var v1);
procedure Read (var f :typedfile; var v1,v2);
procedure Read (var f :typedfile; var v1,v2,v3);
procedure Read (var f :typedfile; var v1,v2,v3,v4);
procedure Read (var f :typedfile; var v1,v2,v3,v4,v5);
{Reads from a typed file}

IMPLEMENTATION

uses
  DOS,     {Some file handling routines}
  STRINGS, {Null terminated strings handling routines}
  UNZIP    {tweaked InfoZIP's uncompression routines}
  ;

var
  OldExitProc :Pointer;     {Stores a temporal ExitProc}

type
  Local_File_Header_Type = packed record
    Extract_Version_Reqd   :Word;
    Bit_Flag               :Word;
    Compress_Method        :Word;
    Last_Mod_Time          :Word;
    Last_Mod_Date          :Word;
    Crc32                  :LongInt;
    Compressed_Size        :LongInt;
    Uncompressed_Size      :LongInt;
    Filename_Length        :Word;
    Extra_Field_Length     :Word;
  end;

var
  LocalHdr      :Local_File_Header_Type;  { temp. var. for a local file header }
  Hdr_FileName  :string; {temporal strings}
  Hdr_ExtraField:string;
  {Hdr_Comment   :string;}

{ Define the ZIP file header types }
const
  LOCAL_FILE_HEADER_SIGNATURE   = $04034B50;
  CENTRAL_FILE_HEADER_SIGNATURE = $02014B50;
  END_OF_CENTRAL_DIR_SIGNATURE  = $06054B50;


function IOResult :DWord;
var a:word;
begin
  a:=System.IOResult;
  if a<>0 then IOResult:=a
          else IOResult:=ZFSResult;
  ZFSResult:=0;
end;


{iNITIALIZES a zIP oBJECT}
Procedure InitZO(fname:string; var zo :tZIP);
var res:longint;
begin
  {oppening the ZIP file}
  system.assign( zo.zFile, fName);
  system.FileMode := 64;  { Read Only }
  {$I-}system.reset(zo.zFile, 1);{$I+}
  res:=system.IOResult;
  if res<>0 then
   begin
     ZFSResult:=res;
     exit;
   end;
  {Initializing ZIP object variables}
  zo.Active:=true;
  zo.Name := fname;
  zo.FirstMemDir:=NIL;
end;

{Closes a ZIP file}
procedure CloseZO(var zo:tZIP);
begin
  {$I-}System.Close(zo.zFile);{$I+}
  ZFSResult:=System.IOResult;
  zo.Active:=false;
end;

{Gets a local header of a zipped file}
function Read_Local_Hdr(var zo:tZip) : LongBool;
var
  Sig : longInt;
begin
  ZFSResult:=0;

  BlockRead(zo.zFile, sig, SizeOf(Sig));
  if Sig = CENTRAL_FILE_HEADER_SIGNATURE then
   begin
     Read_Local_Hdr := false;
   end {then}
  else
   begin
     if Sig <> LOCAL_FILE_HEADER_SIGNATURE then
      begin
        ZFSResult:=306;  {Missing or invalid local file header in Zip file}
        EXIT;
      end;
     BlockRead(zo.zFile, LocalHdr, SizeOf(LocalHdr));
     with LocalHdr do
      begin
        if FileName_Length > 255 then
         begin
           ZFSResult:=307; {Filename of a compressed file exceeds 255 chars!}
           EXIT;
         end;
        BlockRead(zo.zFile, Hdr_FileName[1], FileName_Length);
        Hdr_FileName[0] := Chr(FileName_Length);
        if Extra_Field_Length > 255 then
         begin
           ZFSResult:=308; {Extra Field of a compressed file exceeds 255 chars!}
           EXIT;
         end;
        BlockRead(zo.zFile, Hdr_ExtraField[1], Extra_Field_Length);
        Hdr_ExtraField[0] := Chr(Extra_Field_Length);
      end {with};
     Read_Local_Hdr := true;
   end {else};
end;


{nORMALIZE a fILENAME for the routines - Uppercase with inverted slayer '\'}
function FormatFileName(s:string) : string;
var l:string; i:integer;
begin
  l:=s;
  for i:=1 to Length(s) do
   begin
     if s[i]='/' then l[i]:='\'
                 else l[i]:=UpCase(s[i]);
   end;
  FormatFileName:=l;
end;


{aDDS a nODE aT tHE eND oF tHE lINKED lIST oF tHE .zIP's dIRECTORY}
Procedure AddMEMDir(ZIPf:pZIP; data:tFileZIP);
var
  tmp, tmp2 :pMEMDir;
begin
  ZFSResult:=0;

  new(Tmp);
  if Tmp=NIL then
   begin
     ZFSResult:=301;  {Unable to create a new node on the packed files linked list}
     exit;
   end;

  if ZIPf^.FirstMemDir=NIL then {if the list is empty...}
   begin
     ZIPf^.FirstMemDir:=Tmp;
     with ZIPf^.FirstMemDir^ do
      begin
        next:=NIL;
      end;
   end
  else
   begin
     tmp2:=ZIPf^.FirstMemDir;
     while tmp2^.next<>NIL do   {searching for the last item in the list}
      begin
        tmp2:=tmp2^.next;
      end;
     tmp2^.next:=tmp;
     tmp^.next:=NIL;
   end;

  with tmp^ do   {Fills the list item with the file information}
   begin
     move(data, obj, sizeof(obj));
   end;
end;


{eRASE tHE wHOLE lINKED lIST oF a .zIP dIRECTORY}
Procedure KillMEMDir(ZIPf:pZIP);
var
  tmp,
  tmp2 :pMEMDir;

begin
  tmp:=ZIPf^.FirstMemDir;
  if tmp=NIL then {there isn't nothing to erase}
   begin
     exit;
   end
  else
   begin
     While tmp<>NIL do
      begin
        tmp2:=tmp^.NEXT;
        FreeMem(tmp^.Obj.Nom, StrLen(tmp^.Obj.Nom));
        Dispose(tmp);
        tmp:=Tmp2;
      end;
   end;
  ZIPf^.FirstMemDir:=NIL;
end;

{aDDS a nODE tO tHE eND oF tHE zIP fILES lINKED lIST}
Function AddZIP :pZIP;
var
  tmp, tmp2 :pZIP;
begin
  ZFSResult:=0;

  AddZIP:=NIL;
  new(Tmp);
  if Tmp=NIL then
   begin
     ZFSResult:=302;  {Unable to create a new node on ZIP linked list}
     exit;
   end;

  if FirstZIP=NIL then     {If it's the first ZIP oppened}
   begin
     FirstZIP:=Tmp;
     FirstZIP^.next:=NIL;
   end
  else
   begin
     tmp2:=FirstZIP;
     while tmp2^.next<>NIL do    {Searching for the last item in the list}
      begin
        tmp2:=tmp2^.next;
      end;
     tmp^.next:=NIL;
     tmp2^.next:=tmp;
   end;
  AddZIP:=tmp;
end;

{eRASES a nODE fROM tHE zIPS lINKED lIST}
Procedure EraseZIP(ZIPf :pZIP);
Var
  tmp :pZIP;  { Used to move through the list. }
begin
  if ZIPf = nil then
   begin
     {The pointer is NIL, can not be erased}
     exit;
   end;

  if FirstZIP = nil then   { Is the list empty? }
   begin
     {No nodes to be delete in this linked list}
     exit;
   end
  else
   begin
     if FirstZIP=ZIPf then
      begin
        FirstZIP:=ZIPf^.next;
        Dispose(ZIPf);
        exit;
      end;
   end;

  tmp := FirstZIP;
  While tmp^.Next <> ZIPf do
   begin
     tmp := tmp^.Next;
     if tmp=NIL then
      begin
        {node not found in this linked list}
        exit;
      end;
   end;
  tmp^.Next := ZIPf^.Next;       { Point around old link }
  Dispose(ZIPf);              { Get rid of the link }
end;

{oPENS a zIP aRCHIVE aND iNITIALIZE iT}
Function OpenZIP (ZIPName :string; Mode:LongBool ):pZIP;
var
  ZIPf:pZIP;
  dat:tfileZip;
  zdir, zname, zext :string;
begin
{$I-}
  ZFSResult:=0;
  OpenZIP:=NIL;

  ZIPf:=AddZIP;
  if ZFSResult<>0 then Exit;

  InitZO(ZipName, ZIPf^);
  if ZFSResult<>0 then
   begin
     EraseZIP(ZIPf);
     Exit;
   end;

  FSplit(ZIPName, Zdir, Zname, Zext);

  ZIPf^.Active:=true;
  ZIPf^.Name:=Zname+Zext;
  ZIPf^.DirMode:=Mode;

  ZIPf^.NumFiles:=0;
  ZIPf^.FirstMemDir:=NIL;
  while Read_Local_Hdr(ZIPf^) do
   begin
     inc(ZIPf^.NumFiles,1);

     if Mode then
      begin
        Zname:=Zname+'\'+Hdr_FileName;
        GetMem(dat.Nom, sizeof(zname));
        StrPCopy(dat.Nom, Zname);
      end
     else
      begin
        GetMem(dat.Nom, sizeof(Hdr_FileName));
        StrPCopy(dat.Nom, Hdr_FileName);
      end;

     dat.Kind:=LocalHdr.Compress_Method;
     dat.pos:=system.FilePos(ZIPf^.zfile);
     dat.Size:=LocalHdr.Compressed_Size;
     dat.USize:=LocalHdr.UnCompressed_Size;
     dat.Time:=(LocalHdr.Last_Mod_Date SHL 16)+LocalHdr.Last_Mod_Time;

     addMemDir(ZIPf, dat);

     Seek(ZIPf^.zFile, dat.Pos+dat.Size);
   end;

  OpenZIP:=ZIPf;
{$I+}
end;

{cLOSES a zIP fILE}
Procedure CloseZIP(ZIPf:pZIP);
begin
{$I-}
  ZFSResult:=0;

  if ZIPf=NIL then
   begin
     ZFSResult:=300; {the ZIP file is NIL and can't be erased}
     Exit;
   end;

  KillMEMDir(ZIPf); {Kills the list of files}
  CloseZO(ZIPf^);   {Closes a ZIP Object}
  EraseZIP(ZIPf);   {Kills the ZIP object from the oppened list}
{$I+}
end;


{lOOKS iF eXIST a fILE or a directory iN tHE dISK oR iN tHE oPPENED zIPS}
FUNCTION FileExist( sExp :STRING ): LongBool;
VAR
  s       :SearchRec;
  tmpZIP  :pZIP;
  tmpDir  :pMEMDir;
BEGIN
{$I-}
  DOS.FindFirst( sExp, Archive+Directory, s );
  if DOSError=0 then
   begin
     FileExist:=true;
     Exit;
   end;

  if FirstZIP=NIL then {No oppened ZIPS}
   begin
     FileExist:=False;
     exit;
   end;

  tmpZIP:=FirstZIP;
  REPEAT

    if tmpZIP^.FirstMemDir<>NIL then {If there are any file in the ZIP}
     begin
       tmpDir:=tmpZIP^.FirstMemDir;
       REPEAT
         if FormatFileName(tmpDir^.Obj.Nom)=FormatFileName(sExp) then
          begin
            FileExist:=TRUE;
            exit;
          end;

         tmpDir:=tmpDir^.Next;
       UNTIL tmpDir=NIL;
     end;

    tmpZIP:=tmpZIP^.next;
  UNTIL tmpZIP=NIL;

  FileExist := False;
{$I+}
END;


Procedure Assign(var f:file; fname:string);
BEGIN
  FillChar(f, SizeOf(FileRec), 0);
  FileRec(f).Handle:=UnusedHandle;
  FileRec(f).mode:=fmClosed;
  Move(fName[1], FileRec(f).Name, Length(fName));
end;


Procedure Reset(var f:file; l:DWord);
VAR
  fname   :string;
  s       :SearchRec;
  tmpZIP  :pZIP;
  tmpDir  :pMEMDir;
  tmpFile :pZFSFile;
  res     :longint;
BEGIN
{$I-}
  ZFSResult:=0;

  fname:=strpas(filerec(f).Name);

  DOS.FindFirst( fname, Archive, s );
  if DOSError=0 then
   begin
     FileRec(f).Handle:=UnusedHandle;
     system.reset(f,l);
     res:=system.IOResult;
     If res<>0 then
      begin
        ZFSResult:=res;
        exit;
      end;

     fillchar( filerec(f).UserData[1], 16, 0 ); {Clear the UserData field}

     Exit;
   end;

  if FirstZIP=NIL then {No oppened ZIPS}
   begin
     ZFSResult:=2; {File not found}
     exit;
   end;

  tmpZIP:=FirstZIP;
  REPEAT

    if tmpZIP^.FirstMemDir<>NIL then {If there are any file in the ZIP}
     begin
       tmpDir:=tmpZIP^.FirstMemDir;
       REPEAT
         if FormatFileName(tmpDir^.Obj.Nom)=FormatFileName(fname) then
          begin
            fillchar( f, sizeof(f), 0 );

            new(tmpFile);

            tmpFile^.ZIP:=tmpZIP;
            tmpFile^.comp:=tmpDir^.Obj.Kind;
            tmpFile^.pos:=tmpDir^.Obj.Pos;
            tmpFile^.size:=tmpDir^.Obj.Size;
            tmpFile^.USize:=tmpDir^.Obj.USize;
            tmpFile^.curr:=0;

            filerec(f).handle:=$FFFF;
            filerec(f).Mode:=fmZipped;
            filerec(f).RecSize:=l;
            move( ofs(tmpfile), filerec(f).UserData[1], 4);
            move( fName[1], FileRec(f).Name, Length(fName));

            exit;
          end;

         tmpDir:=tmpDir^.Next;
       UNTIL tmpDir=NIL;
     end;

    tmpZIP:=tmpZIP^.next;
  UNTIL tmpZIP=NIL;

  ZFSResult:=2; {File not found}
{$I+}
end;

Procedure Reset(var f:file);
begin
  Reset(f, 128);
end;

{BUGGY!!}
Procedure Reset(var f:typedfile);
begin
  {Use Reset(typed file, sizeof(component))}
  ZFSResult:=65535;
end;


Procedure Close(var f:file);
var res:longint;
BEGIN
{$I-}
  ZFSResult:=0;

  if filerec(f).handle=0 then
   begin
     ZFSResult:=303;  {the file record isn't initialized (assign+reset)}
     exit;
   end;

  if filerec(f).Mode<>fmZipped then   {if it's not inside a ZIP...}
   begin
     system.Close(f);
     res:=system.IOResult;
     if res<>0 then
      begin
        ZFSResult:=res;
        exit;
      end;
   end
  else
   begin
     fillchar( f, sizeof(f), #0 );
   end;
{$I+}
end;

var
  pf:pZFSfile; {temporal var for store the address of the pZFSfile data}

procedure seek(var f:file; N:DWord);
begin
{$I-}
  ZFSResult:=0;

  if filerec(f).handle=0 then
   begin
     ZFSResult:=303;  {the file record isn't initialized (assign+reset)}
     exit;
   end;

  if filerec(f).Mode=fmZipped then    {if it's in a ZIP}
   begin
     move(filerec(f).userdata[1], pf, 4);

     if pf^.Comp=0 then
      begin
        if (N>pf^.USize) or (N<0) then
         begin
           ZFSResult:=304;  {Error seeking the packed file : Out of boundaries}
           exit;
         end;

        system.seek(pf^.ZIP^.zfile, pf^.POS+N);
        ZFSResult:=system.IOResult;
        pf^.Curr:=N;
      end
     else
      begin
        if (N>pf^.USize) or (N<0) then
         begin
           ZFSResult:=304;
           exit;
         end;

        pf^.Curr:=N;
      end;
   end
  else
   begin
     system.seek(f, n);
     ZFSResult:=System.IOResult;
   end;
{$I-}
end;


function FilePos(var f :file):DWord;
begin
  move(filerec(f).UserData[1], pf, 4);
  if filerec(f).Mode=fmZipped then
   FilePos:=pf^.Curr
  else
   FilePos:=System.FilePos(f);
end;


function FileSize(var f :file):DWord;
begin
  move(filerec(f).UserData[1], pf, 4);

  if filerec(f).Mode=fmZipped then  {If it's on a ZIP file ...}
   FileSize:=pf^.USize div filerec(f).RecSize
  else                              {If not ...}
   FileSize:=System.FileSize(f);
end;


function ReadFromZip( fil :pZFSfile; dst :pointer; count :DWord) :Longint;
var readed :DWord;
begin
{$I-}
  result:=0;

  if fil^.Curr>=fil^.Usize then EXIT;

  if (fil^.curr+Count)>fil^.USize then
   begin {The requested data exceeds the file limits}
     Count:=fil^.USize-fil^.Curr;
     if count<=0 then exit;
   end;

  case fil^.Comp of
    0 :begin {stored}
         system.Seek(fil^.ZIP^.zfile, fil^.pos+fil^.curr);
         result:=system.IOResult;
         if result<>0 then exit;

         system.BlockRead(fil^.ZIP^.zfile, dst^, Count, Readed);
         result:=system.IOResult;
         if result<>0 then exit;
         fil^.curr:=fil^.curr+Readed;
       end {case|0};
    else
     begin
       zfs_reachedbytes:=0;
       zfs_toreachbytes:=fil^.Curr;
       zfs_neededsize:=Count;
       zfs_outofs:=0;
       zfs_outbuf:=dst;

       result:=unzipfile (fil^.ZIP^.zfile, fil^.pos,
                          fil^.size, fil^.usize, fil^.comp);
       case result of
         unzip_WriteErr     : ZFSResult:=309; {Write error while decompressing}
         unzip_ReadErr      : ZFSResult:=310; {Read error while decompressing}
         unzip_ZipFileErr   : ZFSResult:=311; {Error in Zip file}
         unzip_NotSupported : ZFSResult:=312; {Compression method not supported}
       end;
       if result<>0 then exit;

       fil^.Curr:=fil^.Curr+zfs_neededsize;
       if fil^.Curr>fil^.USize then fil^.Curr:=fil^.Usize;
     end {case|else};
  end {case};
{$I+}
end;


procedure BlockRead(var F :file; var Buf; Count :DWord; var readed :DWord);
var res:longint;
begin
{$I-}
  ZFSResult:=0;

  if Count=0 then EXIT;

  if filerec(f).handle=0 then
   begin
     ZFSResult:=303;  {the file record isn't initialized (assign+reset)}
     exit;
   end;

  if filerec(f).Mode=fmZipped then
   begin
     move(filerec(f).UserData[1], pf, 4);

     res := ReadFromZip(pf, pointer(@buf), Count*filerec(f).RecSize);
     if res<>0 then
      begin
        ZFSResult:=res;
        exit;
      end;

     readed:=zfs_reachedbytes;
   end
  ELSE {Not InZIP}
   begin
     system.BlockRead(f, Buf, Count, Readed);
     res:=system.IOResult;
     if res<>0 then
      begin
        ZFSResult:=res;
        exit;
      end;
   end;
{$I+}
end;

procedure BlockRead(var F :file; var Buf; Count :DWord);
var l:DWord;
begin
  BlockRead(f, buf, count, l);
end;


procedure Read (var f :typedfile; var v1);
begin
  BlockRead(f, v1, 1);
end;

procedure Read (var f :typedfile; var v1,v2);
begin
  BlockRead(f, v1, 1); BlockRead(f, v2, 1);
end;

procedure Read (var f :typedfile; var v1,v2,v3);
begin
  BlockRead(f, v1, 1); BlockRead(f, v2, 1); BlockRead(f, v3, 1);
end;

procedure Read (var f :typedfile; var v1,v2,v3,v4);
begin
  BlockRead(f, v1, 1); BlockRead(f, v2, 1); BlockRead(f, v3, 1);
  BlockRead(f, v4, 1);
end;

procedure Read (var f :typedfile; var v1,v2,v3,v4,v5);
begin
  BlockRead(f, v1, 1); BlockRead(f, v2, 1); BlockRead(f, v3, 1);
  BlockRead(f, v4, 1); BlockRead(f, v5, 1);
end;


Function EOF(f:file):LongBool;
begin
{$I-}
  EOF:=FALSE;

  if filerec(f).handle<>0 then
   begin
     if filerec(f).Mode=fmZipped then
      begin
        move(filerec(f).UserData[1], pf, 4);
        EOF:=(pf^.Curr>=pf^.USize);
      end
     else
      begin
        EOF:=SYSTEM.EOF(f);
      end;
   end
  else
   begin
     ZFSResult:=303; {The file record isn't initialized (assign+reset)}
     exit;
   end;

  ZFSResult:=system.IOResult;
{$I+}
end;


Procedure KillZIPs;
var
  tmp,
  tmp2 :pZIP;
begin
  tmp:=FirstZIP;
  if tmp=NIL then              {there is nothing to close}
   begin
   end
  else
   begin
     While tmp<>NIL do
      begin
        tmp2:=tmp^.NEXT;
        CloseZIP(tmp);
        tmp:=Tmp2;
      end;
   end;
  FirstZIP:=NIL;
end;

{cLOSES aLL tHE oPPENED zIP fILES aUTOMATICLY wHEN wE sTOP tHE pROGRAM}
procedure ZIPExitProc;
begin
  ExitProc:=OldExitProc;        {restore the original exit}
  KillZIPs;
end;


BEGIN
  OldExitProc:=ExitProc;
  ExitProc:=@ZIPExitProc;
END.
