;	Rylan's DJGPP blob conversion demo program.
; 	Based on Code by Tom Hammersley
;	rylan@intekom.co.za
;	http://home.intekom.com/rylan/

;	NASM Assembly module
;  See the BLOBS.C file for info about this small conversion tutorial

	BITS 32	;Tells NASM that this will be a 32bit protected mode program

	GLOBAL	_blob_main	;This is how the BLOBSA.ASM module is linked to BLOBS.C.
	                     ;If you look at BLOBS.C you will see that blob_main()
								;is declared as extern (i. e. "external"). It can
								;thus be called in the C module BLOBS.C, but it is
								;actually a NASM assembly function declared in this
								;file.

	EXTERN	_vid_descr	;This is the video descriptor - it is obtained in the
								;C module by calling a DPMI function. It is then used
								;in this file to access the VGA video RAM directly,
								;just like we do in real mode programming.

	SECTION	.data	;NASM flag - indicates to it that the data section follows

	%include	"blobh.inc"		;This includes tables from Tom Hammersley's original
	%include "blobv.inc"    ;blob program - sinewaves for the vertical and horisontal
	%include "blobpal.inc"  ;movement, and the blob palette

	htableptr	dd 0	;Pointer to the horisontal sinewave table. Note that we
							;use a "dd" i. e. "declare double word". Protected mode
							;has no "NEAR" or "FAR" pointers - every pointer can be
							;considered "NEAR" and all of them are dword sized.
							;Compare this with Tom Hammersley's original TASM ASM
							;file (included in the distribution archive) in which
							;these values are word sized - dead giveaway it is a
							;real mode program.

	vtableptr	dd 0	;Do the same as above, only for the vertical sinewave
							;table

	row_count	dw 0	;This variable was added by me - it is one of the problems
	                  ;of protected mode coding. In the original blob code by
							;Tom Hammersley he used a register to store this value -
							;a register I did not happen to have available, or was not
							;able to keep available due to having already used it to
							;circumvent problems with simulating the real mode code
							;in acceptable protected mode code - so I had to make use
							;of a variable to keep this particular value that he could
							;leave in a register. You might often find yourself running
							;out of registers when converting real mode code to protected
							;mode - this is one way to circumvent this problem.
							;Actuall I did not "use" it - the register Tom Hammersley
							;used was the BP register - he left a loop counter value in
							;it. In protected mode it seems to me that "touching" BP or
							;EBP is a no-no - so I had to use a variable.

	screen_buf	TIMES 64000 db 0 ;Declare the hidden screen buffer. This feature
										  ;was added by me during the conversion. The
										  ;original code was only designed to work in
										  ;screen RAM - which of course slowed it down.
										  ;I made it work in a buffered manner, i. e.
										  ;instead of working in screen RAM all the time,
										  ;the converted program works in an off-screen
										  ;buffer, which is only copied to the screen
										  ;each time the process of generating a new
										  ;full screenframe is done - this greatly speeds
										  ;up the effect. In fact, it goes so fast it loses
										  ;some of its beauty - if you want to slow it down
										  ;again see the rest of this code file.

	SECTION	.text	;NASM flag - indicates to the assembler that the actual
						;code now follows.

_testline	;This is a simple function that can be used to test if the
				;palette was correctly set up.

	push	ebp 		;This is how NASM function entry code looks under DJGPP
	mov	ebp,esp	;You can leave it out, but you will encounter problems if
						;you want to call C library functions from here, for instance
						;'cause you would have ruined the stack frame.
	mov	cx,255
	xor	ebx,ebx
  loopit:
   mov	[es:bx],cx
	inc	bx
   dec	cx
  jnz		loopit

   xor 	ax,ax
	int	16h

	mov	esp,ebp	;This restores the stack frame - between here and the
	pop	ebp		;mov ebp,esp line above it was UNSAFE to attempt to call
						;any C library functions.
	ret

_setvgamode	;This function gets the system display into VGA mode, and clears
				;the off-screen buffer to black. It also sets ES to vid_descr,
				;which means that for the rest of the program we can access
				;VGA 320x200x256 screen memory directly using the ES register -
				;this implies of course that you must keep ES inviolate.

	push	ebp	 	;Store stack frame
	mov	ebp,esp

	mov	ax,13h	;These two lines gets us into 320x200x256 VGA
	int	10h

	cld						;Clear the direction flag
	xor	eax,eax			;Set EAX to 0
	mov	cx,16000			;16000 doublewords in the 64k of 320x200x256 VGA RAM
	mov	edi,screen_buf	;Screen buffer is exactly that size - this puts its
								;address into the EDI register. Note that you can
								;never put the address of a variable in protected mode
								;into any register smaller than an extended 32 bit one -
								;"mov di,screen_buf" (which would've worked under real mode)
								;CANNOT work here. You need doublewords to store
								;pointers in protected mode.
								;Also note that in NASM,
								; mov register,variable
								;means move the ADDRESS of "variable" into "register"
								;and
								; mov register,[variable]
								;means move the VALUE of "variable" into "register"
								;The square brackets [] thus distinghuises between
								;"address" (no []) and value ([])

	rep	stosd	;Use the "repeat store double words" opcode to clear the buffer.

	mov	esi,screen_buf	;Store the address of the screen buffer in the ESI
								;register - note that this conforms to the rule set
								;forth above in that it is a 32 bit register!

	mov	es,[_vid_descr]	;Store the video descriptor into the ES register

	mov	esp,ebp	;Restore stack frame
	pop	ebp
	ret

_settextmode
	push	ebp
	mov	ebp,esp

	mov	ax,03h
	int	10h

	mov	esp,ebp
	pop	ebp
	ret

_setpalette		;This function sets up the palette from the "blobpal.inc"
					;file included in the %include above.
	push	ebp		;Stack frame
	mov	ebp,esp

	mov	dx,3C8h	;03C8h is the hex address of the VGA palette mode register.
	xor	ax,ax		;Setting a 0 to this register with the "out" opcode
						;notifies the VGA that any values you write to 03C9h
						;(which is the hex address of the VGA palette data register)
						;are to be written to the card's palette.
						;Remeber that "xor reg,reg" makes a register get a value
						;of 0 - it is faster than saying "mov reg,0"

	out	dx,ax		;Viola! Send 0 to the palette mode reg to tell the card to
						;expect color values in 03C9
	mov	cx,768	;256 colors, each having 3 bytes (RedGreenBlue) = 768 bytes
   inc	dx			;Set DX to 03C9
	mov	ebx,palette	;Get the address of the "palette" array, declared in
	                  ;"blobpal.inc" into EBX.
							;Note that EBX is 32 bit - attempting
							; mov bx,palette
							;will result in an error at compile time.
  pal_loop:
   mov	al,[ebx]		;Move the value pointed to by the pointer in EBX to
							;the AL register
	out	dx,al			;Send this value to the VGA card palette data port 03C9h
	inc	ebx			;Go the next byte in the palette
	dec	cx				;The CX register is used for looping - decrement it to
							;indicate one iteration of the loop is finished

  jnz		pal_loop		;If CX = 0, end the loop, else loop again.
  							;JNZ = Jump if Not Zero

	mov	esp,ebp	;Restore the stack
	pop	ebp
	ret

_dopixel			;Compare this to the PROC dopixel in BLOBS.ASM - the differences
					;are mainly due to to protected mode "proofing" of the real
					;mode TASM PROC in BLOBS.ASM
	push	ebp
	mov	ebp,esp

	mov	si,[htableptr]	;Ah ha! Though you had me, eh? Yes, SI is a word sized
								;register, so how can I get the address of htableptr
								;into it? No mistake tho - the brackets around htableptr
								;means "value" - in plain English here stands
								;"move the VALUE of htableptr into the processor's
								;SI register"
	mov	bx,cx
	mov	ax,[si]
	add	bx,ax

	mov	si,[vtableptr]
	mov	dx,[row_count]
	mov	ax,[si]
	add	dx,ax

	xor	ah,ah
	add	bx,dx
   ;mov	al,[es:di]		;Uncomment this line by removing the comma before it
								;and then comment the three lines below to see the
								;effect work on the screen only as Tom originally
								;wrote it.

	mov	esi,screen_buf ;These lines write to the buffer - comment these
	add	esi,edi			;three and the three similar ones below,
	mov	al,[esi]			;uncommenting the line above them to see the original
								;unbuffered effect.

	add	bx,ax

	;mov	[es:di],bl 		;Uncomment and recompile for original effect

	mov	esi,screen_buf	;Comment these three for original effect and
	add   esi,edi			;recompile, of course!
	mov	byte[esi],bl

	inc   di

	mov	eax,[vtableptr]
	add	eax,2
	dec	cx
	mov	[vtableptr],eax

	mov	esp,ebp
	pop	ebp
	ret

_wobble_pattern
	push	ebp
	mov	ebp,esp

	mov	eax,htable
	mov	[htableptr],eax
	xor	edi,edi

	mov	cx,200
  row_loop:
	mov	[row_count],cx		;Here is the variable I use instead of the BP
									;register - see the original code in BLOBS.ASM
	mov	eax,vtable			;Move the address of vtable into EAX - not the value!
	mov	[vtableptr],eax
	mov	cx,320
  pixel_loop:
   call	_dopixel
	call	_dopixel
	call	_dopixel
	call	_dopixel
  jnz		pixel_loop

	mov	cx,[row_count]
	dec	cx
  jnz		row_loop;

	mov	esp,ebp
	pop	ebp
	ret

_image_move_wobbles
	push	ebp
	mov	ebp,esp

	push  es
	push	ds
	mov	ax,ds
	mov	es,ax

	cld
	mov	ax,[htable]
	mov   [htable + 400],ax
	mov	esi,htable
	mov	edi,htable
	add	esi,2
	mov   cx,100
	rep   movsd

	mov	ax,[vtable]
	mov	[vtable + 640],ax
	mov	esi,vtable
	mov	edi,vtable
	add	esi,2
	mov   cx,160
	rep	movsd

	pop	ds
	pop	es
	mov	esp,ebp
	pop	ebp
	ret

_show_buf		  	;Show buffer - this copies the hidden buffer which is in
	push	ebp		;memory to the screen
	mov	ebp,esp

	pusha
	cld
	mov	es,[_vid_descr]
	mov	esi,screen_buf
	xor	edi,edi
	mov	cx,16000
	mov	dx,03DAh
  retrace:
   in		al,dx
	test	al,8
  jz		retrace
	rep	movsd
	popa

	mov	esp,ebp
	pop	ebp
	ret

_blob_main
	push	ebp
	mov	ebp,esp

	call	_setvgamode
	call	_setpalette

  frame_loop:
	call	_wobble_pattern
	call	_image_move_wobbles

	call	_show_buf ;If you want the original unbuffered effect, comment
						 ;this line too along with the two sections of three in
						 ;the do_pixel function

	in		al,60h	;Waits for the user to hit escape
	cmp	al,1
  jne    frame_loop

	call	_settextmode

	mov	esp,ebp
	pop	ebp
	ret



