// These packing routines are based on the LZHUF.C program by
// Haruyasu Yoshizaki. Copyright (C) 1995 Christian Worm.

#include <pack.hpp>
#include <packtabl.hpp>
#include <string.h>

#define small(a,b) ((a)>(b) ? (b) : (a))
// Returnerer den mindste af de to

//------------------------------BITWRITE--------------------------------------
// Denne klasse tager sig af at gemme/skrive et vist antal bit.
// Bemrk: putbits(code,bitcount) skriver bitcount bits fra code.

void bitwrite::initwrite() {
  bufbyteofs=bufbitofs=0; memset(buffer,0,FILEBUFLEN);
}

void bitwrite::flushbuf() {
  write(buffer,bufbyteofs+(bufbitofs!=0));
  // Lg en til hvis der er skrevet noget i den nuvrende position
}

void bitwrite::putbits(unsigned code, char bitcount) {
  if(bufbyteofs+((bitcount+bufbitofs)>>3)+1>=FILEBUFLEN) {
    write(buffer,bufbyteofs); // Skriv det der er fyldt op
    buffer[0]=buffer[bufbyteofs];
    //^ St det der er i den sidste byte i start af nye buffer
    bufbyteofs=0;
    memset(&buffer[1],0,FILEBUFLEN-1);
  }

  for(signed char a=0; a<bitcount; a++) {
    if(getbytebit(code, a)) // Hvis den nuvrende bit er sat
      buffer[bufbyteofs]|=1<<bufbitofs;
    // ellers er biten i bufferen i forvejen resat via memset
    if(++bufbitofs==8) { bufbyteofs++; bufbitofs=0; }
  }
}
//-----------------------------BYTEREAD---------------------------------------
// Dette er en simpel buffer-klasse til lsning af bytes

void byteread::initread() {
  bufbyteofs=bytesback=0;
}

int byteread::getbyte() {
  if(bytesback<=0) {
    bytesback=read(buffer,FILEBUFLEN);
    if(bytesback<=0) return -1;
    bufbyteofs=0;
  }
  bytesback--;
  return (unsigned char)buffer[bufbyteofs++];
}

//------------------------------PACK_HUFFMAN----------------------------------
/* Denne klasse gemmer en konstant i den pakkede fil i intervallet
   [0..N_CHAR[ Den giver samtidig via putbits mulighed for at skrive
   bits direkte til den pakkede fil.
*/

void pack_huffman::writecode(unsigned c) {
  unsigned i;
  int j, k;

  i = 0;
  j = 0;
  k = prnt[c + T];

  /* travel from leaf to root */
  do {
    i <<= 1;

    /* if node's address is odd-numbered, choose bigger brother node */
    if (k & 1) i += 1;

    j++;
  } while ((k = prnt[k]) != R);
  update(c);
  putbits(i, j);

  // En kode vil benbart aldrig n op over 16 bits....
}

//------------------------------BINTREE---------------------------------------

/* Den flgende klasse bruges til et lidt specielt interface til et
   binrt tr.

   Der er erklret en ringbuffer med BUFSIZE elementer. Det binre tr
   srger for at man MEGET hurtigt kan sge efter en bestemt streng
   der starter et vilkrligt sted i denne buffer. Tret bestr
   sledes og af BUFSIZE noder - en til hvert af tegene i arrayet.

   Der er to funktioner tilgngelige for brugeren:

   void insertnode(int &matchofs, int& matchlen, int insertofs);
   |
   Denne indstter en node p insertofs. Samtidig vil den returnere
   offset i bufferen p det sted hvor starten af den indsatte streng
   passer s langt som muligt. offsetet p dette gemmes i matchofs
   og lngden af dette gemmes i matchlen

   void deletenode(unsigned ofs);
   Denne sletter den node der starter p ofs. insertnode vil ikke
   lngere bruge den...

   Der sker ikke noget hvis man kalder deletenode p et offset som
   insertnode ikke har indsat.
*/

/* Sdan slettes et element:
   Hvis vi haver flgende tr

    16
  /    \
      8        17
    /   \
  4       13
  /    \
     11        15
   /    \
 9        12
   \
     10


og sletter node 8 fr vi:

    16
  /    \
      9        17
    /   \
  4       13
  /    \
     11        15
   /    \
10        12

Vi gr med andre ord flgende
1) Tager noden til hjre for den der skal slettes (node 13)
   Hvis denne node ikke eksisterer s g til punkt 3
2) Sger fra denne (node 13) ned af og til venstre ind til
   der i et element ikke er noget til venstre (vi nr til node 9)
   Med dette finder vi den node der ligger ttest p den der skal slettes
   (node 13).
3) Den node (node 10) der mtte vre til hjre for den fundne (node 9)
   sttes direkte op p venstre siden af noden fr den fundne (node 11).
4) Den fundne node (node 9) sttes op p den gamles plads.

*/

/*

Bemrk:

Nr kode som denne find-om-jeg-er-left-eller-right laves

  if (left[father[deleteofs]] == deleteofs)
    left[father[deleteofs]] = q;
  else
    right[father[deleteofs]] = q;

S skal det IKKE skrives som flger:

  if (right[father[deleteofs]] == deleteofs)
    right[father[deleteofs]] = q;
  else
    left[father[deleteofs]] = q;

Dette vil nemlig kunne addresere ud over right's grnser hvis
father[deleteofs]>BUFSIZE - og det er den hvis noden stamme
direkte fra det verste af left (=er roden i et selvstndigt tr)

*/

void bintree::initbintree() {
  for(int a=0; a<BUFSIZE; a++) father[a]=BIN_NULL;
  for(a=BUFSIZE; a<BUFSIZE+1+256; a++) left[a]=BIN_NULL;
  // Ingen noder er brugt
}

//BX=oldplace
//BP=newplace
// Sletter noden i oldplace og stter noden i newplace i stedet
#define movenode(oldplace, newplace)   \
  father[newplace]=father[oldplace];   \
  right[newplace]=right[oldplace];     \
  left[newplace]=left[oldplace];       \
  father[right[oldplace]]=newplace;    \
  father[left[oldplace]]=newplace;     \
  if(left[father[oldplace]]==oldplace) \
    left[father[oldplace]]=newplace;   \
  else                                 \
    right[father[oldplace]]=newplace;  \
  father[oldplace]=BIN_NULL;

// Flgende funktion er taget (nsten) direkete fra LZHUF. Princippet
// der benyttes er illusteret i ovenstende tegning

void bintree::deletenode(unsigned deleteofs) {
  int  q;

  if (father[deleteofs] == BIN_NULL) return;  /* not in tree */

  if (right[deleteofs] == BIN_NULL) q = left[deleteofs];
  else if (left[deleteofs] == BIN_NULL) q = right[deleteofs];
  else {
    q = left[deleteofs];
    if (right[q] != BIN_NULL) {
      do {  q = right[q];  } while (right[q] != BIN_NULL);
      right[father[q]] = left[q];  father[left[q]] = father[q];
      left[q] = left[deleteofs];  father[left[deleteofs]] = q;
    }
    right[q] = right[deleteofs];  father[right[deleteofs]] = q;
  }
  father[q] = father[deleteofs];

  if (left[father[deleteofs]] == deleteofs)  // se kommentar i ovenstende
    left[father[deleteofs]] = q;
  else
    right[father[deleteofs]] = q;

  father[deleteofs] = BIN_NULL;
}

void bintree::insertnode(int &matchofs, int& matchlen, int insertofs) {
  // Indst ny node:

  int compare=-1; // G til venstre da ...
  unsigned curplace=BUFSIZE+1+ascbuf[insertofs];
  // ... der skal kigges i left eftersom det er i denne rderne
  // for de forskellige karakterer er gemt

  matchlen=-1; // Der er endnu ikke noget der passer...
  for(;;) {
    if(compare>0) {
      if(right[curplace]!=BIN_NULL)
        curplace=right[curplace]; // Der er mere p hjre side
      else {
        left[insertofs]=right[insertofs]=BIN_NULL;
        father[insertofs]=curplace;
        right[curplace]=insertofs;
        return; // matchlen etc er sat som de skal
      }
    } else {
      if(left[curplace]!=BIN_NULL)
        curplace=left[curplace];
      else {
        left[insertofs]=right[insertofs]=BIN_NULL;
        father[insertofs]=curplace;
        left[curplace]=insertofs;
        return;
      }
    }

    // Test hver der er strst af nuvrende og sgte streng
    for(int a=0; (a<MAX_PACK_LEN); a++)
      if((compare=ascbuf[curplace+a]-ascbuf[insertofs+a])!=0) break;

    if((a>matchlen) || ((a==matchlen) && (dist(insertofs,curplace)<dist(insertofs,matchofs)))) {
      matchofs=curplace; matchlen=a;
      if(matchlen>=MAX_PACK_LEN) {
        // Der er fundet en node med maksimal matchlen
        matchlen=MAX_PACK_LEN;
        // Flgende sletter den gamle node og indster denne.
        // Vi kan ikke have to ens noder i tret og den gamle
        // node vil blive slettet fr denne og med denne ville vi
        // f drligere komprision via huffman tabelen (se andet sted)
        movenode(curplace,insertofs);
        return;
      }
    }
  }
}

//--------------------------------ENCODE--------------------------------------
/* Denne klasse tager sig af selve kompresionen.
   Man har en ringbuffer som man hele tiden har lst MAX_PACK_LEN
   foran i, iforhold til hvad man vil komprimere (det frste tegn incl).
   Man tager s det tegn man vil komprimere og ser om den streng som den
   starter ved (lad os sige det er "Supjack International" findes et andet
   allerede behandlet sted i bufferen (dvs fra karakteren fr der pakkes
   og tilbage i bufferen til vi rammer det der er lst foran) *). Ordet
   "Supjack " Findes mske og nu gemmes offsetet samt lngde p den fundne
   streng. Hvis vi ikke havde fundet et match streng >=MIN_PACK_LEN, havde
   vi bare gemt karakteren "S" i den pakkede fil.

   *) Man kan faktisk finde en streng indtil karakteren umiddelbart fr.
      Dette er tilladt selvom hele den streng der skal pakkes ud da ikke er
      tilgngelige ved udpak. De enkelte karakterer nr jo lige akkurat
      at komme ind i ringbufferen fr de lses. Hvis man sledes angiver
      offset p karakteren umiddebart fr, s vil denne blive gentaget.
      Karakteren fr denne vil give  en word gentagelse med len/2 lngden osv.

   Ovenstende er ikke helt rigtigt: "S" bliver i sidste tilflde gemt
   via dynamisk huffmann, og "Supjack " bliver gemt ved at gemme karakter
   nr 256-MIN_PACK_LEN+strlen("Supjack "); Herefter bliver offsetet
   i bufferen gemt.

   Og det er heller ikke helt rigtigt ;-) Som offset bliver gemt en tal vrdi
   udregnet p flgende mde:
   (offsetet p "Supjack International")-MIN_PACK_LEN+(antal af positioner
   vi skal g tilbage fr vi nr den fundne streng). Dette vil sikre
   et s lavt offset som muligt, da der er strre sandsynlighed for at
   vi finder en match tt p en langt fra. Dette gres via makroen dist

   #ifdef USETABL
     Dette muligr en endnu bedre komprision: De mestbetydende 6 bit
     af det relative offset p 12 bits komprimeres via en huffman tabel.
     Man lgger simpelthen 6 bit tallet ind som offset i tabelerne
     huff_code og huff_length. S vil man f huffman koden samt antal bits
     p det sgte. De mindst betydende 6 bit stores bare.
     Denne tabel er taget direkte fra LZHUF.
  #endif
*/

#ifdef USETABL
  extern unsigned char huff_pack_code[];
  extern unsigned char huff_pack_len[];
#endif

void encode::writepacked(unsigned bufofs, unsigned buflen) {
  writecode(256+buflen-MIN_PACK_LEN);
  //^Gem en "karakter" der angiver lngden p strengen

  #ifdef USETABL
    // Pak den koddede frst da udpakkeren kan bruge eventuelt for
    // mange lst bits til den ukoddede
    putbits(huff_pack_code[bufofs>>6],huff_pack_len[bufofs>>6]);
    putbits(bufofs, 6);
  #else
    putbits(bufofs, 12);
  #endif
}

void encode::do_encode() {
  initread();  pack_huffman::inithuff(); initbintree(); initwrite();

  int curpos=0, // Hvor er vi i bufferen
      bestofs,  // Det offset hvor der ligger en streng der passer bedst
                // til den nuvrende
      bestlen,  // Lngden af denne streng
      curlen=0; // Det antal bytes vi har lst foran i bufferen. Denne
                // bliver formindsket nr vi nr slutningen af det der
                // skal pakkes

  // Ls foran i bufferen:
  for(int a=0; a<MAX_PACK_LEN; a++) {
    int b=getbyte(); if(b==-1) break;
    ascbuf[a]=b; curlen++;
  }

  // Der er nu en hel streng i bufferen - fortl dette til objektet
  // der styrer sgningen i bufferen:
  insertnode(bestofs,bestlen,0);

  while(curlen!=0) {
    // Hvis vi har fundet en match ud over hvad der er tilbage at finde
    // matches i:
    if(bestlen>curlen) bestlen=curlen;

    // Skriv en kode:
    if(bestlen<MIN_PACK_LEN) {
      writecode(ascbuf[curpos]); bestlen=1;
    } else {
      writepacked(dist(curpos,bestofs),bestlen);
    }
    // bestlen indeholder det antal bytes der er lst fra bufferen
    // i denne omgang. Ls det antal bytes:

    int q=bestlen;
    for(int a=0; a<q; a++) {
      curpos=normal(curpos+1);
      // G en position frem

      int c=(curlen==MAX_PACK_LEN) ? getbyte() : -1;
      // Ved EOF returnerer getbyte -1, og derefter kaldes den ikke mere:
      if(c==-1) curlen--;

      // Indst den lste char i bufferen. Hvis c er == -1 gr det ikke noget
      // da -1 strengen p MAX_PACK_LEN bytes lngde lige akkurat ikke nr at
      // blive indsat. Det gr derimod nogle strenge der kommer til at slut-
      // te med -1. Det gr imidlertid ikke noget, se kommentar til insertnode
      // og den efterflgende linie i nedenstende. Tilgengld er det nd-
      // vendigt at slette de gamle noder:

      ascbuf[curpos+MAX_PACK_LEN-1]=c;

      if(curpos+MAX_PACK_LEN-1>=BUFSIZE) {
        // Ved de sidste MAX_PACK_LEN-1 bytes skal vi slette i starten af
        // bufferen:
        ascbuf[curpos+MAX_PACK_LEN-1-BUFSIZE]=c;
        deletenode(curpos+MAX_PACK_LEN-1-BUFSIZE);
      } else deletenode(curpos+MAX_PACK_LEN-1);

      insertnode(bestofs,bestlen,curpos);
    }
  }
  writepacked(ENDOFS,MIN_PACK_LEN);
  flushbuf();
}
