// File from page 190 in "Thinking in C++" by Bruce Eckel
//////////////////////////////////////////////////
// From the compressed package ECKELT01.ZIP 2/21/95
// Copyright (c) Bruce Eckel, 1995 
// Source code file from the book "Thinking in C++", 
// Prentice Hall, 1995, ISBN: 0-13-917709-4
// All rights reserved EXCEPT as allowed by the following 
// statements: You may freely use this file for your own 
// work, including modifications and distribution in 
// executable form only. You may copy and distribute this 
// file, as long as it is only distributed in the complete 
// (compressed) package with the other files from this 
// book and you do not remove this copyright and notice. 
// You may not distribute modified versions of the source 
// code in this package. This package may be freely placed 
// on bulletin boards, internet nodes, shareware disks and 
// product vendor disks. You may not use this file in 
// printed media without the express permission of the 
// author. Bruce Eckel makes no 
// representation about the suitability of this software 
// for any purpose. It is provided "as is" without express 
// or implied warranty of any kind. The entire risk as to 
// the quality and performance of the software is with 
// you. Should the software prove defective, you assume 
// the cost of all necessary servicing, repair, or 
// correction. 
// If you think you've found an error, please 
// email all modified files with loudly commented changes 
// to: eckel@aol.com (please use the same 
// address for non-code errors found in the book).
//////////////////////////////////////////////////

//: BITVECT.CPP -- BitVector Implementation
#include <stdio.h>
#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include "..\4\bitvect.h"
#include <limits.h> //CHAR_BIT = # bits in char
// A byte with the high bit set:
const unsigned char highbit =
  1 << (CHAR_BIT - 1);

BitVector::BitVector() {
  numBytes = 0;
  Bits = 0;
  bytes = 0;
}
// Notice default args are not duplicated:
BitVector::BitVector(unsigned char* init,
                     int size) {
  numBytes = size;
  Bits = numBytes * CHAR_BIT;
  bytes = (unsigned char*)calloc(numBytes, 1);
  assert(bytes);
  if(init == 0) return; // Default to all 0
  // Translate from bytes into bit sequence:
  for(int index = 0; index<numBytes; index++)
    for(int offset = 0;
        offset < CHAR_BIT; offset++)
      if(init[index] & (highbit >> offset))
         set(index * CHAR_BIT + offset);
}

BitVector::BitVector(char* binary) {
  Bits = strlen(binary);
  numBytes =  Bits / CHAR_BIT;
  // If there's a remainder, add 1 byte:
  if(Bits % CHAR_BIT) numBytes++;
  bytes = (unsigned char*)calloc(numBytes, 1);
  assert(bytes);
  for(int i = 0; i < Bits; i++)
    if(binary[i] == '1') set(i);
}

BitVector::~BitVector() {
  free(bytes);
}

void BitVector::set(int bit) {
  assert(bit >= 0 && bit < Bits);
  int index = bit / CHAR_BIT;
  int offset = bit % CHAR_BIT;
  unsigned char mask = (1 << offset);
  bytes[index] |= mask;
}

int BitVector::read(int bit) {
  assert(bit >= 0 && bit < Bits);
  int index = bit / CHAR_BIT;
  int offset = bit % CHAR_BIT;
  unsigned char mask = (1 << offset);
  return bytes[index] & mask;
}

void BitVector::clear(int bit) {
  assert(bit >= 0 && bit < Bits);
  int index = bit / CHAR_BIT;
  int offset = bit % CHAR_BIT;
  unsigned char mask = ~(1 << offset);
  bytes[index] &= mask;
}

int BitVector::bits() { return Bits; }

void BitVector::bits(int size) {
  int oldsize = Bits;
  Bits = size;
  numBytes =  Bits / CHAR_BIT;
  // If there's a remainder, add 1 byte:
  if(Bits % CHAR_BIT) numBytes++;
  void* v = realloc(bytes, numBytes);
  assert(v);
  bytes = (unsigned char*)v;
  for(int i = oldsize; i < Bits; i++)
    clear(i); // Erase additional bits
}

void BitVector::print(const char* msg) {
  puts(msg);
  for(int i = 0; i < Bits; i++){
    if(read(i)) putchar('1');
      else putchar('0');
    // Format into byte blocks:
    if((i + 1) % CHAR_BIT == 0) putchar(' ');
  }
  putchar('\n');
}
