/******************************************************************************\

  Copyright 1995 The University of North Carolina at Chapel Hill.
  All Rights Reserved.

  Permission to use, copy, modify and distribute this software and its
  documentation for educational, research and non-profit purposes, without
  fee, and without a written agreement is hereby granted, provided that the
  above copyright notice and the following three paragraphs appear in all
  copies.

  IN NO EVENT SHALL THE UNIVERSITY OF NORTH CAROLINA 
  AT CHAPEL HILL BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, 
  OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS 
  SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY
  OF NORTH CAROLINA HAVE BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.


  Permission to use, copy, modify and distribute this software and its
  documentation for educational, research and non-profit purposes, without
  fee, and without a written agreement is hereby granted, provided that the
  above copyright notice and the following three paragraphs appear in all
  copies.

  THE UNIVERSITY OF NORTH CAROLINA SPECIFICALLY 
  DISCLAIM ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED 
  HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF NORTH CAROLINA HAS NO OBLIGATIONS
  TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

  The authors may be contacted via:

  US Mail:             J. Cohen/M. Lin/D. Manocha/K. Ponamgi
                       Department of Computer Science
                      Sitterson Hall, CB #3175
                       University of N. Carolina
                       Chapel Hill, NC 27599-3175

  Phone:               (919)962-1749

  EMail:              {cohenj,lin,manocha,ponamgi}@cs.unc.edu


\*****************************************************************************/


/*****************************************************************************\
  bounds.c
  --
  Description :  Routines to update a polytope's bounding box to reflect its
                 current transformation

\*****************************************************************************/


/*----------------------------- Local Includes -----------------------------*/

#include <collision.h>
#include <collision_types.h>
#include <matrix.h>
#include <stdio.h>

/*----------------------------- Local Constants -----------------------------*/


/*------------------------------ Local Macros -------------------------------*/


/*
   XFORM_SINGLE_COORD: transform either the X, Y, or Z coordinate of a point 
   note : It's OK to leave out the translation as long as you put it in
          for the final box_vert->value.  Leaving out this addition didn't
	  seem to speed it up for the case where we never walk anywhere, but
	  maybe it will for rotating objects with lots of vertices.

        : Using a macro instead of a function call gave a HUGE speedup,
	  however, because this transformation does such a small amount
	  of work and is called so often
*/

#define XFORM_SINGLE_COORD(transform, coords, result, axis) \
    (result) =  (transform)[(axis)][0]*(coords)[0] \
              + (transform)[(axis)][1]*(coords)[1] \
              + (transform)[(axis)][2]*(coords)[2] 

/*------------------------------- Local Types -------------------------------*/


/*------------------------ Local Function Prototypes ------------------------*/

static void UpdateMin(col_Mat4 transform, __col_BoxVert *box_vert, int axis);
static void UpdateMax(col_Mat4 transform, __col_BoxVert *box_vert, int axis);

/*------------------------------ Local Globals ------------------------------*/


/*---------------------------------Functions-------------------------------- */


/*****************************************************************************\
 @ UpdateMin()
 -----------------------------------------------------------------------------
 description : Walk from a polytope's previous min vertex for X, Y, or Z to its
               new min vertex for X, Y, or Z.
 input       : The polytope's transformation, the vertex of the previous
               minimum for this coordinate axis, and the coordinate axis (X,
	       Y, or Z) 
 output      : The vertex is updated to the current minimum for this axis
 notes       : UpdateMin and UpdateMax should be maintained identically except
	       for the replacement of '<' with '>'.
\*****************************************************************************/
static void UpdateMin(col_Mat4 transform, __col_BoxVert *box_vert, int axis)
{
    int edge_count;
    int done;
    col_Fnode *fnode;
    __col_Edge *edge;
    __col_Vertex *vert2;
    double vert2_val;

    /*
       Walk along adjacent vertices to minimize the value for the appropriate
       coordinate axis
    */
    done = COL_FALSE;
    while (done != COL_TRUE)
    {
	/*
	   Transform the vertex (except for translation, which isn't necessary
	   for comparing vertices of a single polytope)
	*/
	XFORM_SINGLE_COORD(transform, box_vert->vert->coords, box_vert->value,
			   axis); 

	/*
	   Compare the axis value of this vertex to all the adjacent vertices
	*/
	fnode = box_vert->vert->edges;
	done = COL_TRUE;
	while (fnode != COL_NULL)
	{
	    edge = fnode->f.e;
	    vert2 = edge->v1;
	    if (vert2 == box_vert->vert)
		vert2 = edge->v2;
	    XFORM_SINGLE_COORD(transform, vert2->coords, vert2_val, axis);

	    /*
	       If an adjacent vertex has a smaller axis value, use it to
	       replace the old minimum
	    */
	    if (vert2_val < box_vert->value)
	    {
		done = COL_FALSE;
		box_vert->value = vert2_val;
		box_vert->vert = vert2;
	    }
	    fnode = fnode->next;
	}
    }

    /*
       put the translation in so the sorting will work
    */
    box_vert->value += transform[axis][3];
    return;
} /** End of UpdateMin() **/

/*****************************************************************************\
 @ UpdateMax()
 -----------------------------------------------------------------------------
 description : Same as UpdateMin, except we find the maximum instead of the
               minimum 
 input       : 
 output      : 
 notes       : UpdateMin and UpdateMax should be maintained identically except
	       for the replacement of '<' with '>'.
\*****************************************************************************/
static void UpdateMax(col_Mat4 transform, __col_BoxVert *box_vert, int axis)
{
    int edge_count;
    int done;
    col_Fnode *fnode;
    __col_Edge *edge;
    __col_Vertex *vert2;
    double vert2_val;
    
    /*
       Walk along adjacent vertices to maximize the value for the appropriate
       coordinate axis
    */
    done = COL_FALSE;
    while (done != COL_TRUE)
    {
	/*
	   Transform the vertex (except for translation, which isn't necessary
	   for comparing vertices of a single polytope)
	*/
	XFORM_SINGLE_COORD(transform, box_vert->vert->coords, box_vert->value, 
			   axis);

	/*
	   Compare the axis value of this vertex to all the adjacent vertices
	*/
	fnode = box_vert->vert->edges;
	done = COL_TRUE;
	while (fnode != COL_NULL)
	{
	    edge = fnode->f.e;
	    vert2 = edge->v1;
	    if (vert2 == box_vert->vert)
		vert2 = edge->v2;
	    XFORM_SINGLE_COORD(transform, vert2->coords, vert2_val, axis);

	    /*
	       If an adjacent vertex has a larger axis value, use it to
	       replace the old maximum
	    */
	    if (vert2_val > box_vert->value)
	    {
		done = COL_FALSE;
		box_vert->value = vert2_val;
		box_vert->vert = vert2;
	    }
	    fnode = fnode->next;
	}
    }

    /*
       put the translation in so the sorting will work
    */
    box_vert->value += transform[axis][3];
    return;
} /** End of UpdateMax() **/


/*****************************************************************************\
 @ __col_update_bounding_box()
 -----------------------------------------------------------------------------
 description : Update the bounding box for a polytope when it has moved
 input       : Polytope to update
 output      : The polytope's bounding box is updated to reflect its current
               transformation
 notes       :
\*****************************************************************************/
void __col_update_bounding_box(__col_Polytope *col_poly)
{
    int axis;
    col_Vect3 center;
    float extra_radius;
    
    /*
       For bounding cubes, just transform the cube center and add/subtract the
       box radius -- note that some extra "buffer" space is already built into
       the box radius
    */
    if (col_poly->box.use_cuboid == COL_TRUE)
    {
	__col_xformPoint(col_poly->polytope->pose, col_poly->box.center,
			 center);
	for (axis=0; axis < 3; axis++)
	{
	    col_poly->box.min[axis].value = center[axis] - col_poly->box.radius;
	    col_poly->box.max[axis].value = center[axis] + col_poly->box.radius;
	}
    }
    else
    {
	/*
	   for rectangular bounding boxes, update the min and max for each
	   axis using local hill-climbing algorithm
	*/
	for (axis=0; axis<3; axis++)
	{
	    UpdateMin(col_poly->polytope->pose, &col_poly->box.min[axis],
		      axis);
	    UpdateMax(col_poly->polytope->pose, &col_poly->box.max[axis],
		      axis);

	    /*
	       increase the length of the box along this axis by some
	       percentage to add some extra "buffer" space
	    */
	    extra_radius =
		(col_poly->box.max[axis].value -
		 col_poly->box.min[axis].value) *
		     COL_DYNAMIC_BOX_BUFFER_PERCENT / 200.0;	    
	    col_poly->box.min[axis].value -= extra_radius;
	    col_poly->box.max[axis].value += extra_radius;
	}
    }
    return;
} /** End of __col_update_bounding_box() **/


/*****************************************************************************\
\*****************************************************************************/

