/******************************************************************************\

  Copyright 1995 The University of North Carolina at Chapel Hill.
  All Rights Reserved.

  Permission to use, copy, modify and distribute this software and its
  documentation for educational, research and non-profit purposes, without
  fee, and without a written agreement is hereby granted, provided that the
  above copyright notice and the following three paragraphs appear in all
  copies.

  IN NO EVENT SHALL THE UNIVERSITY OF NORTH CAROLINA
  AT CHAPEL HILL BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL,
  OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS
  SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY
  OF NORTH CAROLINA HAVE BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.


  Permission to use, copy, modify and distribute this software and its
  documentation for educational, research and non-profit purposes, without
  fee, and without a written agreement is hereby granted, provided that the
  above copyright notice and the following three paragraphs appear in all
  copies.

  THE UNIVERSITY OF NORTH CAROLINA SPECIFICALLY
  DISCLAIM ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED
  HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF NORTH CAROLINA HAS NO OBLIGATION
S
  TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

  The authors may be contacted via:

  US Mail:             J. Cohen/M. Lin/D. Manocha/K. Ponamgi
                       Department of Computer Science
                      Sitterson Hall, CB #3175
                       University of N. Carolina
                       Chapel Hill, NC 27599-3175

  Phone:               (919)962-1749

  EMail:              {cohenj,lin,manocha,ponamgi}@cs.unc.edu




\*****************************************************************************/


/*****************************************************************************\
  collision.h
  --
  Description :  Main header file for the collision detection library

\*****************************************************************************/

/* Protection from multiple includes. */
#ifndef INCLUDED_COLLISION_H
#define INCLUDED_COLLISION_H


/*------------------ Includes Needed for Definitions Below ------------------*/

#include <collision_types.h>

/*-------------------------------- Constants --------------------------------*/

#define COL_NOT_COLLIDING 2
#define COL_COLLIDING 1

#define COL_OK     0

#define COL_ERROR -1
#define COL_UNKNOWN -2


#define COL_TRUE  1
#define COL_FALSE 0

/*--------------------------------- Macros ----------------------------------*/


/*---------------------------------- Types ----------------------------------*/


/*---------------------------- Function Prototypes --------------------------*/


/*
   ==========================================================================
   ==========================================================================

		       INITIALIZE THE COLLISION LIBRARY

   ==========================================================================
   ==========================================================================
*/

/*****************************************************************************\
 @ col_open()
 -----------------------------------------------------------------------------
 description : Call to activate collision detection library
 input       : Maximum number of polytopes to be instanced, name of polytope
               data file, scale factor (1.0 to use the polytopes' original
	       scale), and an (int *) to get the number of polytopes loaded 
 output      : COL_OK or COL_ERROR is returned, and num_polytopes indicates
               the number of polytopes loaded from the polytope file 
 notes       : passing in -1 for max_polytopes uses the number of objects in
               the library as max_polytopes, indicating that the user plans to
	       instance each polytope no more than once.
\*****************************************************************************/
int col_open(int max_polytopes, char *filename,
	     double scale, int *num_polytopes);


/*
   ==========================================================================
   ==========================================================================

			 ROUTINES TO INSTANCE OBJECTS

   ==========================================================================
   ==========================================================================
*/

/*****************************************************************************\
 @ col_instance_polytope()
 -----------------------------------------------------------------------------
 description : Instance a polyhedron from the library.
 input       : name of library object to instance
 output      : id number to use when referencing the polytope
 notes       : 
\*****************************************************************************/
int col_instance_polytope(char *lib_object, int *id);

/*****************************************************************************\
 @ col_instance_polytope_by_num()
 -----------------------------------------------------------------------------
 description : Instance a polytope by index within the library rather than by
               the object's name 
 input       : 
 output      : 
 notes       :
\*****************************************************************************/
int col_instance_polytope_by_num(int poly_num, int *id);


/*
   ==========================================================================
   ==========================================================================

			 ROUTINES THAT SET SOME MODES

   ==========================================================================
   ==========================================================================
*/

/*****************************************************************************\
 @ col_enable_nbody()
 -----------------------------------------------------------------------------
 description : Enable the nbody algorithm - only track the distance of
               polytope pairs whose bounding boxes overlap.
 input       : 
 output      : 
 notes       : The nbody algorithm is enabled by default
\*****************************************************************************/
int col_enable_nbody();

/*****************************************************************************\
 @ col_disable_nbody()
 -----------------------------------------------------------------------------
 description : Disable the nbody algorithm - track the distance of all active
               pairs.
 input       : 
 output      : 
 notes       : The nbody algorithm is enabled by default
\*****************************************************************************/
int col_disable_nbody();

/*****************************************************************************\
 @ col_use_cuboid()
 -----------------------------------------------------------------------------
 description : Use a fixed bounding cube for a polytope rather than a
               dynamically resized bounding box
 input       : Polytope ID
 output      : 
 notes       : The default setting is dynamically resized bounding boxes
\*****************************************************************************/
int col_use_cuboid(int id);

/*****************************************************************************\
 @ col_use_dynamic_bounding_box()
 -----------------------------------------------------------------------------
 description : Use dynamic bounding boxes that change shape to fit the object
               as it moves around.
 input       : Polytope id number.
 output      : 
 notes       : These is the default setting for each instanced polytope
\*****************************************************************************/
int col_use_dynamic_bounding_box(int id);


/*
   ==========================================================================
   ==========================================================================

	       CHOOSE WHICH PAIRS OF OBJECTS SHOULD BE TRACKED

   ==========================================================================
   ==========================================================================
*/

/*****************************************************************************\
 @ col_activate_pair()
 -----------------------------------------------------------------------------
 description : Activate a pair of polytopes for collision detection
 input       : IDs of the polytopes to be activated
 output      : 
 notes       :
\*****************************************************************************/
int col_activate_pair(int id1, int id2);

/*****************************************************************************\
 @ col_activate_full()
 -----------------------------------------------------------------------------
 description : Activate collision detection for one polytope with all the
               other polytopes
 input       : ID of polytope to activate with all others
 output      : 
 notes       :
\*****************************************************************************/
int col_activate_full(int id);

/*****************************************************************************\
 @ col_activate_all()
 -----------------------------------------------------------------------------
 description : Activate all pairs of polytopes for collision detection
 input       : 
 output      : 
 notes       :
\*****************************************************************************/
int col_activate_all();

/*****************************************************************************\
 @ col_deactivate_pair()
 -----------------------------------------------------------------------------
 description : Deactivate collision detection for a pair of polytopes.
 input       : 
 output      : 
 notes       :
\*****************************************************************************/
int col_deactivate_pair(int id1, int id2);

/*****************************************************************************\
 @ col_deactivate_full()
 -----------------------------------------------------------------------------
 description : Deactivate collision detection for one polytope with all the
               other polytopes
 input       : ID of polytope to deactivate with all others
 output      : 
 notes       :
\*****************************************************************************/
int col_deactivate_full(int id);

/*****************************************************************************\
 @ col_deactivate_all()
 -----------------------------------------------------------------------------
 description : Dectivate all pairs of polytopes for collision detection
 input       : 
 output      : 
 notes       :
\*****************************************************************************/
int col_deactivate_all();


/*
   ==========================================================================
   ==========================================================================

		   CHANGE AN OBJECT'S POSITION/ORIENTATION

   ==========================================================================
   ==========================================================================
*/

/*****************************************************************************\
 @ col_set_transform
 -----------------------------------------------------------------------------
 description : Set the current transformation for a polytope
 input       : Polytope id, 4x4 transformation matrix
 output      : 
 notes       : The matrix should be a 4x4 used to premultiply a
               column vector. 
\*****************************************************************************/
int col_set_transform(int id, col_Mat4 matrix);


/*
   ==========================================================================
   ==========================================================================

		       THE MAIN COLLISION TEST ROUTINE

   ==========================================================================
   ==========================================================================
*/

/*****************************************************************************\
 @ col_test()
 -----------------------------------------------------------------------------
 description : Do the main collision test for the current set of object
               positions and active pairs
 input       : 
 output      : 
 notes       :
\*****************************************************************************/
int col_test();


/*
   ==========================================================================
   ==========================================================================

			ROUTINES TO REPORT COLLISIONS

   ==========================================================================
   ==========================================================================
*/

/*****************************************************************************\
 @ col_report_collision_pair()
 -----------------------------------------------------------------------------
 description : Checks to see if a pair was colliding in the last collision test.
 input       : Two ids and the address of a col_Report to be filled in
 output      : return valud will be COL_COLLIDING if colliding,
               COL_NOT_COLLIDING if not colliding, or COL_UNKNOWN if the pair
	       isn't active.  If the return value is COL_COLLIDING, the report
	       will be filled in for that pair
 notes       :
\*****************************************************************************/
int col_report_collision_pair(int id1, int id2, col_Report *report);

/*****************************************************************************\
 @ col_report_collision_full()
 -----------------------------------------------------------------------------
 description : Report all collisions from the previous test which involve one
               particular polytope 
 input       : ID of the polytope to be tested, an array of col_Reports, and
               the number of elements in the array
 output      : The return value indicates the number of collisions found and
               the first report_size collisions are report in the report
	       array.
 notes       : 
\*****************************************************************************/
int col_report_collision_full(int id, col_Report *report, int report_size);

/*****************************************************************************\
 @ col_report_collision_all()
 -----------------------------------------------------------------------------
 description : Report all collisions from the most recent test
 input       : Array of col_Reports and the number of elements in the array
 output      : The return value indicates the number of collisions, and the
               first report_size collisions are reported in the report array 
 notes       : 
\*****************************************************************************/
int col_report_collision_all(col_Report *report, int report_size);


/*
   ==========================================================================
   ==========================================================================

		     ROUTINES TO REPORT CLOSEST DISTANCES

   ==========================================================================
   ==========================================================================
*/

/*****************************************************************************\
 @ col_report_distance_pair()
 -----------------------------------------------------------------------------
 description : Return the distance between a pair of polytopes and their
               closest features, if known
 input       : The ids of the polytopes in question and a pointer to a
               col_DistReport to be filled in.
 output      : The return value is COL_UNKNOWN if the pair is inactive or
               non-overlapping.  Otherwise, the return value is COL_OK and the
	       report structure is filled in.
 notes       :
\*****************************************************************************/
int col_report_distance_pair(int id1, int id2, col_DistReport *report);

/*****************************************************************************\
 @ col_report_distance_full()
 -----------------------------------------------------------------------------
 description : Report known distances from one particular polytope with all
               other polytopes 
 input       : Id of polytope in question, array of col_DistReports, and
               number of elements in the array
 output      : The return value is the number of distances known, and the
               first report_size of these are reported in the report array. 
 notes       :
\*****************************************************************************/
int col_report_distance_full(int id, col_DistReport *report, int report_size);

/*****************************************************************************\
 @ col_report_distance_all()
 -----------------------------------------------------------------------------
 description : Report all known distances and closest features between pairs
               of polytopes.
 input       : Array of col_DistReports and number of elements in the array
 output      : The return value is the number of distances known, and the
               first report_size of these are returned in the report array. 
 notes       :
\*****************************************************************************/
int col_report_distance_all(col_DistReport *report, int report_size);


/*
   ==========================================================================
   ==========================================================================

			   A HANDY UTILITY ROUTINE

   ==========================================================================
   ==========================================================================
*/

/*****************************************************************************\
 @ col_calc_cuboid()
 -----------------------------------------------------------------------------
 description : Calculates the center of the instantiated polytopes.  This is
	       used for determining a collision reaction (for example with a
	       boundry box).
 input       : 
 output      : A point inside the polytope and the maximum distance from
               this point to any vertex.
 notes       : It currently computes a point inside the polytope -- the
               unweighted average of the vertices.  This isn't the
               center of mass.
\*****************************************************************************/
int col_calc_cuboid(int poly_id, double *box_center, double *box_radius);



/*----------------------------  Globals (externed) --------------------------*/



/* Protection from multiple includes. */
#endif /* INCLUDED_COLLISION_H */



/*****************************************************************************\
\*****************************************************************************/

