/******************************************************************************\

  Copyright 1995 by Jonathan Cohen, Ming Lin,
                    Dinesh Manocha, Krish Ponamgi, and
		    the University of North Carolina at Chapel Hill.

  Permission to use, copy, and distribute for non-commercial purposes,
  is hereby granted without fee, providing that the above copyright
  notice appear in all copies and that both the copyright notice and this
  permission notice appear in supporting documentation. 

  The software may be modified for your own purposes, but modified versions
  may not be distributed.

  This software is provided "as is" without any expressed or implied warranty.

  The authors may be contacted as follows:

  EMail: cohenj@cs.unc.edu
         lin@cs.unc.edu
	 manocha@cs.unc.edu
	 ponamgi@cs.unc.edu

\*****************************************************************************/


/*****************************************************************************\
  collision_types.h
  --
  Description : Types for the upper layer of the collision detection library.
                The types for the Lin/Canny layer are in dist.h. 

\*****************************************************************************/

/* Protection from multiple includes. */
#ifndef INCLUDED_COLLISION_TYPES_H
#define INCLUDED_COLLISION_TYPES_H


/*------------------ Includes Needed for Definitions Below ------------------*/

#include <options.h>
#include <dist.h>

/*-------------------------------- Constants --------------------------------*/

#define COL_NULL 0

#define __COL_X 0
#define __COL_Y 1
#define __COL_Z 2

/*--------------------------------- Macros ----------------------------------*/

#define __COL_OVERLAPPING(a) 							\
    ((a)->overlap[__COL_X] && (a)->overlap[__COL_Y] && (a)->overlap[__COL_Z])


/*  indexing routines -- given the offset from the start of the allocated
    storage, compute the row and column in the half array, or do the reverse */

#ifdef COL_DENSE_MATRIX

#define __COL_PAIR_INDEX(state,row,col) 				\
       ( ((col) < (state).matrix_cols)   				\
	? ( (row)*((state).matrix_cols)+(col) ) 			\
	: (((state).matrix_rows-(row)-1) * ((state).matrix_cols) + 	\
	     ((state).max_polytopes - (col) - 1)) )

#define __COL_PAIR_ROW_COL(state, pair, row, col) 		\
        {              						\
           int offset; 						\
                       						\
           offset = (pair-(state).pairs); 			\
	   (row) = offset / (state).matrix_cols; 		\
           (col) = offset % (state).matrix_cols; 		\
           if ((row) >= (col)) 					\
           {   							\
	       (row) = (state).matrix_rows - (row) - 1; 	\
	       (col) = (state).max_polytopes - (col) - 1; 	\
	   } 							\
        }

#define __COL_GET_PAIR(state,row,col,pair)						\
     {											\
         (pair) = ( ((row) < (col))							\
		  ? (&((state).pairs[__COL_PAIR_INDEX((state), (row), (col))]))		\
		  : (&((state).pairs[__COL_PAIR_INDEX((state), (col), (row))])) );	\
     }

#define __COL_ALLOCATE_PAIR(state,row, col, pair) 	\
          __COL_GET_PAIR((state), (row), (col), (pair))

#endif /* COL_DENSE_MATRIX */



#ifdef COL_SPARSE_MATRIX

#define __COL_PAIR_INDEX(state,row,col) 				\
       ( ((col) < (state).matrix_cols)   				\
	? ( (row)*((state).matrix_cols)+(col) ) 			\
	: (((state).matrix_rows-(row)-1) * ((state).matrix_cols) + 	\
	     ((state).max_polytopes - (col) - 1)) )

#define __COL_PAIR_ROW_COL(state, pair, row, col)		\
        {							\
           (row) = (pair)->ids[0];				\
	   (col) = (pair)->ids[1];				\
        }

#define __COL_GET_PAIR(state,row,col,pair)					\
     {										\
         (pair) = ( ((row) < (col))						\
		  ? ((state).pairs[__COL_PAIR_INDEX((state), (row), (col))])	\
		  : ((state).pairs[__COL_PAIR_INDEX((state), (col), (row))]) );	\
     }

#define __COL_ALLOCATE_PAIR(state,row,col,pair)						\
     {											\
	 if ((row) < (col))								\
	 {										\
	     (pair) = (state).pairs[__COL_PAIR_INDEX((state), (row), (col))] =		\
		 (__col_PairNode *)calloc(1, sizeof(__col_PairNode));			\
	     (pair)->ids[0] = (row);							\
	     (pair)->ids[1] = (col);							\
	 }										\
	 else										\
	 {										\
	     (pair) = (state).pairs[__COL_PAIR_INDEX((state), (col), (row))] =		\
		 (__col_PairNode *)calloc(1, sizeof(__col_PairNode));			\
	     (pair)->ids[0] = (col);							\
	     (pair)->ids[1] = (row);							\
	 }										\
	 if (!(pair))									\
	 {										\
	     fprintf(stderr,								\
		     "__COL_ALLOCATE_PAIR: couldn't allocate __col_PairNode\n");	\
	     return COL_ERROR;								\
	 }										\
     }

#endif /* COL_SPARSE_MATRIX */










/*---------------------------------- Types ----------------------------------*/


/*
   A __col_BoxVert is an element of a an extrema list.  Each polytope has 6
   __col_BoxVerts -- xmin, ymin, zmin, xmax, ymax, zmax.
*/
typedef struct __col_box_vert_struct
{
    int           id;        /* number of this vertex's polytope */
    char          min_max;   /* 0 if this is a MIN, 1 if it's a MAX */
    double        value;     /* transformed coordinate (X, Y, or Z) */
    __col_Vertex *vert;      /* Pointer to the actual __col_Vertex */
    
                       /* prev and next __col_BoxVert in this list
			  (either x list, y list, or z list) */
    struct __col_box_vert_struct *prev;
    struct __col_box_vert_struct *next;
} __col_BoxVert;


/*
   Each polytope has a bounding box.  For fixed-sized bounding cubes, a center
   a radius defines the cube.  For dynamic bounding boxes, 6 extremal vertices
   define the box
*/
typedef struct __col_bounding_box_struct
{
    __col_BoxVert min[3];
    __col_BoxVert max[3];
    char use_cuboid;
    double center[3];
    double radius;
} __col_BoundingBox;

/*
   Each instanced object is a __col_Polytope.  It has an id number, a
   polyhedral representation (from dist.h), and a bounding box.
*/
typedef struct col_polytope_struct
{
    int id;
    __col_Polyhedron *polytope;
    __col_BoundingBox box;

} __col_Polytope;


/*
   Every pair of objects that may interact is represented by a pair node (in
   the COL_SPARSE_MATRIX representation, only activated pairs have allocated
   __col_PairNodes).  The pair node stores the current closest features for
   that pair, and the current closest distance.  It also contains flags to
   indicate whether the pair is currently active, which dimensions of the
   pair's bounding boxes are currently overlapping, and whether the pair is
   currently colliding.  In addition, this pair may be inserted and deleted
   from several lists, so the node contains points used for traversing these
   lists.
*/
typedef struct __col_pn_struct
{
#ifdef COL_SPARSE_MATRIX
    int          ids[2];
#endif
    void        *closest_features[2];
    float        distance;
    char         active;                          /* boolean flag */
    char         overlap[3];                      /* boolean x,y,z overlap flags */
    char         colliding;                       /* boolean flag */
    struct __col_pn_struct    *next_active;       /* next user-activated pair */
    struct __col_pn_struct    *prev_nbody_active; /* next and previous pairs that have*/
    struct __col_pn_struct    *next_nbody_active; /*   all 3 coordinates overlapping  */
                                                  /*   (i.e. ones requiring Lin/Canny)*/
    struct __col_pn_struct    *next_colliding;    /* next/prev pair in list of */
    struct __col_pn_struct    *prev_colliding;    /*    colliding pairs        */
} __col_PairNode;


/*
   A col_Report is actually a single element of a collision report.  An array
   of these may be past to a report function, which fills in the array.  Each
   report element contains the ids of a pair of colliding polytopes also two
   features which were the closest in the most recent non-colliding test.  One
   might perform further analysis to determine the full set of colliding
   features if necessary.
*/
typedef struct col_report_struct
{
    int     ids[2];
    col_Fnode  *features[2];
} col_Report;

/*
   A col_DistReport is a single element of a distance report.  An array of
   these may be passed to a distance reporting function, which will report
   known distances (distances are known only for pairs with overlapping
   bounding boxes when the nbody algorithm is enabled or all active pairs
   where the nbody algorithm is disabled.
*/
typedef struct distance_report_struct
{
    int     ids[2];
    col_Fnode  *features[2];
    float   distance;
} col_DistReport;

/*
   The __col_State structures contains global state for the collision
   detection library.  It contains all the polytope structures and pair nodes.
   It also contains the list of active pair nodes, overlapping pair nodes,
   colliding pair nodes, and the sorted lists of bounding boxes.
*/
typedef struct collision_state_struct
{
    int         max_polytopes;
    __col_Polytope *polytopes;
    int         num_polytopes;

#ifdef COL_DENSE_MATRIX
    __col_PairNode    *pairs;
#endif /* COL_DENSE_MATRIX */
    
#ifdef COL_SPARSE_MATRIX
    __col_PairNode   **pairs;
#endif /*COL_SPARSE_MATRIX */
    
    int          matrix_rows;
    int          matrix_cols;

    __col_PairNode    *active_list;
    __col_BoxVert     *box_sort_list[3];
    __col_PairNode    *nbody_active_list;
    __col_PairNode    *colliding_list;
    char               using_nbody;
} __col_State;

/*---------------------------- Function Prototypes --------------------------*/


/*---------------------------Globals (externed)------------------------------*/



/* Protection from multiple includes. */
#endif /* INCLUDED_COLLISION_TYPES_H */



/*****************************************************************************\
\*****************************************************************************/

