Unit OS2Door;
{ Copyright (c) 1996 by Andrew Eigus
  Freeware door unit for OS/2

  Version 1.00 }

{$Cdecl-}

interface

uses Crt, OS2def, OS2base, AnsiIO;


  { common flags }

const
  LocalMode : boolean = True;    { local or remote mode flag }
  LocalOutput : boolean = True;  { controls local output }
  RemoteOutput : boolean = True; { controls remote output }
  ChatMode : boolean = False;    { chat mode flag }

const
  doorVersion = '1.02';
  doorVersionStr : PChar = #13#10#10'OS2DOOR v' + doorVersion +
    ' - OS/2 Pascal Door Interface by Andrew Eigus'#13#10 +
    'Copyright (c) 1996 by Andrew Eigus (mrbyte@andrews.net.lv)'+#13#10#10;

  doorSharedMemName = '\SHAREMEM\'; { shared memory name }

  doorChatInfoColor  : byte = White;     { Chat: header and footer text colors }
  doorChatSysopColor : byte = LightCyan; { Chat: Sysop text color }
  doorChatUserColor  : byte = Cyan;      { Chat: User text color }

  { DOORMSG_XXXX constants }

  DOORMSG_NODEINUSE    = 20;  { Node is in use by another OS2DOOR process }
  DOORMSG_BADNODE      = 21;  { Node number is 0 while the door is not running
                                in local mode, or otherwise }

  (*********************************************************************)
  (* Async Device Control                                              *)
  (*********************************************************************)

  IOCTL_ASYNC                  = $0001;

  ASYNC_SETBAUDRATE            = $0041;
  ASYNC_SETLINECTRL            = $0042;
  ASYNC_SETEXTDRATE            = $0043;
  ASYNC_TRANSMITIMM            = $0044;
  ASYNC_SETBREAKOFF            = $0045;
  ASYNC_SETMODEMCTRL           = $0046;
  ASYNC_STOPTRANSMIT           = $0047;
  ASYNC_STARTTRANSMIT          = $0048;
  ASYNC_SETBREAKON             = $004B;
  ASYNC_SETDCBINFO             = $0053;
  ASYNC_SETENHPARM             = $0054;
  ASYNC_GETBAUDRATE            = $0061;
  ASYNC_GETLINECTRL            = $0062;
  ASYNC_GETEXTDRATE            = $0063;
  ASYNC_GETCOMMSTATUS          = $0064;
  ASYNC_GETLINESTATUS          = $0065;
  ASYNC_GETMODEMOUTPUT         = $0066;
  ASYNC_GETMODEMINPUT          = $0067;
  ASYNC_GETINQUECOUNT          = $0068;
  ASYNC_GETOUTQUECOUNT         = $0069;
  ASYNC_GETCOMMERROR           = $006D;
  ASYNC_GETCOMMEVENT           = $0072;
  ASYNC_GETDCBINFO             = $0073;
  ASYNC_GETENHPARM             = $0074;

  (*********************************************************************)
  (* ASYNC_GETBAUDRATE, ASYNC_SETBAUDRATE                              *)
  (*********************************************************************)

  GETBAUDRATE_SIZE             = 2;
  SETBAUDRATE_SIZE             = 2;

  (*********************************************************************)
  (* ASYNC_GETLINECTRL, ASYNC_SETLINECTRL                              *)
  (*********************************************************************)

  GETLINECONTROL_SIZE          = 4;
  SETLINECONTROL_SIZE          = 3;

  (*********************************************************************)
  (* ASYNC_GETEXTDRATE, ASYNC_SETEXTDRATE                              *)
  (*********************************************************************)

  GETEXTDRATE_SIZE             = 15;
  SETEXTDRATE_SIZE             = 5;

  (*********************************************************************)
  (* ASYNC_TRANSMITIMM                                                 *)
  (*********************************************************************)

  TRANSMITIMM_SIZE             = 1;

  (*********************************************************************)
  (* ASYNC_GETCOMMERROR, ASYNC_SETBREAKOFF, ASYNC_SETBREAKON,          *)
  (* ASYNC_SETMODEMCTRL                                                *)
  (*********************************************************************)

  COMERROR_SIZE                = 2; { type: UShort }

  (*********************************************************************)
  (* ASYNC_SETMODEMCTRL                                                *)
  (*********************************************************************)

  MODEMSTATUS_SIZE             = 2;

  (*********************************************************************)
  (* ASYNC_GETDCBINFO, ASYNC_SETDCBINFO                                *)
  (*********************************************************************)

  GETDCBINFO_SIZE              = 11;
  SETDCBINFO_SIZE              = 11;

  (*********************************************************************)
  (* ASYNC_GETENHPARM, ASYNC_SETENHPARM                                *)
  (*********************************************************************)

  GETENHPARM_SIZE              = 5;
  SETENHPARM_SIZE              = 5;

  (*********************************************************************)
  (* ASYNC_GETCOMMSTATUS                                               *)
  (*********************************************************************)

  GETCOMMSTATUS_SIZE           = 1;

  (*********************************************************************)
  (* ASYNC_GETLINESTATUS                                               *)
  (*********************************************************************)

  GETLINESTATUS_SIZE           = 1;

  (*********************************************************************)
  (* ASYNC_GETMODEMINPUT, ASYNC_GETMODEMOUTPUT                         *)
  (*********************************************************************)

  GETMODEMIO_SIZE              = 1;

{ ASYNC_GETMODEMINPUT ----------------------------------- 7654 3210 }
  CTS_ON                       = $10;                   { ...1 .... }
  DSR_ON                       = $20;                   { ..1. .... }
  RI_ON                        = $40;                   { .1.. .... }
  DCD_ON                       = $80;                   { 1... .... }

{ ASYNC_GETMODEMOUTPUT ---------------------------------- 7654 3210 }
  DTR_ON                       = $01;                   { .... ...1 }
  RTS_ON                       = $02;                   { .... ..1. }


  (*********************************************************************)
  (* ASYNC_GETINQUECOUNT, ASYNC_GETOUTQUECOUNT                         *)
  (*********************************************************************)

type
  PIOQueue = ^TIOQueue;
  TIOQueue = record
    usQueueCount,
    usQueueSize : UShort
  end;

const
  GETIOQUEUE_SIZE              = 4;

  (*********************************************************************)
  (* ASYNC_GETCOMMEVENT                                                *)
  (*********************************************************************)

  GETCOMMEVENT_SIZE            = 2;

  { --------------------------------------------- FEDC BA98 7654 3210 }
  CHAR_RECEIVED                = $0001;         { .... .... .... ...1 }
  RX_TIMEOUT_INT               = $0002;         { .... .... .... ..1. }
  LAST_CHAR_SENT               = $0004;         { .... .... .... .1.. }
  CTS_CHANGED                  = $0008;         { .... .... .... 1... }
  DSR_CHANGED                  = $0010;         { .... .... ...1 .... }
  DCD_CHANGED                  = $0020;         { .... .... ..1. .... }
  BREAK_DETECTED               = $0040;         { .... .... .1.. .... }
  ERROR_OCCURRED               = $0080;         { .... .... 1... .... }
  RI_DETECTED                  = $0100;         { .... ...1 .... .... }

  (*********************************************************************)
  (* General Device Control                                            *)
  (*********************************************************************)

  IOCTL_GENERAL                = $000B;

  DEV_FLUSHINPUT               = $0001;
  DEV_FLUSHOUTPUT              = $0002;

  (*********************************************************************)
  (* DEV_FLUSHINPUT, DEV_FLUSHOUTPUT                                   *)
  (*********************************************************************)

  DEVFLUSHPRM_SIZE             = 1;
  DEVFLUSHDAT_SIZE             = 1;

  (*********************************************************************)
  (* General definitions                                               *)
  (*********************************************************************)

  MAX_SLEEP                    = 5;      { maximum time wait/read poll }

  RC_ASYNC_TIMED_OUT           = -1;      { Function timed out }
  RC_ASYNC_NO_SIGNAL           = -2;      { Function lost signal ie.DCD }
  RC_ASYNC_NO_INPUT            = -3;      { Function didn't detect data }


  { Miscellaneous constants not defined in COMSFUNC.H }

  NULLHANDLE                   = 0;

  TimeOutDelay : word = 1000; { timeout delay value for DoorKeyPressed,
                                DoorCarrier, ReadPort and other comm funcs }
  InputTimeOut : longint = 60 * 10; { set 10 minutes input timeout (default) }

  MustClosePort : boolean = False; { specifies whether ClosePort() should be
                                     called when the program finishes }

  DoorInputTimeoutSet : boolean = False;

  CRLF  = #13#10;
  CR    = #13;
  LF    = #10;
  BS    = #8;  { backspace }
  BELL  = #7;  { bell char }
  SPACE = ' '; { space }

  TERM_TTY  = 0;
  TERM_ANSI = 1;
  TERM_AVT  = 2;

type
  PByte = ^byte;

  PDoorData = ^TDoorData;
  TDoorData = record
    UserName : string[21];     { User's alias }
    UserCity : string[36];     { User's city }
    UserAccLev : word;         { User's access level }
    UserTime : longint;        { Time allowed while in a door (initially in minutes) }
    UserNode : byte;           { BBS Node number }
    UserBaud : ULong;          { Baud rate }
    UserTerm : byte;           { Terminal emulation (0=TTY,1=ANSI,2=AVATAR) }
  end;

var
  DoorData : TDoorData;
  doorScrWidth, doorScrHeight : byte;
  doorLastWindMin, doorLastWindMax : word;
  doorTimeStart, doorTimeNow, doorTimeElapsed, doorTimeLeft,
  doorTimeOutStart : longint;
  DoorExitProcSave, DoorXcptProcSave : pointer;
  LocalKeyPressed : boolean;
  WatchdogTid : TID;

function DoorInitWatchdog : ApiRet;
function DoorDisableWatchdog : ApiRet;
function DoorEnableWatchdog : ApiRet;
function DoorDoneWatchdog : ApiRet;
procedure DoorInit(hComPort : HFile); { initialize door interface }
function DoorTermStr(Term : byte) : string;
procedure DoorOutDebugInfo; { show the info about all internal variables }
function DoorKeyPressed : boolean;
function DoorReadKey : char;
procedure DoorWrite(const S : string);
procedure DoorWriteLn(const S : string);
procedure DoorWriteAZ(S : PChar);
procedure DoorWriteLnAZ(S : PChar);
procedure DoorPipeWrite(const S : string);
procedure DoorPipeWriteLn(const S : string);
function DoorReadLn(var S : string; Default : string; MaxLen : byte;
  Password : boolean) : byte;
function DoorKeyPrompt(Prompt, Keys : string; Default : char; PromptColor, KeysColor : byte) : char;
function DoorWhereX : byte;
function DoorWhereY : byte;
procedure DoorClrLine;
function DoorLoadConfig(DorInfPath : string) : boolean;
procedure DoorShowStatus(RedrawStatusLine : boolean);
function DoorMorePrompt(var CLine : byte) : boolean;
function DoorWaitToGo(Prompt : boolean) : char;
procedure DoorChat;
function DoorShowHelp(var HelpFile : text;
  const HelpPrompt : string; HelpIdxColor : byte) : boolean;
function DoorCarrier(hComPort : HFile; lTimeLimit : longint) : ApiRet;
procedure DoorInitTimer;
function DoorCalcTime : boolean; { returns false if over }
function DoorInputTimeout : boolean; { returns true if input timeout }
function DoorCarrierLost : boolean;
procedure DoorExitProc;

function CallDevice(hcomPort : HFile; ulCategory, ulFunction : ULong;
  pParameters, pDataPacket : pointer) : ApiRet;
function WaitInput(hComPort : HFile; lTimeLimit : longint) : ApiRet;
function OpenPort(pszDeviceName : PChar; var hComPort : HFile) : ApiRet;
function ClosePort(hComPort : HFile) : ApiRet;
function WritePort(hComPort : HFile;
  const Buffer; ulBuffLen : ULong; var ulRemains : ULong) : ApiRet;
function ReadPort(hComPort : HFile; var Buffer; ulBuffLen : ULong;
  var ulReadSize : ULong; lTimeLimit : longint) : ApiRet;
function GetModemInput(hcomPort : HFile; pbMdmInput : PByte) : ApiRet;
function GetComEvent(hcomPort : HFile; pusComEvent : PUShort) : ApiRet;
function GetRxCount(hComPort : HFile; var usRxCount, usRxSize : UShort) : ApiRet;
function GetTxCount(hComPort : HFile; var usTxCount, usTxSize : UShort) : ApiRet;


implementation

uses Dos, Strings, OS2Misc, VPUtils;

const
  CarrierAlreadyLost : boolean = False;
var
  ActivePort : HFile;
  LastTime : longint;
  pchShare : pointer; { shared memory block containing DoorData }

Procedure ShowLocalOutputInfo;
const OnOffStr : array[Boolean] of String[8] = ('disabled', 'enabled');
var OldWindMin, OldWindMax : word;
Begin
  OldWindMin := WindMin;
  OldWindMax := WindMax;
  Window(1, 1, Succ(Lo(doorLastWindMax)), Succ(Hi(doorLastWindMax)));
  TextAttr := LightGray; ClrScr;
  WindMin := OldWindMin;
  WindMax := OldWindMax;
  WriteLn('Local output ', OnOffStr[LocalOutput], '.  Press <Alt-O> to toggle.');
  WriteLn('Press <Alt-C> for chat.'+CRLF);
  if LocalOutput then DoorShowStatus(True)
End; { ShowLocalOutputInfo }

Procedure DoorWarning(const WMsg : string; MsgCode : byte);
Begin
  DoorWriteLn(CRLF + LF + AnsiAttr(LightRed) +
    'OS2DOOR' + LeadingZero(MsgCode, 3) + ': ' + WMsg + CRLF);
  DoorWaitToGo(False)
End; { DoorWarning }

Procedure DoorInit(hComPort : HFile);
var
  ShareName : string[20];
  rc : ApiRet;
Begin
  ActivePort := hComPort;
  if LocalMode then
    DoorData.UserTerm := TERM_ANSI; { ansi terminal for local }
  AnsiEnabled := DoorData.UserTerm <> TERM_TTY; { enable ansi output if non-tty }

  doorLastWindMin := WindMin;
  doorLastWindMax := WindMax;
  if not LocalMode then Dec(WindMax, $0100); { reserve space for status line }
  DoorShowStatus(True);
  DoorInitWatchdog; { init watchdog procedure }
  if (LocalMode and (DoorData.UserNode <> 0)) or
     (not LocalMode and (DoorData.UserNode = 0)) then
  begin
    DoorWarning('Node number 0 is always for local mode.', DOORMSG_BADNODE);
    LocalMode := True;
    DoorData.UserNode := 0
  end;
  ShareName := doorSharedMemName +
    GetNoExtPart(GetNamePart(ParamStr(0))) + Int2Str(DoorData.UserNode) + #0;
  if not LocalMode then
  begin
    rc := DosGetNamedSharedMem(pchShare, @ShareName[1], PAG_READ OR PAG_WRITE);
    DosFreeMem(pchShare);
    if rc = 0 then
    begin { current node is in use by another process }
      DoorWarning('BBS node ' + Int2Str(DoorData.UserNode) +
        ' is in use. The program is incorrectly configured.', DOORMSG_NODEINUSE);
      Halt(DOORMSG_NODEINUSE)
    end;
    DosAllocSharedMem(pchShare, @ShareName[1], 1024, PAG_READ OR PAG_WRITE OR PAG_COMMIT);
    Move(DoorData, pchShare, SizeOf(TDoorData));

    if LocalOutput = False then ShowLocalOutputInfo

  end else LocalOutput := True
End; { DoorInit }

Function DoorTermStr(Term : byte) : string;
Begin
  case Term of
    TERM_TTY: DoorTermStr := 'TTY';
    TERM_ANSI: DoorTermStr := 'ANSI';
    TERM_AVT: DoorTermStr := 'AVATAR';
    else DoorTermStr := 'UNKNOWN'
  end
End; { DoorTermStr }

Procedure DoorOutDebugInfo;

function GetBoolStr(Val : boolean) : string;
begin
  if Val = True then
    GetBoolStr := 'TRUE' else GetBoolStr := 'FALSE'
end; { GetBoolStr }

Begin
  DoorWriteLn('OS2DOOR INTERNAL VARIABLES:'+CRLF);
  with DoorData do
  begin
    DoorWriteLn(Concat('DoorData.UserName   : ', UserName));
    DoorWriteLn(Concat('DoorData.UserCity   : ', UserCity));
    DoorWriteLn(Concat('DoorData.UserAccLev : ', Int2Str(UserAccLev)));
    DoorWriteLn(Concat('DoorData.UserTime   : ', Int2Str(UserTime)));
    DoorWriteLn(Concat('DoorData.UserNode   : ', Int2Str(UserNode)));
    DoorWriteLn(Concat('DoorData.UserBaud   : ', Int2Str(UserBaud)));
    DoorWriteLn(Concat('DoorData.UserTerm   : ', Int2Str(UserTerm),
      ' (', DoorTermStr(UserTerm), ')'))
  end;
  DoorWrite(CRLF);
  DoorWriteLn(Concat('LocalMode           : ', GetBoolStr(LocalMode),
             '    ', 'LocalOutput         : ', GetBoolStr(LocalOutput)));
  DoorWriteLn(Concat('ActivePort          : ', Int2Str(ActivePort)));
  DoorWrite(CRLF);
  DoorWriteLn(Concat('doorScrWidth        : ', Int2Str(doorScrWidth)));
  DoorWriteLn(Concat('doorScrHeight       : ', Int2Str(doorScrHeight)));
  DoorWriteLn(Concat('doorLastWindMin     : ',
    Int2Str(doorLastWindMin), ' (', Int2Str(Succ(Lo(doorLastWindMin))), ',',
    Int2Str(Succ(Hi(doorLastWindMin))), ')'));
  DoorWriteLn(Concat('doorLastWindMax     : ',
    Int2Str(doorLastWindMax), ' (', Int2Str(Succ(Lo(doorLastWindMax))), ',',
    Int2Str(Succ(Hi(doorLastWindMax))), ')'));
  DoorWriteLn(Concat('doorTimeStart       : ', Int2Str(doorTimeStart)));
  DoorWriteLn(Concat('doorTimeNow         : ', Int2Str(doorTimeNow)));
  DoorWriteLn(Concat('doorTimeElapsed     : ', Int2Str(doorTimeElapsed)));
  DoorWriteLn(Concat('doorTimeLeft        : ', Int2Str(doorTimeLeft)));
  DoorWriteLn(Concat('doorTimeOutStart    : ', Int2Str(doorTimeOutStart)));
  DoorWrite(CRLF);
  DoorWaitToGo(True)
End; { DoorOutDebugInfo }

Function DoorKeyPressed : boolean;
var
  rxCount, rxSize : UShort;
  rc : ApiRet;
Begin
  LocalKeyPressed := False;
  DoorKeyPressed := False;
  if DoorCarrierLost then Exit;
  if LocalMode then
    DoorKeyPressed := KeyPressed
  else
  begin
    if KeyPressed then { key pressed on local terminal by the sysop? }
    begin
      LocalKeyPressed := True;
      DoorKeyPressed := True
    end else
    begin
      rc := GetRxCount(ActivePort, rxCount, rxSize);
      if (rxCount > 0) and (rc = 0) then
      begin
        DoorKeyPressed := True;
        DoorCalcTime; { calculate doorTimeNow }
        doorTimeOutStart := doorTimeNow
      end
    end
  end
End; { DoorKeyPressed }

Function DoorReadKey : char;
var
  C : char;
  rc : ApiRet;
  NumRead : ULong;
Begin
  if LocalMode or LocalKeyPressed then
  begin
    C := ReadKey;
    if (C = #0) and not ChatMode and not LocalMode then
      case ReadKey of
        #$18: { Alt-O }
        begin
          LocalOutput := not LocalOutput;
          ShowLocalOutputInfo;
          C := #0
        end;
        #$2E: { Alt-C }
        begin
          DoorChat;
          C := #0
        end
      end;
    if not LocalOutput then C := #0; { if no local output, then return 0 }
    DoorReadKey := C
  end else
  begin
    repeat
      NumRead := 1;
      rc := ReadPort(ActivePort, C, 1, NumRead, TimeOutDelay)
    until (rc <> RC_ASYNC_TIMED_OUT);
    DoorReadKey := C
  end
End; { DoorReadKey }

Procedure DoorWrite(const S : string);
var
  rc : ApiRet;
  NumWritten, NumLeft : ULong;
  Len : byte;
Begin
  if S = '' then Exit;
  if LocalMode then
  begin
    if LocalOutput then
      AnsiWrite(S)
  end else
  begin
    if RemoteOutput then
    begin
      NumWritten := 0;
      Len := Length(S);
      repeat
        rc := WritePort(ActivePort,
          S[NumWritten + 1], Len - NumWritten, NumLeft);
        Inc(NumWritten, Len - NumLeft)
      until (NumLeft = 0) or (rc <> NO_ERROR)
    end;
    if LocalOutput then AnsiWrite(S)
  end
End; { DoorWrite }

Procedure DoorWriteLn(const S : string);
Begin
  DoorWrite(S + CRLF)
End; { DoorWriteLn }

Procedure WritePChar(S : PChar);
var
  i, c : word;
  p : string;
Begin
  {
  b := StrLen(S);
  if b > 0 then
    for i := 0 to b - 1 do AnsiWrite(S[I])
    }
  c := 0;
  while c < StrLen(S) do
  begin
    i := StrLen(S) - c;
    if i > High(p) then i := High(p);
    p[0] := Chr(i);
    Move(S[c], p[1], i);
    AnsiWrite(p);
    Inc(c, i)
  end
End; { WritePChar }

Procedure DoorWriteAZ(S : PChar);
var
  rc : ApiRet;
  NumWritten, NumLeft : ULong;
  Len : UShort;
Begin
  if (StrLen(S) = 0) or not Assigned(S) then Exit;
  if LocalMode then
  begin
    if LocalOutput then
      WritePChar(S)
  end else
  begin
    if RemoteOutput then
    begin
      NumWritten := 0;
      Len := StrLen(S);
      repeat
        rc := WritePort(ActivePort,
          S[NumWritten], Len - NumWritten, NumLeft);
        Inc(NumWritten, Len - NumLeft)
      until (NumLeft = 0) or (rc <> NO_ERROR)
    end;
    if LocalOutput then WritePChar(S)
  end
End; { DoorWriteAZ }

Procedure DoorWriteLnAZ(S : PChar);
Begin
  DoorWriteAZ(S);
  DoorWrite(CRLF)
End; { DoorWriteLnAZ }

Procedure DoorPipeWrite(const S : string);
var
  i, lens : byte;
  ccode : string[3];
  out : string;
Begin
  i := 1;
  out := '';
  lens := Length(S);
  while i <= lens do
  begin
    if S[i] = '|' then
    begin
      ccode := '';
      if (i < lens) and (S[i + 1] in ['0'..'9']) then
      begin
        Inc(i);
        while S[i] in ['0'..'9'] do
        begin
          ccode := ccode + S[i];
          Inc(i)
        end;
        Dec(i);
        if ccode <> '' then
        begin
          DoorWrite(out); { flush output }
          DoorWrite(AnsiAttr(Str2Int(ccode)));
          out := ''
        end
      end else out := out + S[i]
    end else out := out + S[i];
    Inc(i)
  end;
  DoorWrite(out) { flush output }
End; { DoorPipeWrite }

Procedure DoorPipeWriteLn(const S : string);
Begin
  DoorPipeWrite(S + CRLF)
End; { DoorPipeWriteLn }

Function DoorReadLn(var S : string; Default : string; MaxLen : byte;
  Password : boolean) : byte;
var
  DoneInput : boolean;
  Ch : char;
Begin
  DoneInput := False;
  S := '';
  repeat
    repeat
      if not DoorKeyPressed then DosSleep(MAX_SLEEP)
    until DoorKeyPressed or DoorCarrierLost or not DoorCalcTime or DoorInputTimeout;
    if DoorKeyPressed then
    begin
      Ch := DoorReadKey;
      case Ch of
        BS: if S <> '' then
        begin
          Dec(S[0]);
          DoorWrite(#8' '#8)
        end;
        CR:
        begin
          DoorWriteLn('');
          DoneInput := True;
          if S = '' then S := Default
        end;
        else
        if (Length(S) < MaxLen) and not (Ch in [#0,#10]) then
        begin
          S := S + Ch;
          if Password then DoorWrite('*') else DoorWrite(Ch)
        end else DoorWrite(BELL)
      end
    end
  until DoneInput or DoorCarrierLost or not DoorCalcTime or DoorInputTimeout;
  if not DoneInput then S := '';
  { ^^ force output string to be empty if lost carrier or time is over }
  DoorReadLn := Length(S)
End; { DoorReadLn }

Function DoorKeyPrompt(Prompt, Keys : string; Default : char; PromptColor, KeysColor : byte) : char;
var
  I : byte;
  Ch : char;

function Done : boolean;
begin
  I := 1;
  while (I < Length(Keys)) and (Ch <> Keys[I]) do Inc(I);
  Done := (I <= Length(Keys)) or (Ch = CR)
end; { Done }

Begin
  if PromptColor <> 0 then DoorWrite(AnsiAttr(PromptColor));
  DoorWrite(Concat(Prompt, ' ['));
  Keys := ToUpper(Keys);
  Default := UpCase(Default);
  for I := 1 to Length(Keys) do
  begin
    if KeysColor <> 0 then DoorWrite(AnsiAttr(KeysColor));
    if Keys[I] = Default then
      DoorWrite(Default)
    else
      DoorWrite(LoCase(Keys[I]));
    if PromptColor <> 0 then DoorWrite(AnsiAttr(PromptColor));
    if I < Length(Keys) then
      DoorWrite('/')
  end;
  DoorWrite('] ');
  repeat
    repeat
      if not DoorKeyPressed then DosSleep(MAX_SLEEP)
    until DoorKeyPressed or DoorCarrierLost or not DoorCalcTime or DoorInputTimeout;
    Ch := UpCase(DoorReadKey);
    DoorWrite(Ch);
    if not Done then DoorWrite(#7#8)
  until Done;
  DoorWrite(CRLF);
  if Ch = CR then
    DoorKeyPrompt := Default
  else
    DoorKeyPrompt := Keys[I]
End; { DoorKeyPrompt }

Function DoorWhereX : byte;
Begin
  if LocalMode then
    DoorWhereX := WhereX
End; { DoorWhereX }

Function DoorWhereY : byte;
Begin
  if LocalMode then
    DoorWhereY := WhereY
End; { DoorWhereY }

Procedure DoorClrLine;
var I, C : byte;
Begin
  if (DoorData.UserTerm = TERM_TTY) and not LocalMode then
  begin
    if LocalOutput then
      while WhereX > 1 do DoorWrite(#8' '#8)
    else
    begin
      DoorWrite(CR);
      for I := 1 to doorScrWidth do DoorWrite(' ');
      DoorWrite(#8+CR)
    end
  end
  else DoorWrite(#13 + AnsiClrEol)
End; { DoorClrLine }

Function DoorLoadConfig(DorInfPath : string) : boolean;
var
  dorinf : text;
  I : byte;
  S : string[80];
Begin
  FillChar(DoorData, SizeOf(TDoorData), 0);
  DorInfPath := ToUpper(DorInfPath);
  Assign(dorinf, DorInfPath);
  Reset(dorinf);
  if IOResult = 0 then
  begin
    I := Length(DorInfPath);
    while (I >= 1) and (DorInfPath[I] <> '\') do
      Dec(I);
    if DorInfPath[I] = '\' then Inc(I);
    DorInfPath := ToUpper(Copy(DorInfPath, I, Length(DorInfPath)));
    if DorInfPath = 'DOOR.SYS' then
    begin
      ReadLn(dorinf, S); { read com port info }
      LocalMode := (ToUpper(S) = 'COM0:'); { check if local mode }
      ReadLn(dorinf, DoorData.UserBaud); { read baud rate }
      ReadLn(dorinf); { skip stopbits }
      ReadLn(dorinf, DoorData.UserNode); { read node number }
      ReadLn(dorinf); { skip locked baud rate }
      for I := 1 to 4 do ReadLn(dorinf);
      { skip screen display, printer toggle, page bell and caller alarm }
      ReadLn(dorinf, DoorData.UserName); { read user's name (alias) }
      ReadLn(dorinf, DoorData.UserCity); { read user's city }
      for I := 1 to 3 do ReadLn(dorinf);
      { skip user's home, data phones, and password }
      ReadLn(dorinf, DoorData.UserAccLev); { read user's access number }
      for I := 1 to 3 do ReadLn(dorinf);
      { skip # of calls, last date called, seconds remaining }
      ReadLn(dorinf, DoorData.UserTime); { read user's time allowed in a door }
      DoorData.UserTime := DoorData.UserTime * 60; { convert to seconds }
      ReadLn(dorinf, S); { read terminal emulation }
      S := ToUpper(S);
      if S = 'GR' then
        DoorData.UserTerm := TERM_ANSI; { non-TTY }
      ReadLn(dorinf, doorScrHeight); { read page length }
      { ... rest we don't need :) }
    end else
    if Copy(DorInfPath, 1, 7) = 'DORINFO' then
    begin
      for I := 1 to 3 do ReadLn(dorinf);
      { skip BBS name, sysop first and last names }
      ReadLn(dorinf, S);
      LocalMode := (ToUpper(S) = 'COM0'); { check if local mode }
      ReadLn(dorinf, S);
      DoorData.UserBaud := Str2Int(Copy(S, 1, Pos(' ', S) - 1));
      ReadLn(dorinf); { skip networking }
      ReadLn(dorinf, DoorData.UserName); { read user's first name }
      ReadLn(dorinf, S); { read user's last name }
      if S <> '' then
        DoorData.UserName := Concat(DoorData.UserName, ' ', S);
      ReadLn(dorinf, DoorData.UserCity); { read city }
      ReadLn(dorinf, DoorData.UserTerm); { read graphics byte }
      ReadLn(dorinf, DoorData.UserAccLev); { read user's access level }
      ReadLn(dorinf, DoorData.UserTime); { read time remaining }
      DoorData.UserTime := DoorData.UserTime * 60  { convert to seconds }
      { ... rest we don't use }
    end;
    Close(dorinf);
    DoorLoadConfig := True
  end else DoorLoadConfig := False
End; { DoorLoadConfig }

Procedure DoorShowStatus(RedrawStatusLine : boolean);
var
  OldWhereX, OldWhereY, OldTextAttr : byte;
  OldWindMin, OldWindMax : word;
  TimeLeftStr : string[10];
Begin
  if LocalMode or (not RedrawStatusLine and (doorTimeLeft = LastTime)) or
     not LocalOutput then Exit;
  HideCursor;
  OldWhereX := WhereX;
  OldWhereY := WhereY;
  OldWindMin := WindMin;
  OldWindMax := WindMax;
  OldTextAttr := TextAttr;
  Window(1, Succ(Hi(doorLastWindMax)), Succ(Lo(doorLastWindMax)), Succ(Hi(doorLastWindMax)));
  TextAttr := Black + LightGray shl 4;
  if RedrawStatusLine then
  begin
    ClrEol;
    GotoXY(2, 1); Write(DoorData.UserName);
    GotoXY(26, 1);
    Write(Concat('Node: ', Int2Str(DoorData.UserNode),
                 '  Baud: ', Int2Str(DoorData.UserBaud),
                 '  Term: ', DoorTermStr(DoorData.UserTerm)));
    GotoXY(doorScrWidth - 16, 1);
    Write('Time:')
  end;
  TimeLeftStr := LJustify(Concat(Int2Str(doorTimeLeft div 60),
    ':', LeadingZero(doorTimeLeft mod 60, 2)), 7);
  GotoXY(doorScrWidth - 10, 1);
  Write(TimeLeftStr);
  LastTime := doorTimeLeft;
  WindMin := OldWindMin;
  WindMax := OldWindMax;
  GotoXY(OldWhereX, OldWhereY);
  TextAttr := OldTextAttr;
  ShowCursor
End; { DoorShowStatus }

Function DoorMorePrompt(var CLine : byte) : boolean;
Begin
  if CLine = doorScrHeight then
  begin
    DoorWrite('[-more-]');
    DoorMorePrompt := not (DoorWaitToGo(False) in [#27, 'Q', 'q']);
    DoorClrLine;
    CLine := 0
  end else DoorMorePrompt := True
End; { DoorMorePrompt }

Function DoorWaitToGo(Prompt : boolean) : char;
var
  Done : boolean;
  Ch : char;
Begin
  if Prompt then DoorWrite('[-Press ENTER or SPACE to continue-]');
  Done := False;
  repeat
    repeat
      if not DoorKeyPressed then DosSleep(MAX_SLEEP)
    until DoorKeyPressed or DoorCarrierLost or not DoorCalcTime or DoorInputTimeout;
    Ch := DoorReadKey;
    if Prompt then
      Done := Ch in [CR, SPACE]
    else
      Done := Ch <> #0
  until Done or DoorCarrierLost or not DoorCalcTime or DoorInputTimeout;
  if Prompt then DoorClrLine;
  DoorWaitToGo := Ch
End; { DoorWaitToGo }

Procedure DoorChat;
var
  Done : boolean;
  Ch : char;
  OldOutputFlag : boolean;
Begin
  ChatMode := True;
  OldOutputFlag := LocalOutput;
  LocalOutput := True;
  DoorWriteLn(CRLF+CRLF+AnsiAttr(doorChatInfoColor)+
    '* SysOp has dropped you into chat: *'+CRLF);
  DoorWrite(AnsiAttr(doorChatSysopColor));
  Done := False;
  repeat
    if DoorKeyPressed then
    begin
      Ch := DoorReadKey;
      case Ch of
        BS: DoorWrite(BS+' '+BS);
        CR: DoorWrite(CRLF);
        LF,#0:;
        else if LocalKeyPressed then
        begin
          if Ch = #27 then
            Done := True
          else
          begin
            if TextAttr <> doorChatSysopColor then
              DoorWrite(AnsiAttr(doorChatSysopColor)+Ch)
            else DoorWrite(Ch);
          end
        end else
          if TextAttr <> doorChatUserColor then
            DoorWrite(AnsiAttr(doorChatUserColor)+Ch)
          else DoorWrite(Ch)
      end
    end else DosSleep(MAX_SLEEP)
  until Done;
  DoorWriteLn(AnsiAttr(doorChatInfoColor)+CRLF+'* Chat ended *'+CRLF);
  LocalOutput := OldOutputFlag;
  if LocalOutput = False then ShowLocalOutputInfo;
  ChatMode := False
End; { DoorChat }

Function DoorShowHelp(var HelpFile : text;
  const HelpPrompt : string; HelpIdxColor : byte) : boolean;
const
  H_BEGIN_ID : string[5] = '[BEG]';
  H_END_ID : string[5] = '[END]';
var
  S : string;
  InStr : string[2];
  Lines, OldFM : byte;
  Error, Done : boolean;
Begin
  OldFM := FileMode;
  FileMode := 0;
  Error := False;
  Done := False;
  while not Done and not Error and not DoorInputTimeout do
  begin
    Reset(HelpFile);
    if IOResult = 0 then
    begin
      Lines := 0;
      DoorWrite(CRLF+AnsiAttr(HelpIdxColor));
      while not Eof(HelpFile) do
      begin
        ReadLn(HelpFile, S);
        if Copy(S, 1, 5) = H_BEGIN_ID then
        begin
          Inc(Lines);
          if not DoorMorePrompt(Lines) then Break;
          DoorPipeWriteLn(Copy(S, 6, Length(S)));
          repeat
            ReadLn(HelpFile, S)
          until Eof(HelpFile) or (Copy(S, 1, 5) = H_END_ID)
        end
      end;
      DoorWrite(HelpPrompt);
      if DoorReadLn(PString(@InStr)^, '', 2, False) > 0 then
      begin
        Reset(HelpFile);
        while not Eof(HelpFile) do
        begin
          ReadLn(HelpFile, S);
          if Copy(S, 1, 5) = H_BEGIN_ID then
          begin
            Delete(S, 1, 5);
            S := LTrim(S);
            Lines := 0;
            if Copy(S, 1, Length(InStr)) = InStr then
            begin
              repeat
                ReadLn(HelpFile, S);
                if Copy(S, 1, 5) = H_END_ID then Break;
                Inc(Lines);
                if not DoorMorePrompt(Lines) then Break;
                DoorPipeWriteLn(S)
              until Eof(HelpFile);
              DoorWaitToGo(True)
            end
          end
        end
      end else Done := True
    end else Error := True
  end;
  FileMode := OldFM;
  DoorShowHelp := Error
End; { DoorShowHelp }

Function CallDevice(hcomPort : HFile; ulCategory, ulFunction : ULong;
  pParameters, pDataPacket : pointer) : ApiRet;
var
  rc : ApiRet;
  pParmPkt, pDataPkt : pointer;
  ulParmLen, ulDataLen : ULong;
  pulParmLen, pulDataLen : PULong;
  ulSaveLen : ULong;
  bGenParm, bGenData : byte;
Begin
  rc := NO_ERROR;
  ulParmLen := 0;
  ulDataLen := 0;
  pulParmLen := @ulParmLen;
  pulDataLen := @ulDataLen;
  ulSaveLen := 0;
  case ulCategory of
    IOCTL_ASYNC:
    begin
      pParmPkt := pParameters;
      pDataPkt := pDataPacket;
      case ulFunction of
        ASYNC_GETBAUDRATE: ulDataLen := GETBAUDRATE_SIZE;
        ASYNC_SETBAUDRATE: ulParmLen := SETBAUDRATE_SIZE;
        ASYNC_GETLINECTRL: ulDataLen := GETLINECONTROL_SIZE;
        ASYNC_SETLINECTRL: ulParmLen := SETLINECONTROL_SIZE;
        ASYNC_GETEXTDRATE: ulDataLen := GETEXTDRATE_SIZE;
        ASYNC_SETEXTDRATE: ulParmLen := SETEXTDRATE_SIZE;
        ASYNC_TRANSMITIMM: ulParmLen := TRANSMITIMM_SIZE;
        ASYNC_SETBREAKON: ulDataLen := COMERROR_SIZE;
        ASYNC_SETBREAKOFF: ulDataLen := COMERROR_SIZE;
        ASYNC_SETMODEMCTRL:
        begin
          ulParmLen := MODEMSTATUS_SIZE;
          ulDataLen := COMERROR_SIZE
        end;
        ASYNC_GETDCBINFO: ulDataLen := GETDCBINFO_SIZE;
        ASYNC_SETDCBINFO: ulParmLen := SETDCBINFO_SIZE;
        ASYNC_GETENHPARM: ulDataLen := GETENHPARM_SIZE;
        ASYNC_SETENHPARM: ulParmLen := SETENHPARM_SIZE;
        ASYNC_GETCOMMSTATUS: ulDataLen := GETCOMMSTATUS_SIZE;
        ASYNC_GETLINESTATUS: ulDataLen := GETLINESTATUS_SIZE;
        ASYNC_GETMODEMINPUT: ulDataLen := GETMODEMIO_SIZE;
        ASYNC_GETMODEMOUTPUT: ulDataLen := GETMODEMIO_SIZE;
        ASYNC_GETINQUECOUNT: ulDataLen := GETIOQUEUE_SIZE;
        ASYNC_GETOUTQUECOUNT: ulDataLen := GETIOQUEUE_SIZE;
        ASYNC_GETCOMMERROR: ulDataLen := COMERROR_SIZE;
        ASYNC_GETCOMMEVENT: ulDataLen := GETCOMMEVENT_SIZE;
        else rc := ERROR_INVALID_PARAMETER
      end
    end;
    IOCTL_GENERAL:
    begin
      pParmPkt := @bGenParm;
      pDataPkt := @bGenData;
      case ulFunction of
        DEV_FLUSHINPUT:
        begin
          ulParmLen := DEVFLUSHPRM_SIZE;
          ulDataLen := DEVFLUSHDAT_SIZE
        end;
        DEV_FLUSHOUTPUT:
        begin
          ulParmLen := DEVFLUSHPRM_SIZE;
          ulDataLen := DEVFLUSHDAT_SIZE
        end;
        else rc := ERROR_INVALID_PARAMETER
      end
    end;
    else rc := ERROR_INVALID_PARAMETER
  end;
  if hcomPort = NULLHANDLE then
    rc := ERROR_INVALID_PARAMETER;
  if rc = NO_ERROR then
  begin
    ulSaveLen := ulDataLen;
    if ulParmLen = 0 then
    begin
      pParmPkt := nil;
      pulParmLen := nil
    end;
    if ulDataLen = 0 then
    begin
      pDataPkt := nil;
      pulDataLen := nil
    end;
    if pDataPkt <> nil then
      FillChar(pDataPkt^, ulDataLen, 0);
      { memset( pDataPkt, (BYTE)0, ulDataLen );       /* zeroise it */ }
    rc := DosDevIOCtl(hcomPort, ulCategory, ulFunction, pParmPkt, ulParmLen,
      pulParmLen, pDataPkt, ulDataLen, pulDataLen);

    if rc = NO_ERROR then
      if ulDataLen <> ulSaveLen then
        rc := ERROR_GEN_FAILURE;    (* Didn't return enough data  *)
(*
    if ( rc != NO_ERROR ) {
         fprintf( stdout,
                  "DosDevIOCtl(0x%4.4lX,0x%4.4lX) failed with RC=%d.\r\n",
                  ulCategory,
                  ulFunction,
                  rc );
      }
*)
  end;
  CallDevice := rc                      (* Return DosDevIOCtl result  *)
End; { CallDevice }

Function GetModemInput(hcomPort : HFile; pbMdmInput : PByte) : ApiRet;
Begin
  GetModemInput := CallDevice(hcomPort,
    IOCTL_ASYNC, ASYNC_GETMODEMINPUT, nil, pbMdmInput)
End; { GetModemInput }

Function GetComEvent(hComPort : HFile; pusComEvent : PUShort) : ApiRet;
Begin
  GetComEvent := CallDevice(hComPort,
    IOCTL_ASYNC, ASYNC_GETCOMMEVENT, nil, pusComEvent)
End; { GetComEvent }

Function GetRxCount(hComPort : HFile; var usRxCount, usRxSize : UShort) : ApiRet;
var
  rc : ApiRet;
  ioqueStatus : TIOQueue;
Begin
  rc := CallDevice(hComPort, IOCTL_ASYNC, ASYNC_GETINQUECOUNT, nil, @ioqueStatus);
  if rc = NO_ERROR then
  begin
    usRxCount := ioqueStatus.usQueueCount;
    usRxSize := ioqueStatus.usQueueSize
  end;
  GetRxCount := rc
End; { GetRxCount }

Function GetTxCount(hComPort : HFile; var usTxCount, usTxSize : UShort) : ApiRet;
var
  rc : ApiRet;
  ioqueStatus : TIOQueue;
Begin
  rc := CallDevice(hComPort, IOCTL_ASYNC, ASYNC_GETOUTQUECOUNT, nil, @ioqueStatus);
  if rc = NO_ERROR then
  begin
    usTxCount := ioqueStatus.usQueueCount;
    usTxSize := ioqueStatus.usQueueSize
  end;
  GetTxCount := rc
End; { GetTxCount }

Function WaitInput(hComPort : HFile; lTimeLimit : longint) : ApiRet;
{ Wait for modem input; returns: 0=data, -1=timeout, -3=none }
var
  rc : ApiRet;
  usComEvent : UShort;
  fTimedOut : bool;
  lElapsed : longint;
  tTimeStart, tTimeNow : OS2base.DateTime;
Begin
  fTimedOut := False;
  DosGetDateTime(tTimeStart);
  repeat
    rc := GetComEvent(hComPort, @usComEvent); { check com event  }
    if rc <> NO_ERROR then
      fTimedOut := True
    else
    if (usComEvent and CHAR_RECEIVED) <> 0 then
      fTimedOut := True
    else
    if lTimeLimit = 0 then
    begin
      rc := RC_ASYNC_NO_INPUT;
      fTimedOut := True
    end else
    if lTimeLimit < 0 then
      DosSleep(MAX_SLEEP)
    else
    begin { check for timeout }
      DosGetDateTime(tTimeNow);
      lElapsed := MyDiffTime(tTimeNow, tTimeStart);
      if lElapsed > lTimeLimit then
      begin
        rc := RC_ASYNC_TIMED_OUT;        { force timed_out return }
        fTimedOut := True
      end else DosSleep(MAX_SLEEP)
    end
  until fTimedOut;
  WaitInput := rc
End; { WaitInput }

Function OpenPort(pszDeviceName : PChar; var hComPort : HFile) : ApiRet;
var
  ulOpenAction : ULong;
  rc : ApiRet;
Begin
  rc := DosOpen(pszDeviceName, hComPort, ulOpenAction, 0, file_Normal,
    file_Open, open_access_ReadWrite or open_share_DenyNone, nil);
  if (rc <> NO_ERROR) then hComPort := HFile(-1);
  OpenPort := rc
End; { OpenPort }

Function ClosePort(hComPort : HFile) : ApiRet;
Begin
  ClosePort := DosClose(hComPort)
End; { ClosePort }

Function WritePort(hComPort : HFile;
  const Buffer; ulBuffLen : ULong; var ulRemains : ULong) : ApiRet;
var
  rc : ApiRet;
  ulWritten : ULong;
Begin
  ulWritten := 0;
  if ulBuffLen > 0 then
    rc := DosWrite(hComPort, Buffer, ulBuffLen, ulWritten);
  {
      if ( rc != NO_ERROR ) {
         fprintf( stdout, "DosWrite(%s) failed with RC=%d.\r\n", pszBuffer, rc );
  }

  ulRemains := ulBuffLen - ulWritten;
  WritePort := rc
End; { WritePort }

Function ReadPort(hComPort : HFile; var Buffer; ulBuffLen : ULong;
  var ulReadSize : ULong; lTimeLimit : longint) : ApiRet;
var
  rc : ApiRet;
  usComEvent : UShort;
  fWaitForMore, fTimedOut, fStopReading, fByteFound : bool;
  ulNumToRead, ulActuallyRead, ulTotalRead : ULong;
  lElapsed : longint;
  tTimeStart, tTimeNow : OS2base.DateTime;
  pbBuffer : pointer;
Begin
  DosGetDateTime(tTimeStart);
  ulTotalRead := 0;
  fTimedOut := False;
  fStopReading := False;
  if ulReadSize > 0 then
    ulNumToRead := ulReadSize; { block mode is the ONLY supported here }
  if (ulReadSize = 0) or (ulReadSize > ulBuffLen) then
    rc := ERROR_INVALID_PARAMETER
  else
  begin
    rc := GetComEvent(hComPort, @usComEvent); { clear comm event word }
    if rc = NO_ERROR then
    begin
      pbBuffer := @Buffer;
      repeat
        fWaitForMore := False;
        repeat
          ulActuallyRead := 0;
          FillChar(Buffer, ulNumToRead, 0); { zeroise read buffer }
          rc := DosRead(hComPort, pbBuffer^, ulNumToRead, ulActuallyRead);
          if (rc = NO_ERROR) or (rc = ERROR_MORE_DATA) then
          begin
            if ulActuallyRead <> ulNumToRead then
              fWaitForMore := True;
            if ulActuallyRead > 0 then
            begin
              Inc(Longint(pbBuffer), ulActuallyRead);
              Inc(ulTotalRead, ulActuallyRead);
              if ulReadSize > 0 then { block mode }
              begin
                { number to read is the amount left unread }
                { or the block size if that is the lesser. }
                ulNumToRead := ulBuffLen - ulTotalRead;
                if ulReadSize < ulNumToRead then
                  ulNumToRead := ulReadSize;
                if ulNumToRead <= 0 then
                  fStopReading := True
              end else
              if fWaitForMore then
              begin { didn't receive enough data, wait for more } end else
              if lTimeLimit < 0 then
              begin { read again, don't care about timeout } end
              else
              begin
                DosGetDateTime(tTimeNow);
                lElapsed := MyDiffTime(tTimeNow, tTimeStart);
                if lElapsed > lTimeLimit then { explicit timeout }
                begin
                  rc := RC_ASYNC_TIMED_OUT;
                  fTimedOut := True
                end
              end
            end
          end
        until (fStopReading or fTimedOut or fWaitForMore);
        if fWaitForMore then
        begin
          fByteFound := False;
          repeat
            rc := GetComEvent(hComPort, @usComEvent); { check com event }
            if rc <> NO_ERROR then
              fTimedOut := True
            else
            if (usComEvent and CHAR_RECEIVED) <> 0 then { got some data }
              fByteFound := True  { exit wait loop }
            else if lTimeLimit < 0 then
              DosSleep(MAX_SLEEP) { wait 1/100th sec }
            else
            begin { check timeout }
              DosGetDateTime(tTimeNow);
              lElapsed := MyDiffTime(tTimeNow, tTimeStart);
              if lElapsed > lTimeLimit then
              begin { time limit exceeded }
                rc := RC_ASYNC_TIMED_OUT;
                fTimedOut := True
              end else DosSleep(MAX_SLEEP) { wait 1/100th sec }
            end
          until fByteFound or fTimedOut;
        end { endif wait for more data to arrive }
      until fStopReading or fTimedOut;
    end { endif get comm event okay }
  end; { endif read size okay }
  ulBuffLen := ulTotalRead; { return true length }
  ReadPort := rc
End; { ReadPort }

Function DoorCarrier(hComPort : HFile; lTimeLimit : longint) : ApiRet;
{ test for carrier; returns: 0=DCD, -1=timeout, 2=noDCD }
var
  rc : ApiRet;
  bComInput : byte;              (* Modem Input Signals        *)
  fTimedOut : bool;              (* Function has timed out     *)
  lElapsed : longint;            (* Time elapsed in millisecs  *)
  tTimeStart : OS2base.DateTime; (* Timer start time           *)
  tTimeNow : OS2base.DateTime;   (* Time now                   *)
Begin
  if LocalMode then
  begin
    DoorCarrier := 0; { always DCD for local mode }
    Exit
  end;
  if CarrierAlreadyLost then
  begin
    DoorCarrier := 2;
    Exit
  end;

  fTimedOut := False;
  DosGetDateTime(tTimeStart);

  repeat
    rc := GetModemInput(hComPort, @bComInput); { check mdm input }
    if rc <> NO_ERROR then
      fTimedOut := True
    else
    if (bComInput and DCD_ON) <> 0 then
      fTimedOut := True
    else
    if lTimeLimit = 0 then
    begin
      rc := RC_ASYNC_NO_SIGNAL;
      fTimedOut := True
    end else
    if lTimeLimit < 0 then
      DosSleep(MAX_SLEEP) { wait 1/100th second }
    else
    begin
      DosGetDateTime(tTimeNow);
      lElapsed := MyDiffTime(tTimeNow, tTimeStart);
      if lElapsed > lTimeLimit then
      begin
        rc := RC_ASYNC_TIMED_OUT;
        fTimedOut := True
      end else DosSleep(MAX_SLEEP) { wait 1/100th second }
    end
  until fTimedOut;
  if rc <> 0 then
  begin
    RemoteOutput := False; { disable all remote output }
    LocalOutput := False;   { disable all local output }
    CarrierAlreadyLost := True
  end;
  DoorCarrier := rc
End; { DoorCarrier }

Procedure DoorInitTimer;
Begin
  doorTimeStart := GetTimemSec div 1000;
  doorTimeOutStart := doorTimeStart
End; { DoorInitTimer }

Function DoorCalcTime : boolean;
var T1, T2 : longint;
Begin
  if not LocalMode then
  begin
    doorTimeNow := GetTimemSec div 1000;
    T1 := doorTimeStart;
    T2 := doorTimeNow;
    while T2 < T1 do
    begin
      Inc(T1, 12 * 3600);
      Inc(T2, 12 * 3600)
    end;
    doorTimeElapsed := T2 - T1;
    doorTimeLeft := DoorData.UserTime - doorTimeElapsed;
    DoorCalcTime := doorTimeLeft > 0
  end else DoorCalcTime := True
End; { DoorCalcTime }

Function DoorInputTimeout : boolean;
var lTimeOut : longint;
Begin
  { doorTimeNow holds current time (remember to set it first!) }
  if LocalMode then
    DoorInputTimeout := False
  else
  if not DoorInputTimeoutSet then
  begin
    DoorCalcTime;
    DoorInputTimeoutSet := (doorTimeNow - doorTimeoutStart) > InputTimeOut;
    DoorInputTimeout := DoorInputTimeoutSet
  end else DoorInputTimeout := True
End; { DoorInputTimeout }

Function DoorCarrierLost : boolean;
Begin
  DoorCarrierLost := DoorCarrier(ActivePort, TimeOutDelay) <> 0
End; { DoorCarrierLost }

Function DoorWatchdogProc(P : pointer) : longint;
{ checks for carrier, timelimit or input timeout limit }
var TimeLimit : boolean;
Begin
  repeat
    DosSleep(MAX_SLEEP);
    TimeLimit := not DoorCalcTime;
    DoorShowStatus(False)
  until DoorCarrierLost or TimeLimit or DoorInputTimeoutSet;
  Halt(13)
  {asm call DoorExitProc end;
  DosExit(EXIT_PROCESS, 13);} { exit program with fatal exit code }
End; { DoorWatchdogProc }

Function DoorInitWatchdog : ApiRet;
Begin
  if LocalMode then
    DoorInitWatchdog := 0
  else
  begin
    WatchdogTid := VPBeginThread(DoorWatchdogProc, 16384, nil);
    DoorInitWatchdog := WatchdogTid
  end
End; { DoorInitWatchdog }

Function DoorDisableWatchdog : ApiRet;
Begin
  DoorDisableWatchdog := SuspendThread(WatchdogTid)
End; { DoorDisableWatchdog }

Function DoorEnableWatchdog : ApiRet;
Begin
  DoorEnableWatchdog := ResumeThread(WatchdogTid)
End; { DoorEnableWatchdog }

Function DoorDoneWatchdog : ApiRet;
Begin
  if LocalMode then
    DoorDoneWatchdog := 0
  else
    DoorDoneWatchdog := KillThread(WatchdogTid)
End; { DoorDoneWatchdog }

Procedure DoorExitProc;
Begin
  ExitProc := DoorExitProcSave;
  if not LocalMode then DosFreeMem(pchShare);
  DoorDoneWatchdog;
  if MustClosePort then ClosePort(ActivePort);

  if LocalOutput then
  begin
    TextAttr := LightGray;
    WindMin := doorLastWindMin;
    WindMax := doorLastWindMax;
{    ClrScr}
  end
End; { DoorExitProc }
(*
Function DoorTerminateHandler(p1 : PExceptionReportRecord;
                               p2 : PExceptionRegistrationRecord;
                               p3 : PContextRecord;
                               pv : pointer) : ULong; cdecl;
Begin
  if (p1^.ExceptionNum = XCPT_SIGNAL) and
     (p1^.ExceptionInfo[0] = XCPT_SIGNAL_BREAK) then
  begin
    DoorWriteLn(CRLF+LF+AnsiAttr(LightRed)+
      'OS2DOOR: Program terminated by Ctrl-Break or Ctrl-C.');
    DoorTerminateHandler := XCPT_PROCESS_TERMINATE;
  end else DoorTerminateHandler := XCPT_CONTINUE_SEARCH;
  XcptProc := DoorXcptProcSave;
End; { DoorTerminateHandler }
*)


Begin
  DoorExitProcSave := ExitProc;
  ExitProc := @DoorExitProc;
  (*
  DoorXcptProcSave := XcptProc;
  XcptProc := @DoorTerminateHandler;
  *)
  doorScrWidth := 80;  { screen width is 80 by default }
  doorScrHeight := 24; { screen height is 24 by default }
  ActivePort := 0;     { active port is 0 }
  FillChar(DoorData, SizeOf(TDoorData), 0); { zeroise door data }
  DoorData.UserTerm := TERM_ANSI; { user terminal is always ANSI for local mode }
  DoorData.UserName := 'local user';
  LastTime := 0;
  TextAttr := LightGray; { Default text attr }
  doorLastWindMin := WindMin;
  doorLastWindMax := WindMax;
  DoorInitTimer { init door timer }
End.
