Unit OS2Misc;
{ Miscellaneous routines for Virtual Pascal (OS/2) unit }

interface

uses OS2base, Dos;


function Str2Int(const S : string) : longint;
function Punctuate(Number : longint) : string;
function EndingChar(Num : longint) : string;
function StringOf(Ch : char; Count : byte) : string;
function Spaces(Count : byte) : string;
function CountChars(Ch : char; S : string) : byte;
function LoCase(Ch : char) : char;
function ToUpper(const S : string) : string;
function ToLower(const S : string) : string;
function LeadingZero(I : longint; Len : byte) : string;
function LJustify(S : string; Len : byte) : string;
function RJustify(S : string; Len : byte) : string;
function ReplaceChars(const S : string; OldChar, NewChar : char) : string;
function DeleteChars(const S : string; Ch : char) : string;
function LTrim(const S : string) : string;
function RTrim(S : string) : string;
function GetCmdLParam(Switch : char; var CmdLine : string;
  Remove : boolean) : string;
function Pas2PChar(var Str : string) : PChar;
function StrPCat(Dest : PChar; Source : string) : PChar;
function MyDiffTime(Tnow, Tthen : OS2base.DateTime) : longint;

procedure GetDateTime(var dt : DateTime);
function DateStr(dt : DateTime) : string;
function CurDateStr : string;

{ functions to work with filenames }
function AddTrailingSlash(Path : string) : string;
function GetDriveLetter(Path : string) : char;
function GetDirPart(Path : string) : string;
function GetNamePart(const Path : string) : string;
function GetNoExtPart(const Path : string) : string;
function MatchWildcMask(FileName, WildcMask : string) : boolean;
function HasWildcards(const Path : string) : boolean;
function UnixToDosFilename(Path : string) : string;
function TrimPath(Path : string; MaxChars : byte) : string;
function FileExists(const FileName : string) : boolean;

{ time/out routines }

procedure TimerInit(var TimeStart : longint); { initializes timer and associated local variables }
function TimerCalcTime(TimeStart : longint) : longint; { calc passed time since TimerInit (in ms) }
procedure TimeoutInit(Seconds : longint); { sets the timeout interval }
function TimeoutReached : boolean; { returns TRUE if timeout has been reached }

implementation

uses Strings, VPUtils;

var
  lTimeStart, lTimeoutVal : longint;

Function IntToStr(I : longint) : string;
var S : string[10];
Begin
  Str(I, S);
  IntToStr := S
End; { IntToStr }

Function Str2Int(const S : string) : longint;
var
  Code : longint;
  I : longint;
Begin
  Val(S, I, Code);
  if Code <> 0 then
    Str2Int := -1 else Str2Int := I
End; { Str2Int }

Function Punctuate(Number : longint) : string;
var
  RetStr : string[40];
  P : shortint;
Begin
  Str(Number, RetStr);
  P := Succ(Length(RetStr));
  repeat
    Dec(P, 3);
    if P > 1 then Insert(',', RetStr, P)
  until P < 1;
  Punctuate := RetStr
End; { Punctuate }


Function EndingChar(Num : longint) : string;
Begin
  if (Num - 1) mod 10 <> 0 then
    EndingChar := 's'
  else
    EndingChar := ''
End; { EndingChar }

Function LoCase(Ch : char) : char; assembler; {$USES edi,esi} {$FRAME-}
Asm
        mov     al,&Ch
        cmp     al,'A'
        jb      @@1
        cmp     al,'Z'
        ja      @@1
        add     al,20h
@@1:
End; { LoCase }

Function ToUpper(const S : string) : string; assembler; {$USES edi,esi} {$FRAME-}
Asm
        mov     esi,S
        sub     ecx,ecx
        mov     cl,[esi]
        mov     edi,@Result
        mov     [edi],cl
        jcxz    @@3
        inc     esi
        inc     edi
@@1:
        mov     al,[esi]
        cmp     al,'a'
        jb      @@2
        cmp     al,'z'
        ja      @@2
        sub     al,20h
@@2:
        mov     [edi],al
        inc     esi
        inc     edi
        loop    @@1
@@3:
End; { ToUpper }

Function ToLower(const S : string) : string; assembler; {$USES edi,esi} {$FRAME-}
Asm
        mov     esi,S
        sub     ecx,ecx
        mov     cl,[esi]
        mov     edi,@Result
        mov     [edi],cl
        jcxz    @@3
        inc     esi
        inc     edi
@@1:
        mov     al,[esi]
        cmp     al,'A'
        jb      @@2
        cmp     al,'Z'
        ja      @@2
        add     al,20h
@@2:
        mov     [edi],al
        inc     esi
        inc     edi
        loop    @@1
@@3:
End; { ToUpper }


Function StringOf(Ch : char; Count : byte) : string; assembler; {$USES edi} {$FRAME-}
Asm
        mov     edi,@Result
        sub     ecx,ecx
        mov     cl,Count
        mov     [edi],cl
        jcxz    @@2
        inc     edi
        mov     al,&Ch
@@1:
        mov     [edi],al
        inc     edi
        loop    @@1
@@2:
End; { StringOf }

Function Spaces(Count : byte) : string;
Begin
  Spaces := StringOf(' ', Count)
End; { Spaces }

Function CountChars(Ch : char; S : string) : byte; assembler; {$USES esi} {$FRAME-}
Asm
        mov     esi,S
        sub     ecx,ecx
        sub     eax,eax
        mov     cl,[esi]
        jcxz    @@3
@@1:
        inc     esi
        mov     bl,[esi]
        cmp     bl,&Ch
        jne     @@2
        inc     al
@@2:
        loop    @@1
@@3:
End; { CountChars }

Function LeadingZero(I : longint; Len : byte) : string;
Begin
  LeadingZero := ReplaceChars(RJustify(IntToStr(I), Len), ' ', '0')
End; { LeadingZero }

Function LJustify(S : string; Len : byte) : string;
Begin
  while (Len > Length(S)) and (Len < High(S)) do S := S + ' ';
  LJustify := S
End; { LJustify }

Function RJustify(S : string; Len : byte) : string;
Begin
  while (Len > Length(S)) and (Len < High(S)) do S := ' ' + S;
  RJustify := S
End; { RJustify }

Function ReplaceChars(const S : string; OldChar, NewChar : char) : string;
  assembler; {$USES edi,esi} {$FRAME-}
Asm
        mov     esi,S
        sub     ecx,ecx
        mov     cl,[esi]
        mov     edi,@Result
        mov     [edi],cl
        jcxz    @@3
        inc     esi
        inc     edi
@@1:
        mov     al,[esi]
        cmp     al,OldChar
        jne      @@2
        mov     al,NewChar
@@2:
        mov     [edi],al
        inc     esi
        inc     edi
        loop    @@1
@@3:
End; { ReplaceChars }

Function DeleteChars(const S : string; Ch : char) : string;
  assembler; {$USES edi,esi} {$FRAME-}
Asm
        mov     esi,S
        sub     ecx,ecx
        mov     cl,[esi]
        mov     edi,@Result
        mov     [edi],cl
        jcxz    @@3
        inc     esi
        inc     edi
@@1:
        mov     al,[esi]
        cmp     al,&Ch
        je      @@2
        mov     [edi],al
        inc     edi
@@2:
        inc     esi
        loop    @@1
@@3:
End; { DeleteChars }

Function LTrim(const S : string) : string;
var P : byte;
Begin
  P := 1;
  while (S[P] in [' ', #9]) and (P <= Length(S)) do Inc(P);
  LTrim := Copy(S, P, Length(S))
End; { LTrim }

Function RTrim(S : string) : string;
Begin
  if S <> '' then
    while S[Length(S)] in [' ', #9] do Dec(S[0]);
  RTrim := S
End; { RTrim }

Function GetCmdLParam(Switch : char; var CmdLine : string; Remove : boolean) : string;
var
  i, swp : byte;
  su, rs : string;

function GetArg : string;
var s : string;
begin
  Inc(i);
  s := '';
  if CmdLine[i] in ['''','"'] then
  begin
    Inc(i);
    while (not (CmdLine[i] in ['''','"'])) and (i <= Length(CmdLine)) do
    begin
      s := s + CmdLine[i];
      Inc(i)
    end
  end else
  while (not (CmdLine[i] in ['-','/',' '])) and (i <= Length(CmdLine)) do
  begin
    s := s + CmdLine[i];
    Inc(i)
  end;
  GetArg := s
end; { GetArg }

Begin
  Switch := UpCase(Switch);
  su := ToUpper(CmdLine);
  i := 1; rs := '';
  while (i <= Length(su)) do
  begin
    if su[i] in ['-','/'] then
    begin
      swp := i;
      Inc(i);
      if su[i] = Switch then
      begin
        rs := GetArg;
        if Remove then Delete(CmdLine, swp, i - swp);
        GetCmdLParam := rs;
        Exit
      end else GetArg
    end else Inc(I)
  end;
  GetCmdLParam := ''
End; { GetCmdLParam }

Function StrPCat(Dest : PChar; Source : string) : PChar;
Begin
  StrPCat := StrCat(Dest, Pas2PChar(Source))
End; { StrPCat }

Function Pas2PChar(var Str : string) : PChar; assembler; {$USES edi} {$FRAME-}
Asm
        mov     edi,Str
        mov     eax,edi
        inc     eax
        sub     ecx,ecx
        mov     cl,[edi]
        cmp     cl,0FFh
        jne     @@1
        dec     ecx
@@1:
        add     edi,ecx
        sub     bl,bl
        mov     [edi+1],bl
End; { Pas2PChar }

Function MyDiffTime(Tnow, Tthen : OS2base.DateTime) : longint;
{ Purpose:   Calculates the difference in hundredths of a second
             between two specified times. The maximum difference
             permitted is 23hrs 59min 99/100ths.
  Return:    difference in milliseconds. }
var
  Tdiff : longint;
  Tdays, Thours, Tmins, Tsecs, Thunds : word;
Begin
  Tdays := Tnow.weekday - Tthen.weekday;
  Thours := Tnow.hours - Tthen.hours;
  Tmins := Tnow.minutes - Tthen.minutes;
  Tsecs := Tnow.seconds - Tthen.seconds;
  Thunds := Tnow.hundredths - Tthen.hundredths;

  if Thunds < 0 then
  begin
    Thunds := Thunds + 100;
    Dec(Tsecs)
  end;
  if Tsecs < 0 then
  begin
    Tsecs := Tsecs + 60;
    Dec(Tmins)
  end;
  if Tmins < 0 then
  begin
    Tmins := Tmins + 60;
    Dec(Thours)
  end;
  if Thours < 0 then
  begin
    Thours := Thours + 24;
    Dec(Tdays)
  end;
  if Tdays < 0 then
    Tdays := Tdays + 7;

  if Tdays <> 0 then
    Tdiff := -1    (* do not handle, too long a period *)
  else
    Tdiff := Thunds + (100 * Tsecs) + (100 * 60 * Tmins) +
             (100 * 60 * 60 * Thours);

  MyDiffTime := Tdiff * 10                 { return as milliseconds }
End; { MyDiffTime }

Procedure GetDateTime(var dt : DateTime);
var Dummy : longint;
Begin
  with dt do
  begin
    GetDate(Year, Month, Day, Dummy);
    GetTime(Hour, Min, Sec, Dummy)
  end
End; { GetDateTime }

Function DateStr(dt : DateTime) : string;
const
  Month : array[0..12] of string[3] =
    ('???', 'Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun',
     'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec');
Begin
  if not (DT.Month in [Low(Month)..High(Month)]) then
    DT.Month := 0;
  DateStr := Concat(RJustify(IntToStr(DT.Day), 2), ' ',
    Month[DT.Month], ' ', Copy(IntToStr(DT.Year), 3, 2), ' ',
    RJustify(IntToStr(DT.Hour), 2), ':', LeadingZero(DT.Min, 2),
    ':', LeadingZero(DT.Sec, 2));
End; { DateStr }

Function CurDateStr : string;
var CurrentDate : DateTime;
Begin
  GetDateTime(CurrentDate);
  CurDateStr := DateStr(CurrentDate)
End; { CurDateStr }

Function AddTrailingSlash(Path : string) : string;
Begin
  if Path[Length(Path)] <> '\' then
    AddTrailingSlash := Path + '\'
  else
    AddTrailingSlash := Path
End; { AddTrailingSlash }

Function GetDriveLetter(Path : string) : char;
var Drive : char;
Begin
  Drive := UpCase(Path[1]);
  if (Path[1] in ['A'..'Z']) and (Path[2] = ':') then
    GetDriveLetter := Drive
  else
  begin
    GetDir(0, Path);
    GetDriveLetter := UpCase(Path[1])
  end
End; { GetDriveLetter }

Function GetDirPart(Path : string) : string;
Begin
  if Pos('\', Path) > 0 then
    while Path[Length(Path)] <> '\' do Dec(Path[0]);
  GetDirPart := Path
End; { GetDirPart }

Function GetNamePart(const Path : string) : string;
var I : byte;
Begin
  I := Length(Path);
  while (I > 0) and not (Path[I] in ['/', '\', ':']) do Dec(I);
  GetNamePart := Copy(Path, I + 1, Length(Path))
End; { GetNamePart }

Function GetNoExtPart(const Path : string) : string;
var I : byte;
Begin
  I := Length(Path);
  while (I > 0) and (Path[I] <> '.') do Dec(I);
  if I = 0 then
    GetNoExtPart := Path
  else
    GetNoExtPart := Copy(Path, 1, I - 1)
End; { GetNoExtPart }

Function MatchWildcMask(FileName, WildcMask : string) : boolean;
var
  rc : boolean;
  p1, p2 : byte;
Begin
  rc := False;
  WildcMask := ToUpper(WildcMask);
  FileName := ToUpper(FileName);
  if WildcMask = '*' then
    rc := Pos('.', FileName) = 0
  else
  if WildcMask = '*.*' then
    rc := True;
  MatchWildcMask := rc
End; { MatchWildcMask }

Function HasWildcards(const Path : string) : boolean;
Begin
  HasWildcards := (Pos('*', Path) > 0) or (Pos('?', Path) > 0)
End; { HasWildcards }

Function UnixToDosFilename(Path : string) : string;
var
  P, N : byte;
  DosFilename : string[12];
Begin
  if Path = '' then
  begin
    UnixToDosFilename := '';
    Exit
  end;
  Path := GetNamePart(Path);
  repeat
    P := Pos('.', Path);
    if (P = 1) or (P = Length(Path)) then Delete(Path, P, 1)
  until (P <> 1) and (P <> Length(Path));
  N := Length(Path) - P;
  if P = 0 then
  begin
    if N > 8 then
      DosFileName := Concat(Copy(Path, 1, 7), '~')
    else
      DosFileName := Path;
    N := 0
  end
  else
  if P < 10 then
    DosFileName := Copy(Path, 1, P)
  else
    DosFileName := Concat(Copy(Path, 1, 7), '~.');
  if N > 0 then
    if N > 3 then
    begin
      while CountChars('.', DosFileName) > 1 do
        Delete(DosFileName, Pos('.', DosFileName), 1);
      DosFileName := Concat(DosFileName, Copy(Path, P + 1, 2), '~')
    end else DosFileName := Concat(DosFileName, Copy(Path, P + 1, N));
  UnixToDosFilename := DosFileName
End; { UnixToDosFilename }

Function TrimPath(Path : string; MaxChars : byte) : string;
const Separator : string = '...';
var
  ln, L, R, P : byte;
  rs : string;
Begin
  ln := Length(Path);
  if ln > MaxChars then
  begin
    P := ln div 2;
    L := P; R := P;
    repeat
      while (L > 1) and (not (Path[L] in ['/', '\'])) do Dec(L);
      while (R < ln) and (not (Path[R] in ['/', '\'])) do Inc(R);
      rs := Concat(Copy(Path, 1, L), Separator, Copy(Path, R, ln));
      Dec(L); Inc(R)
    until (Length(rs) <= MaxChars) or (L = 1) or (R = ln);
    TrimPath := rs
  end else TrimPath := Path
End; { TrimPath }

Function FileExists(const FileName : string) : boolean;
var
  F : SearchRec;
  rc : integer;
Begin
  FindFirst(FileName, Archive or ReadOnly or SysFile, F);
  rc := DosError;
  FindClose(F);
  FileExists := rc = 0
End; { FileExists }

Procedure TimerInit(var TimeStart : longint);
Begin
  TimeStart := GetTimemSec div 1000 { use it in seconds }
End; { TimerInit }

Function TimerCalcTime(TimeStart : longint) : longint;
{ Warning: this function can only work with time difference not exceeding
           24 hours. }
var
  lTimeNow, lTimeElapsed : longint;
Begin
  lTimeNow := GetTimemSec div 1000; { use it in seconds }
  while lTimeNow < TimeStart do
  begin
    Inc(TimeStart, 12 * 3600);
    Inc(lTimeNow, 12 * 3600)
  end;
  lTimeElapsed := lTimeNow - TimeStart;
  TimerCalcTime := lTimeElapsed
End; { TimerCalcTime }

Procedure TimeoutInit(Seconds : longint);
Begin
  TimerInit(lTimeStart);
  lTimeoutVal := Seconds
End; { TimeoutInit}

Function TimeoutReached : boolean;
Begin
  if lTimeoutVal > 0 then
    TimeoutReached := TimerCalcTime(lTimeStart) > lTimeoutVal
  else
    TimeoutReached := False
End; { TimeoutReached }


End.
