;----------------------------------------------------------------------------
; KEYBOARD.ASM
;----------------------------------------------------------------------------
;   Low level input/output support routines for event manager
;
;   Andrew Cheung (July 1996)
;   Keyboard routines for keyboard handler


;   Basic keyboard stuff
;
;   keyboard interrupt is interrupt 09H
;   When the keyboard handler receives an interrupt from the 8042
;   it stores it in an array called the keyboard buffer. Then when
;   an application program calls the INT 16H functions in the ROM
;   BIOS, the BIOS removes the scan code from the buffer and returns
;   it to the program, along with the ASCII code symbol that represents
;   that key e.g.
;   'a' has a scan code of 30 and an ASCII code of 97
;----------------------------------------------------------------------------

		.MODEL large
		.CODE
		PUBLIC _initkey,_remkey

		IntKey_Seg   dw 0        ;16 bit segment of old keyboard vector
		IntKey_Off   dw 0        ;16 bit offset of old keyboard vector
		KeyBuf_Seg   dw 0        ;16 bit segment of keyboard buffer
		KeyBuf_Off   dw 0        ;16 bit offset of keyboard buffer



;--------------------------------------------------------------------------
; int initkey(char far *buffer);
;--------------------------------------------------------------------------
;  	install the keyboard handler
;--------------------------------------------------------------------------

_initkey  PROC
; Initialize keyboard interrupt handler
		ARG		buffer:DWORD    ; Pointer to scan code buffer
		push		bp
		mov   	bp,sp
		push		ds
		mov		ax,3509h        ; Call INT 21h Function 35h
		int		21h             ; to get vector for INT 09h (keyboard)
		mov   	IntKey_Seg,es   ; Save its 16 bit segment
		mov      IntKey_Off,bx 	 ; Save its 16 bit offset
		mov		dx,seg _newkey  ; Set ds:dx to the new Key INT routine
		mov		ds,dx
		mov		dx,offset _newkey
		mov		ax,2509h        ; Call INT 21h Function 25h
		int		21h				 ; to install new keyboard handler
		les		dx,buffer       ; Point ES:DX to keyboard buffer
		mov		KeyBuf_Seg,es	 ; Save buffer into KeyBuf_Seg
		mov   	KeyBuf_Off,dx	 ; and KeyBuf_Off
		pop		ds
		pop		bp
		ret
_initkey  ENDP


;--------------------------------------------------------------------------
; void remkey(void);
;--------------------------------------------------------------------------
;  	remove the keyboard handler
;--------------------------------------------------------------------------

_remkey PROC
		push		ds
		mov   	dx,IntKey_Seg	 ; Get segment and offset of
		mov  		ds,dx           ; old interrupt handler
		mov   	dx,IntKey_Off
		mov		ax,2509h        ; Call INT 21h Function 25h
		int		21h             ; to install old keyboard handler
		pop		ds
		ret
_remkey ENDP


;--------------------------------------------------------------------------
;  void newkey(void);
;--------------------------------------------------------------------------
;  The keyboard handler
;	Reads the scan code returned from the 8042 and sets the
;	keyboard buffer (KeyBuf_Seg:KeyBuf_Off which is global)
;  Make codes have values between 0-127
;  Break codes have values between 128-255
;--------------------------------------------------------------------------

_newkey  PROC
		push		ax     	 		 ; Save all registers used
		push		di
		push		bp
		push		es

		mov		ax,KeyBuf_Seg   ; Point es:di at scan code buffer
		mov		es,ax
		mov		di,KeyBuf_Off
		in			al,60h          ; Get scan code from keyboard port
		mov		ah,0            ; Zero high byte of ax
		cmp		al,128          ; Was code a "make" or a "break"?
		jnb		break           ; If break, skip ahead
		mov		bp,ax           ; Else use code as index into buffer
		mov		byte ptr [es:di+bp],1  ; Mark key as pressed
		jmp		endkey
	break:                  	 ; break code so remove high bit
		and		al,127          ; by subtracting 128 from scan code
		mov		bp,ax           ; Use code as index into buffer
		mov		byte ptr [es:di+bp],0  ; Mark key as released
	endkey:
;		in			al,61h          ; Clear the keyboard interrupt
;		mov		ah,al
;		or			al,80h
;		out		61h,al
		mov		al,20h          ; Allow additional interrupts
		out		20h,al
		pop		es              ; Restore registers
		pop		bp
		pop		di
		pop		ax
		iret
_newkey  ENDP

END
