/****************************************************************************
*
*                               S H A N D Y
*
*						  			  1996 Andrew Cheung
*
* 	FILENAME	:     $RCSfile: object.cpp $
* 	VERSION	:    	$Revision: 1.0 $
* 	DATE     :     $Date: 1996/07/28 17:03:00 $
* 	LANGUAGE	:     C++
*
* 	ENVIRONMENT:
*	IBM PC (MSDOS) Real Mode and 32 bit Protected Mode.
*	Watcom C/C++ 10.0,Borland C++ 3.1 - flat or large memory model
*
* 	DESCRIPTION:
*	Object Class and member functions for the SHANDY program
*
*
* 	$Id: object.cpp 1.0 1996/07/28 17:03:00 apwc Release apwc $
*
****************************************************************************/
#include <stdlib.h>
#include <string.h>
#include "shandy.hpp"
#include "globals.hpp"

extern int SearchMatList(MatStruct **texture_list,char *material_name);
extern int SearchMatFile(FILE *texfile,MatStruct **texture_list,char *material_name);

//Used for qsorting planes
int poly_sort_function( Polygon **a, Polygon **b)
{
	return( (*b)->AvgZ() - (*a)->AvgZ() );
}

/****************************************************************************
*
*                             OBJECT CLASS
*
****************************************************************************/

Object::Object()
{
}

Object::Object(float x, float y, float z)
{
	origin = new Point(x, y, z);
	midpoint = new Point;

	numPoints = 0;
	numNormals = 0;
	numPolys = 0;
	StartShade = GOURAUD;
	shading = GOURAUD;
	point = new Point* [1];
	normal = new Point* [1];
	poly = new Polygon* [1];

	matrix = new float [6];
}

Object::~Object()
{
	for (count = 0; count < numPolys; count++)
		delete poly[count];

	for (count = 0; count < numNormals; count++)
		delete normal[count];

	for (count = 0; count < numPoints; count++)
		delete point[count];

	delete []matrix;
	delete poly;
	delete point;
	delete origin;
	delete midpoint;
}

void Object::SetShading(int s)
{
	shading = s;
	for (count = 0; count < numPolys; count++)
		poly[count]->SetShading(s);
}

// add a point whose coordinates are given as local to the origin of
// this object to this objects list of points
void Object::AddLocalPoint(float x, float y, float z)
{
	Point **temp;

	temp = new Point* [numPoints + 1];
	memcpy(temp, point, numPoints * sizeof(Point*));

	delete point;
	point = temp;

	point[numPoints] = new Point(x, y, z);
	point[numPoints]->TranslateLocalOrigin(origin);
	numPoints++;
}

void Object::AddLocalPoint(Point *p)
{
	Point **temp;

	temp = new Point* [numPoints + 1];
	memcpy(temp, point, numPoints * sizeof(Point*));

	delete point;
	point = temp;

	p->TranslateLocalOrigin(origin);
	point[numPoints] = p;
	numPoints++;
}

void Object::AddNormal(float x, float y, float z)
{
	Point **temp;

	temp = new Point* [numNormals + 1];
	memcpy(temp, normal, numNormals * sizeof(Point*));

	delete normal;
	normal = temp;

	normal[numNormals] = new Point(x, y, z);
	normal[numNormals]->TranslateLocalOrigin(origin);
	numNormals++;
}

void Object::AddNormal(Point *p)
{
	Point **temp;

	temp = new Point* [numNormals + 1];
	memcpy(temp, normal, numNormals * sizeof(Point*));

	delete normal;
	normal = temp;

	p->TranslateLocalOrigin(origin);
	normal[numNormals] = p;
	numNormals++;
}

// Add a point whose coordinates are given in terms of THE (0,0,0)
// origin to this object's list of points x,y,z are GLOBAL coords
void Object::AddGlobalPoint(float x, float y, float z)
{
	AddLocalPoint(x - origin->x3d, y - origin->y3d, z - origin->z3d);
}

void Object::AddGlobalPoint(Point *p)
{
	p->SetNewLocalOrigin(origin);
	AddLocalPoint(p);
}

//	Add a local polygon (whose vertices are local points)
void Object::AddLocalPoly(Polygon *pg)
{
	 Polygon **temp;

	 temp = new Polygon*[numPolys + 1];
	 memcpy(temp, poly, numPolys * sizeof(Polygon*));

	 delete poly;
	 poly = temp;

	 poly[numPolys] = pg;
	 numPolys++;
	 AddNormal(pg->normal);
}

void Object::AddLocalPoly(int p1, int p2, int p3, int c)
{
	 Polygon *pg = new Polygon(point[p1], point[p2], point[p3], c);
	 AddLocalPoly(pg);
}

// Add a poly whose vertices are given in global coordinates
void Object::AddGlobalPoly(Polygon *pg)
{
	 pg->SetNewLocalOrigin(origin);
	 AddLocalPoly(pg);
}

// rotate this object about it's origin
void Object::RotateLocal(int XDeg, int YDeg, int ZDeg)
{
	XDeg %= 360;
	YDeg %= 360;
	ZDeg %= 360;

	matrix[SINX] = SinFP(XDeg);
	matrix[COSX] = CosFP(XDeg);
	matrix[SINY] = SinFP(YDeg);
	matrix[COSY] = CosFP(YDeg);
	matrix[SINZ] = SinFP(ZDeg);
	matrix[COSZ] = CosFP(ZDeg);

	midpoint->RotateLocal(matrix);

	for (count = 0; count < numPoints; count++)
		point[count]->RotateLocal(matrix);

	for (count = 0; count < numNormals; count++)
		normal[count]->RotateLocal(matrix);
}

// Rotate this object about THE (0,0,0) origin
void Object::RotateGlobal(float *matrix)
{
	origin->RotateGlobal(matrix);
	midpoint->RotateGlobal(matrix);
	for (count  = 0; count < numPoints; count++)
		point[count]->RotateGlobal(matrix);

	for (count = 0; count < numNormals; count++)
		normal[count]->RotateGlobal(matrix);
}

void Object::RotateLocalX(int XDeg)
{
	XDeg %= 360;

	matrix[SINX] = SinFP(XDeg);
	matrix[COSX] = CosFP(XDeg);

	midpoint->RotateLocalX(matrix);

	for (count = 0; count < numPoints; count++)
		point[count]->RotateLocalX(matrix);

	for (count = 0; count < numNormals; count++)
		normal[count]->RotateLocalX(matrix);
}

void Object::RotateLocalY(int YDeg)
{
	YDeg %= 360;

	matrix[SINY] = SinFP(YDeg);
	matrix[COSY] = CosFP(YDeg);

	midpoint->RotateLocalY(matrix);

	for (count = 0; count < numPoints; count++)
		point[count]->RotateLocalY(matrix);

	for (count = 0; count < numNormals; count++)
		normal[count]->RotateLocalY(matrix);
}

void Object::RotateLocalZ(int ZDeg)
{
	ZDeg %= 360;

	matrix[SINZ] = SinFP(ZDeg);
	matrix[COSZ] = CosFP(ZDeg);

	midpoint->RotateLocalZ(matrix);

	for (count = 0; count < numPoints; count++)
		point[count]->RotateLocalZ(matrix);

	for (count = 0; count < numNormals; count++)
		normal[count]->RotateLocalZ(matrix);
}


// translate this object
void Object::TranslateLocal(float XDist, float YDist, float ZDist)
{
	origin->TranslateLocal(XDist, YDist, ZDist);
	midpoint->TranslateLocal(XDist, YDist, ZDist);

	for (count = 0; count < numPoints; count++)
	{
		point[count]->TranslateLocal(XDist, YDist, ZDist);
		point[count]->SetNewLocalOrigin(origin);
	}
}

// depth sort this object's planes (selection sort - O(n^2/2) I think)
void Object::SortPlanes()
{
	Polygon *temp;
	register int pos, index;

	for (pos = 0; pos < (numPolys - 1); pos++)
	{
		index = pos;
		for (count = index + 1; count < numPolys; count++)
			if (poly[count]->AvgZ() > poly[index]->AvgZ())
				index = count;

		if (index != pos)
		{
			temp = poly[pos];
			poly[pos] = poly[index];
			poly[index] = temp;
		}
	}
}

//***********************  DISPLAY OPTIONS  *********************************

// Display as dots
void Object::Dots(char color)
{
	for (count = 0; count < numPoints; count++)
		point[count]->Display(color);
}

// display as a wireframe
void Object::WireFrame()
{
	for (count = 0; count < numPolys; count++)
		poly[count]->WireFrame();
}

// Display Lambert (flat) shaded
void Object::FlatShade()
{
	//SortPlanes();
	for (count = 0; count < numPolys; count++)
		poly[count]->FlatShade();
}

//Gouraud shade this object
void Object::GouraudShade()
{
	//SortPlanes();
	for (count = 0; count < numPolys; count++)
		poly[count]->GouraudShade();
}


//Universal display function
void Object::Display()
{
#ifndef PROFILE
	qsort(poly, numPolys,sizeof(Polygon *),(int(*)(const void *,const void *))poly_sort_function);
	//SortPlanes();
#endif
	//Dots doesn't need plane sorting, so call it directly - wireframe doesn't
	//need sorting either but looks messy (all polys are triangles) without it

	if(shading != DOTS)
		for (count = 0; count < numPolys; count++)
			poly[count]->Display();
	else
		Dots(255);
}

// set the location of this object in global coordinates
void Object::SetLocation(float x, float y, float z)
{
	origin->SetCoords(x, y, z);
	for (count = 0; count < numPoints; count++)
		point[count]->SetNewLocalOrigin(origin);
}

// Have this object set up its normal vectors needed for Gouraud shading
void Object::SetGNormals()
{
	for (count = 0; count < numPolys; count++)
		poly[count]->CalcGNormals();

	for (count = 0; count < numPoints; count++)
		point[count]->AvgNormal();

	for (count = 0; count < numPoints; count++)
		if (point[count]->numNormals)
		{
			AddNormal(point[count]->normalX, point[count]->normalY, point[count]->normalZ);
			point[count]->normal = (void *) normal[numNormals - 1];
		}
}

void Object::AddMidPoint()
{
	float tmpX=0.0, tmpY=0.0, tmpZ=0.0;

	for(count=0; count < numPoints; count++)
	{
		tmpX += point[count]->x3d;
		tmpY += point[count]->y3d;
		tmpZ += point[count]->z3d;
	}
	tmpX = tmpX/numPoints;
	tmpY = tmpY/numPoints;
	tmpZ = tmpZ/numPoints;

	midpoint->SetCoords(tmpX,tmpY,tmpZ);
}

//Transform 3d world coords to 3d eye (camera) coords
void Object::EyeSpace()
{
	midpoint->EyeSpace();
	for (count = 0; count < numPoints; count++)
		point[count]->EyeSpace();
}

void Object::LoadObject(FILE *fin, FILE *texture_file, int *ShadeModel)
{
	int vertices, faces, count, vertexnum, tempA, tempB, tempC;
	char tempChar, cptr[1]="";
	float tempX, tempY, tempZ, tempU=0.0, tempV=0.0;
	char *tempString = new char [80];

	while(strncmp(tempString, "Vertices",8))
	{
		fscanf(fin, "%s", tempString);
		if (feof(fin))
		{
			printf("String \"Vertices\" not found in file.Exitting...\n");
			exit(1);
		}
	}

	tempChar = fgetc(fin);				//Get ':'
	fscanf(fin, "%d", &vertices);
	printf("Object has %d vertices\n", vertices);

	while(strncmp(tempString, "Faces", 5))
	{
		fscanf(fin, "%s", tempString);
		if (feof(fin))
		{
			printf("String \"Faces\" not found in file.Exitting...\n");
			exit(1);
		}
	}

	tempChar = fgetc(fin);				//Get ':'
	fscanf(fin, "%d", &faces);
	printf("Object has %d faces\n", faces);

	while(strncmp(tempString, "Vertex",6))
	{
		if((strncmp(tempString,"Mapped", 6)==0) && (*ShadeModel==TEXTURE_FLAT) )
			StartShade=TEXTURE_FLAT;

		fscanf(fin, "%s", tempString);
		if (feof(fin))
		{
			printf("String \"Vertex\" not found in file.Exitting...\n");
			exit(1);
		}
	}

	if(StartShade !=TEXTURE_FLAT)
	{
		if(*ShadeModel >= 1 && *ShadeModel <= 8) //DOTS...PHONG
			StartShade = *ShadeModel;
		else
			{StartShade=GOURAUD; *ShadeModel=GOURAUD;}
	}

	while(strncmp(tempString,"list:",5))
	{
		fscanf(fin, "%s", tempString);
		if (feof(fin))
		{
			printf("String \"list\" not found in file.Exitting...\n");
			exit(1);
		}
	}
#ifdef _DEBUG
	printf("\nVertex data:\n");
#endif

	//********************  LOAD VERTEX DATA  ********************************

	for (count = 0; count < vertices; count++)
	{
		while(strncmp(tempString,"Vertex",6))
		{
			fscanf(fin, "%s", tempString);
			if(feof(fin))
			{
				printf("String \"Vertex\" not found in file.Exitting...\n");
				exit(1);
			}
		}

		fscanf(fin, "%d", &vertexnum);
		fscanf(fin, "%s", tempString);      //get ':   '

		fscanf(fin, "%[^:]", tempString);   //get '     X'
		fgetc(fin);                      	//get ':'
		fscanf(fin, "%f", &tempX);

		fscanf(fin, "%[^:]", tempString);
		fgetc(fin);
		fscanf(fin, "%f", &tempY);

		fscanf(fin, "%[^:]", tempString);
		fgetc(fin);
		fscanf(fin, "%f", &tempZ);

		if(StartShade == TEXTURE_FLAT)
		{
			fscanf(fin, "%[^:]", tempString);
			fgetc(fin);
			fscanf(fin, "%f", &tempU);

			fscanf(fin, "%[^:]", tempString);
			fgetc(fin);
			fscanf(fin, "%f", &tempV);
		}
#ifdef _DEBUG
		printf("Vertex %d:  X:%4.2f Y:%4.2f Z:%4.2f U:%4.2f  V:%4.2f\n",
				  vertexnum, tempX, tempY, tempZ, tempU, tempV);
#endif
		AddLocalPoint(tempX, tempZ, tempY);  //swap tempY & tempZ because of 3DStudio
		point[numPoints-1]->SetTexCoords(tempU,tempV);
	}

	while(strncmp(tempString,"Face",4))
	{
		fscanf(fin, "%s", tempString);
		if (feof(fin))
		{
			printf("String \"Face\" not found in file.Exitting...\n");
			exit(1);
		}
	}

	while(strncmp(tempString,"list",4))
	{
		fscanf(fin, "%s", tempString);
		if (feof(fin))
		{
			printf("String \"list\" not found in file.Exitting...\n");
			exit(1);
		}
	}

#ifdef _DEBUG
	printf("\nFace data:\n");
#endif

	//**********************  LOAD FACE DATA  ********************************

	for (count = 0; count < faces; count++)
	{
		while(strncmp(tempString,"Face",4))
		{
			fscanf(fin, "%s", tempString);
			if (feof(fin))
			{
				printf("String \"Face\" not found in file.Exitting...\n");
				exit(1);
			}
		}

		fscanf(fin, "%d", &vertexnum);
		fscanf(fin, "%s", tempString);

		while (fgetc(fin) != 'A');
		fgetc(fin);                  // get the ':' character
		fscanf(fin, "%d", &tempA);   // get value for vertex A

		while (fgetc(fin) != 'B');
		fgetc(fin);
		fscanf(fin, "%d", &tempB);

		while (fgetc(fin) != 'C');
		fgetc(fin);
		fscanf(fin, "%d", &tempC);
#ifdef _DEBUG
		printf("Face %d:  %8d %8d %8d\n", vertexnum, tempA, tempB, tempC);
#endif
		AddLocalPoly(tempA, tempB, tempC, 1);//1 is the starting colour : CHANGES numPolys

		//**********************  LOAD TEXMAP DATA  ***************************
		//now load textures if texture mapping was specified AND there are texture coords
		if(StartShade == TEXTURE_FLAT)
		{
			//Get rest of AB: BC: CA: line
			fgets(tempString,80,fin);
			while((tempChar=fgetc(fin)) !='"')
			{
				if (feof(fin))
				{
					printf("Starting \" not found for material in file.Exitting...\n");
					exit(1);
				}
			}

			char texture_name[80]="";
			while((tempChar=fgetc(fin)) !='"')
			{
				if (feof(fin))
				{
					printf("Ending \" not found for material in file.Exitting...\n");
					exit(1);
				}
				cptr[0]=tempChar;
				strncat(texture_name,cptr,1);
			}
#ifdef _DEBUG
			printf("Material: %s\n", texture_name);
#endif
			int result;
			result = SearchMatList(texture_list,texture_name);//texture_list is global at the moment
			if(result >= 0)
				poly[numPolys-1]->texture = texture_list[result]->TFile;
			else
			{
				result = SearchMatFile(texture_file,texture_list,texture_name);
				if(result >= 0)
					poly[numPolys-1]->texture = texture_list[result]->TFile;
				else
				{
					printf("Error - couldn't find %s in textures.dat. Using GOURAUD\n",texture_name);
					StartShade=GOURAUD;
					*ShadeModel=GOURAUD;
				}
			}
		}	//if(StartShade == TEXTURE_FLAT)
	}
	SetShading(StartShade);		// Set the global shading for this object
	SetGNormals();					// Set up Gouraud normals
	AddMidPoint();					// Calculate the midpoint of the object
}
