/****************************************************************************
*
*                               S H A N D Y
*
*									  1996 Andrew Cheung
*
*
* 	FILENAME	:     $RCSfile: point.cpp $
* 	VERSION	:    	$Revision: 1.0 $
* 	DATE     :     $Date: 1996/07/28 17:03:00 $
* 	LANGUAGE	:     C++
*
* 	ENVIRONMENT:
*	IBM PC (MSDOS) Real Mode and 32 bit Protected Mode.
*	Watcom C/C++ 10.0,Borland C++ 3.1 - flat or large memory model
*
* 	DESCRIPTION:
*	Points Class and member functions for the SHANDY program
*
* 	$Id: point.cpp 1.0 1996/07/28 17:03:00 apwc Release apwc $
*
****************************************************************************/
#include "shandy.hpp"
#include "globals.hpp"
#include "defines.h"

#ifdef SVGAKIT
	#include "svga.h"
#endif

/****************************************************************************
*
*                          3D POINT CLASS
*
****************************************************************************/

Point::Point()
{
}

Point::Point(float x, float y, float z)
{
	SetCoords(x,y,z);
}

Point::~Point()
{
	delete normal;
}

//Default local origin of point at (0.0,0.0,0.0) in Global (World) coords
//Global coords of point = Local coords of point
void Point::SetCoords(float x, float y, float z)
{
	x3d = x;
	y3d = y;
	z3d = z;

	localX = x;
	localY = y;
	localZ = z;

	localOX = 0.0;
	localOY = 0.0;
	localOZ = 0.0;

	ZeroNormal();
}

//Set the texture mapping (u,v) coords of the point
void Point::SetTexCoords(float tempU,float tempV)
{
	tempU = fabs(tempU);
	tempV = fabs(tempV);

	//Wrap coords round if u,v values > 1.0
	if(tempU > 1.0)
		tempU = (float)(tempU - (int)tempU);
	if(tempV > 1.0)
		tempV = (float)(tempV - (int)tempV);

	u = FLOAT_TO_FIXED(tempU);
	v = FLOAT_TO_FIXED(tempV);
}

//Transform a point from World coords to Eye coords
void Point::EyeSpace()
{
	float tmpX,tmpY,tmpZ;

	tmpX = x3d - camera->position->x3d;
	tmpY = y3d - camera->position->y3d;
	tmpZ = z3d - camera->position->z3d;

	eyeX = (tmpX*camera->Ux) + (tmpY*camera->Uy) + (tmpZ*camera->Uz);
	eyeY = (tmpX*camera->Vx) + (tmpY*camera->Vy) + (tmpZ*camera->Vz);
	eyeZ = (tmpX*camera->Nx) + (tmpY*camera->Ny) + (tmpZ*camera->Nz);

	Project();
}

//Transform a normal from World coords to Eye coords
void Point::NEyeSpace()
{
	eyeX = (x3d*camera->Ux) + (y3d*camera->Uy) + (z3d*camera->Uz);
	eyeY = (x3d*camera->Vx) + (y3d*camera->Vy) + (z3d*camera->Vz);
	eyeZ = (x3d*camera->Nx) + (y3d*camera->Ny) + (z3d*camera->Nz);
}

//Project the point from Eye space into Screen space
void Point::Project()
{
	if(eyeZ > 0.0)
	{
		x2d = (int)((eyeX / eyeZ)*256.0) + XPROJ;
		y2d = YPROJ - (int)((eyeY / eyeZ)*256.0);
	}
}

//Display the Point as a dot
void Point::Display(char color)
{
	if(y2d>=0  && y2d <=MAXY && x2d >=0 && x2d <=MAXX)
	{
		if(video_mode == 0x13)
			Screen[ScreenY[y2d]+x2d] = color;
#ifdef SVGAKIT
		else
			putPixel(x2d,y2d,color);
#endif
	}
}

//Set a new local origin for the point to a point p
//The local coordinates for the point changes, but the global
//coords remain the same i.e. translate the local origin ONLY
void Point::SetNewLocalOrigin(Point *p)
{
	localOX = p->x3d;
	localOY = p->y3d;
	localOZ = p->z3d;

	localX = x3d - localOX;
	localY = y3d - localOY;
	localZ = z3d - localOZ;
}

//The local coordinates of the point remain unchanged
//but the global coords change i.e. translate the point (globally) AND
//the local origin
void Point::TranslateLocalOrigin(Point *p)
{
	localOX = p->x3d;
	localOY = p->y3d;
	localOZ = p->z3d;

	FormGlobalCoords();
}

void Point::RotateLocal(float *matrix)
{
	float i,j,k;

	// Z axis rotation first
	i = (localX * matrix[COSZ] - localY * matrix[SINZ]);
	j = (localX * matrix[SINZ] + localY * matrix[COSZ]);
	k = localZ;

	// now X axis rotation
	localY = (j * matrix[COSX] - k * matrix[SINX]);
	localZ = (j * matrix[SINX] + k * matrix[COSX]);
	k = localZ;

	// now Y axis
	localX = (k * matrix[SINY] + i * matrix[COSY]);
	localZ = (k * matrix[COSY] - i * matrix[SINY]);

	FormGlobalCoords();
}

void Point::RotateLocal(int xDeg, int yDeg, int zDeg)
{
	float cosX, sinX, cosY, sinY, cosZ, sinZ;
	float i,j,k;

	xDeg %= 360;
	yDeg %= 360;
	zDeg %= 360;

	cosX = CosFP(xDeg);
	sinX = SinFP(xDeg);
	cosY = CosFP(yDeg);
	sinY = SinFP(yDeg);
	cosZ = CosFP(zDeg);
	sinZ = SinFP(zDeg);

	 // Z axis rotation first
	i =  (localX * cosZ - localY * sinZ);
	j =  (localX * sinZ + localY * cosZ);
	k =  localZ;

	// now X axis rotation
	localY = (j * cosX - k * sinX);
	localZ = (j * sinX + k * cosX);
	k = localZ;

	// now Y axis
	localX = (k * sinY + i * cosY);
	localZ = (k * cosY - i * sinY);

	FormGlobalCoords();
}

void Point::RotateLocalX(float *matrix)
{
	float j;

	// X axis rotation
	j = (localY * matrix[COSX] - localZ * matrix[SINX]);
	localZ = (localY * matrix[SINX] + localZ * matrix[COSX]);
	localY = j;

	FormGlobalCoords();
}

void Point::RotateLocalY(float *matrix)
{
	float i;
	// Y axis rotation
	i = (localX * matrix[COSY] + localZ * matrix[SINY]);
	localZ = (localZ * matrix[COSY] - localX * matrix[SINY]);
	localX = i;

	FormGlobalCoords();
}

void Point::RotateLocalZ(float *matrix)
{
	float i;

	// Z axis rotation
	i = (localX * matrix[COSZ] - localY * matrix[SINZ]);
	localY = (localX * matrix[SINZ] + localY * matrix[COSZ]);
	localX = i;

	FormGlobalCoords();
}

void Point::RotateLocalX(int XDeg)
{
	float j,matrix_SINX,matrix_COSX;

	XDeg %= 360;

	matrix_SINX = SinFP(XDeg);
	matrix_COSX = CosFP(XDeg);

	// X axis rotation
	j = (localY * matrix_COSX - localZ * matrix_SINX );
	localZ = (localY * matrix_SINX + localZ * matrix_COSX );
	localY = j;

	FormGlobalCoords();
}

void Point::RotateLocalY(int YDeg)
{
	float i,matrix_SINY,matrix_COSY;

	YDeg %= 360;

	matrix_SINY = SinFP(YDeg);
	matrix_COSY = CosFP(YDeg);

	// Y axis rotation
	i = (localX * matrix_COSY + localZ * matrix_SINY );
	localZ = (localZ * matrix_COSY - localX * matrix_SINY );
	localX = i;

	FormGlobalCoords();
}

void Point::RotateLocalZ(int ZDeg)
{
	float i,matrix_SINZ,matrix_COSZ;

	ZDeg %= 360;

	matrix_SINZ = SinFP(ZDeg);
	matrix_COSZ = CosFP(ZDeg);

	// Z axis rotation
	i = (localX * matrix_COSZ - localY * matrix_SINZ );
	localY = (localX * matrix_SINZ + localY * matrix_COSZ );
	localX = i;

	FormGlobalCoords();
}

void Point::RotateGlobal(float *matrix)
{
	float i,j,k;

	// Z axis rotation first
	i = (x3d * matrix[COSZ] - y3d * matrix[SINZ]);
	j = (x3d * matrix[SINZ] + y3d * matrix[COSZ]);
	k = z3d;

	// now X axis rotation
	y3d = (j * matrix[COSX] - k * matrix[SINX]);
	z3d = (j * matrix[SINX] + k * matrix[COSX]);
	k = z3d;

	// now Y axis
	x3d = (k * matrix[SINY] + i * matrix[COSY]);
	z3d = (k * matrix[COSY] - i * matrix[SINY]);
}

//Translate a point by x,y,z units
void Point::TranslateLocal(float x, float y, float z)
{
	localX += x;
	localY += y;
	localZ += z;

	FormGlobalCoords();
}

void Point::TranslateLocalX(float xDist)
{
	localX += xDist;
	FormGlobalCoords();
}

void Point::TranslateLocalY(float yDist)
{
	localY += yDist;
	FormGlobalCoords();
}

void Point::TranslateLocalZ(float zDist)
{
	localZ += zDist;
	FormGlobalCoords();
}

//Form global (world) coordinates from local coords + local origin
void Point::FormGlobalCoords()
{
	x3d = localX + localOX;
	y3d = localY + localOY;
	z3d = localZ + localOZ;
}

void Point::Normalise()
//normalise a point (normally a normal)
{
	float magnitude;

	magnitude = sqrt((x3d * x3d) + (y3d * y3d) + (z3d * z3d));
	x3d /= magnitude;
	y3d /= magnitude;
	z3d /= magnitude;

	magnitude = sqrt( (localX * localX) + (localY * localY) + (localZ * localZ));
	localX /= magnitude;
	localY /= magnitude;
	localZ /= magnitude;
}

//Used for Gouraud shading
void Point::ZeroNormal()
{
	normalX = 0.0;
	normalY = 0.0;
	normalZ = 0.0;
	numNormals = 0;
}

void Point::AddNormals(float x,float y, float z)
{
	normalX += x;
	normalY += y;
	normalZ += z;
	numNormals++;
}

void Point::AvgNormal()
{
	float mag;
	if(numNormals>0)
	{
		normalX /= numNormals;
		normalY /= numNormals;
		normalZ /= numNormals;

		mag = sqrt(normalX * normalX + normalY * normalY + normalZ * normalZ);
		//normalise the normal
		normalX /= mag;
		normalY /= mag;
		normalZ /= mag;
	}
}