/****************************************************************************
*
*                               S H A N D Y
*
*                    Copyright (C) 1996 Andrew Cheung
*                          All rights reserved.
*
*  FILENAME :     $RCSfile: shandy.cpp $
*  VERSION  :     $Revision: 1.0 $
*  DATE     :     $Date: 1996/07/28 17:03:00 $
*  LANGUAGE :     C++
*
*  ENVIRONMENT:
*  IBM PC (MSDOS) Real Mode and 32 bit Protected Mode.
*  Watcom C/C++ 10.0,Borland C++ 3.1 - flat or large memory model
*
*  DESCRIPTION:
*  The main routines for the SHANDY 3D renderer and display program for
*  3D Studio .ASC files
*
*  $Id: shandy.cpp 1.0 1996/07/28 17:03:00 apwc Release apwc $
*
****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <time.h>
#include <dos.h>
#include "shandy.hpp"
#include "globals.hpp"
#include "defines.h"
#include "video.h"
#include "eventmgr.h"
#include "screen.h"

#ifdef SVGAKIT
	#include "svga.h"
#endif

//**************************   CONSTANTS   **********************************

// As defined in GLOBALS.HPP

//************************  GLOBAL VARIABLES  *******************************

int video_mode;
int MAXX,MAXY;
int XPROJ,YPROJ;

unsigned char *Screen;        // Pointer to Screen or Buffer
unsigned char *Buffer;        // a 64000 byte buffer for mode13h
int ScreenY[480];             // Index for start of each y scanline

unsigned char vgapal[3*256];  // 256 colour (rgb) palette
//Lite lookup table - 32 light levels for each of 256 colours in palette
unsigned char litetable[MAXLIGHT+1][PALETTESIZE];

event_struct events;
MatStruct **texture_list;     // materials list used in scene
Scene *scene;                 // The scene
Camera *camera;               // The active camera
Point *tPoint;                // A global point used for calculations
#ifdef SVGAKIT
palette svgapal[256];         // SVGA palette for 256 cols (if used)
#endif
//int texarray[64][64];       // 64*y values for 64x64 texture maps

//Globals used only by other modules:----------------------------------------
//SHADING.CPP  : long edge1[480],edge2[480],cedge1[480],cedge2[480];
//MATERIALS.CPP: int numTextures;
//TRIG.CPP     : float sin_table[360], float cosine_table[360];
//EVENTMGR.CPP : char keybuffer[128];

//****************************   FUNCTIONS   ********************************

void DisplayUsage(void);
void DisplayControls(void);
void PaletteOps(int global_shading);
void EventLoop();
void Demo(int x,int y,int z);
void Beep(unsigned frequency);

void main(int argc, char *argv[])
{
	FILE *file_asc,*file_tex;
	int global_shading;        //the global shading model specified by user

	unsigned long StartTime, EndTime, *timer;

	if(argc < 2)
	{
		DisplayUsage();
		exit(0);
	}

	if(argc > 2)
		global_shading = atoi(argv[2]);
	else
		global_shading = TEXTURE_FLAT;           // try for the 'best' shading

	//Step size for rotations around x,y,z axes (for demo)
	int x=4,y=0,z=0;
	if(argc>=4) x = atoi(argv[3]);
	if(argc>=5) y = atoi(argv[4]);
	if(argc>=6) z = atoi(argv[5]);

	if((file_asc = fopen(argv[1], "rt"))==NULL)
	{
		printf("Error opening file %s\n", argv[1]);
		DisplayUsage();
		exit(1);
	}

	//Open TEXTURES.DAT for reading whatever the .asc file is
	if(global_shading == TEXTURE_FLAT)
	{
		if((file_tex = fopen("textures.dat","rt"))==NULL)
		{
			printf("Error opening file TEXTURES.DAT\n");
			exit(1);
		}
		// Set up an array of 100 pointers to materials
		// Tried to have it dynamically set up without 100 materials max.
		// but it didn't work properly
		texture_list = new MatStruct*[100];
	}

	//Initialise a global point (used for vector calculations)
	tPoint = new Point(0,0,1);

	setmode(3);                // Clear the screen
	printf("S H A N D Y v1.0  - 3D Studio object viewer (July 1996)\n");
	printf("by Andrew Cheung (c) 1996\n\n");
	printf("Loading %s...\n",argv[1]);
	//**********************  LOAD THE SCENE  ********************************

	scene = new Scene();       // The only scene
	scene->LoadScene(file_asc,file_tex,&global_shading);

	//************************************************************************

	CalcTrigTables();          // Initialise sine & cos tables
	printf("Cameras = %d, Lights = %d \n",scene->numCameras,scene->numLights);
	printf("Press a key...\n");
	getch();
	setmode(3);                // Clear the screen
	DisplayControls();
	printf("Press a key...\n");
	getch();
	setmode(3);                // Clear the screen
#ifdef SVGAKIT
	video_mode = GetVideoMode();
#else
	video_mode = 0x13;         // Mode13h
#endif
	InitVideoMode(video_mode);
	SetVideoMode(video_mode);

	// Load texmap palette,litetable & finally set the VGA/SVGA palette depending
	// on the shading model specified
	PaletteOps(global_shading);

	//*************************   EVENTLOOPS   *******************************
	timer = (unsigned long *) 0x046C;
	StartTime = *timer;

	if(argc <= 3)
		EventLoop();
	else
		Demo(x,y,z);

	EndTime = *timer;

	//************************************************************************
	EndGraphics();

	printf("Time taken = %4.2f seconds\n", (float)(EndTime - StartTime)/18.2 );
	printf("		     ``\"''\n");
	puts("                     (o o)");
	puts("    ------------oooO--(_)--Oooo---------------");
	puts("   //                                         \\\\");
	puts("  //   Andrew Cheung                           \\\\");
	puts(" //\\   e-mail : apwc@doc.ic.ac.uk              /\\\\");
	puts(" \\\\/   WWW    : http://mkn.co.uk/help/         \\//");
	puts("  \\\\            extra/people/andrewcv.html     //");
	puts("   \\\\                         		      //");
	puts("    ------------------------------------------");
	puts("                   _( ) ( )_ ");
	puts("                  (___) (___)");

	fclose(file_asc);
	fclose(file_tex);
	delete []texture_list;
#ifdef _DEBUG
	printf("Size Of Point class is: %d\n",sizeof(Point));
	printf("Size Of Polygon class is: %d\n",sizeof(Polygon));
	printf("Size Of Object class is: %d\n",sizeof(Object));
	printf("Size Of Scene class is: %d\n",sizeof(Scene));
	printf("Size Of Camera class is: %d\n",sizeof(Camera));
#endif
}

//***************************************************************************

// A template for all the transformations an item can undergo
template <class T> Transform(T Item,int step)
{
	//translations - move relative to the active camera
	if(events.go_forward)
		Item->TranslateLocal(-step*camera->Nx,-step*camera->Ny,-step*camera->Nz);
	if(events.go_back)
		Item->TranslateLocal(step*camera->Nx,step*camera->Ny,step*camera->Nz);
	if(events.go_left)
		Item->TranslateLocal(-step*camera->Ux,-step*camera->Uy,-step*camera->Uz);
	if(events.go_right)
		Item->TranslateLocal(step*camera->Ux,step*camera->Uy,step*camera->Uz);
	if(events.go_up)
		Item->TranslateLocal(step*camera->Vx,step*camera->Vy,step*camera->Vz);
	if(events.go_down)
		Item->TranslateLocal(-step*camera->Vx,-step*camera->Vy,-step*camera->Vz);

	//rotations
	if(events.rote_up)
		Item->RotateLocalX(-step);
	if(events.rote_down)
		Item->RotateLocalX(step);
	if(events.rote_right)
		Item->RotateLocalY(-step);
	if(events.rote_left)
		Item->RotateLocalY(step);
	if(events.rote_clock)
		Item->RotateLocalZ(step);
	if(events.rote_aclock)
		Item->RotateLocalZ(-step);
}

void EventLoop()
{
	int i,step=4;

	int CamNum=0,LightNum=0,ObjNum=0;
	int type = S_SCENE;

	void *Ptr;
	Object *curr_obj;
	Point *curr_light;

	curr_light = scene->light[0];    // set initial light to the first light
	curr_obj = scene->object[0];     // set initial item to the first object
	Ptr = curr_obj;

	//init keyboard handler, set up mouse and joystick
	InitEvents();
	ClearEvents(&events);

	for (;;)
	{
		i = GetEvents(KEYBOARD_EVENTS,&events);
		if(events.quit)
			break;

		if(events.change_mode)
		{
			if( (scene->object[(ObjNum)%(scene->numObjs)]->StartShade != TEXTURE_FLAT)
			 && ((events.change_mode==TEXTURE_FLAT) || (events.change_mode==TEXTURE)) )
				;
			else
			{
				if(type == S_SCENE)
				{
					for(i=0; i < scene->numObjs; i++)
						scene->object[i]->SetShading(events.change_mode);
				}
				else
					scene->object[(ObjNum)%(scene->numObjs)]->SetShading(events.change_mode);
				switch(events.change_mode)
				{
					case DOTS      : Set256GreyPal(vgapal); break;
					case GOURAUD   : Set256Pal(vgapal); break;
					case GOURAUD_PHONG : SetPhongPal(vgapal,0.5,5.0); break;
					case TEXTURE       :
					case TEXTURE_FLAT  : Set256TexMapPal(vgapal) ;break;
				}
			}
		}

		if(events.selected)
		{
			type = events.selected;
			switch(type)
			{
				case S_CAMERA :
					camera = scene->cameras[(CamNum++)%(scene->numCameras)];
					camera->position->SetNewLocalOrigin(camera->target);
					Ptr = camera->position;break;

				case S_CAMERA_TARGET :
					//camera->target->SetNewLocalOrigin(curr_obj->midpoint);
					//camera->target = scene->object[(ObjNum++)%(scene->numObjs)]->midpoint;
					Ptr = camera->target;Beep(700); break;

				case S_OBJECT :
					curr_obj = scene->object[(ObjNum++)%(scene->numObjs)];
					Ptr = curr_obj;
					curr_obj->SetLocation(curr_obj->origin->x3d,curr_obj->origin->y3d,\
											 curr_obj->origin->z3d);
					break;

				case S_ORIGIN :
					Ptr = curr_obj->origin;
					Beep(3500);break;

				case S_LIGHT :
					curr_light = scene->light[(LightNum++)%(scene->numLights)];
					Ptr = curr_light; break;
			}
		}
		//Transformations
		switch(type)
		{
			case S_CAMERA        :
			case S_CAMERA_TARGET : camera->SetCamSpace();
			case S_LIGHT         :
			case S_ORIGIN        : Transform((Point *)Ptr,step);
										  break;
			case S_OBJECT        : Transform((Object *)Ptr,step);
										  break;
			case S_SCENE         : for(i=0; i < scene->numObjs; i++)
										  {
											  Ptr = scene->object[i];
											  Transform((Object *)Ptr,step);
										  }
										  break;
		}

		if(events.change_step)
			step += events.change_step;
		if(events.set_origin)
			curr_obj->SetLocation(curr_obj->midpoint->x3d,curr_obj->midpoint->y3d,\
										 curr_obj->midpoint->z3d);

		curr_obj->midpoint->SetNewLocalOrigin(curr_obj->origin);

		if(type == S_LIGHT)
			curr_light->EyeSpace();
		if(type == S_ORIGIN)
			curr_obj->origin->EyeSpace();

		scene->EyeSpace();
#ifndef PROFILE
		scene->SortObjs();
#endif
		scene->Display();

		if(type == S_LIGHT)
			curr_light->Display(255);
		if(type == S_ORIGIN)
			curr_obj->origin->Display(255);
		//curr_obj->midpoint->Display(255);
		PageFlip(Screen);
		ClearScreen(Screen);
	}
	EndEvents();
}

//Just rotate a scene around the x,y,z axes 500 times
//I use it for profiling performance
void Demo(int x,int y,int z)
{
	int i,j;

	for (i = 1; (i <= 500) && !kbhit(); i++)
	{
		for(j = 0; j < scene->numObjs; j++)
			scene->object[j]->RotateLocal(x,y,z);
			//scene->object[j]->RotateLocalY(y);

		scene->EyeSpace();
		scene->Display();
		PageFlip(Screen);
		ClearScreen(Screen);
	}
}

void DisplayUsage()
{
	printf("S H A N D Y v1.0  - 3D Studio object viewer (July 1996)\n");
	printf("by Andrew Cheung (c) 1996\n\n");
	printf("USAGE: SHANDY <filename.asc> <shading>\n");
	printf("       shading: 1=dots, 2=wireframe, 3=Lambert, 5,6,7=Gouraud, 9,0=texture\n");
}

void DisplayControls()
{
	printf("Controls");
	printf("\n--------\n");
	printf("Selecting items to move:\n");
	printf("   S,O,C,L        :Select a (S)cene,(O)bject,(C)amera or (L)ight\n");
	printf("   SHIFT + O,C    :Select origin of (O)bject, target of (C)amera \n");
	printf("   SPACEBAR       :Toggle continuous motion\n");
	printf("   NUMPAD: +-     :Increase/Decrease step size for translation/rotation\n\n");
	printf("   TAB            :Set origin of object to its midpoint\n");
	printf("   ESC            :Quit\n\n");
	printf("Transformations:\n");
	printf("   [UP][DOWN] ARROW                      :Move item forwards|backwards\n");
	printf("   [LEFT][RIGHT] ARROW                   :Rotate \"  left|right\n");
	printf("   ALT + [UP][DOWN][LEFT][RIGHT] ARROW	 :Move   \"  up,down,left right\n");
	printf("   CTRL + [UP][DOWN][LEFT][RIGHT] ARROW	 :Rotate \"  up,down,anti/clockwise\n\n");
	printf("Rendering modes:\n");
	printf("   [1]   DOTS          [6]   GOURAUD (flat shaded)\n");
	printf("   [2]   WIREFRAME     [7]   GOURAUD (Phong palette)\n");
	printf("   [3]   FLAT SHADED   [8]   ---\n");
	printf("   [4]   ---           [9]   TEXTURE MAPPED (flat shaded)\n");
	printf("   [5]   GOURAUD       [0]   TEXTURE MAPPED\n");
}

void Beep(unsigned frequency)
{
	sound(frequency);
	delay(100);
	nosound();
}

// Load texture palette, set the 256 VGA/SVGA colour palette depending on
// the shading model specified & load litetable
void PaletteOps(int global_shading)
{
	if(global_shading == TEXTURE_FLAT)
	{
		FILE *handle;
		char DatFileName[80]="";
		char *ptr;

		//Can only have one Texture light table, so arbitrarily choose the first texture
		ptr = strchr(texture_list[0]->FileName,'.');
		strncpy(DatFileName,texture_list[0]->FileName,ptr - texture_list[0]->FileName);
		strcat(DatFileName,".dat");

		if((handle=fopen(DatFileName,"rb"))==NULL)
		{
			printf("Error opening file %s\n",DatFileName);
			exit(1);
		}

		fread(litetable,MAXLIGHT+1,PALETTESIZE,handle);
		fclose(handle);
		printf("Loaded %s as THE LiteTable file\n",DatFileName);

		//set the palette to the texmap palette
		Set256TexMapPal(vgapal);
	}
	else
		if(global_shading == GOURAUD_PHONG)
			SetPhongPal(vgapal,0.5,5.0);
		else
			Set256Pal(vgapal);
}

#ifdef WATCOM32
void setmode(short mode);
//Set VGA adapter to BIOS MODE mode
#pragma aux setmode = \
	"int     10h"     \
parm [ax];

#endif
