/* Draws a bitmap, mapped to a convex polygon (draws a texture-mapped
   polygon). "Convex" means that every horizontal line drawn through
   the polygon at any point would cross exactly two active edges
   (neither horizontal lines nor zero-length edges count as active
   edges; both are acceptable anywhere in the polygon), and that the
   right & left edges never cross. Nonconvex polygons won't be drawn
   properly. Can't fail. */

#include <stdio.h>
#include <xlib_all.h>
#include "project.h"
#include "gfx.h"
#include "math.h"
#include "object.h"

#define INT_TO_FIXED(x) (((long)(int)x) << 16)
#define DOUBLE_TO_FIXED(x) ((long) (x * 65536.0 + 0.5))
#define FIXED_TO_DOUBLE(x) (((double)x) / 65536.0)
#define FIXED_TO_INT(FixedVal) ((int) (FixedVal >> 16))
#define ROUND_FIXED_TO_INT(FixedVal) \
   ((int) ((FixedVal + DOUBLE_TO_FIXED(0.5)) >> 16))
#define GET_IMAGE_PIXEL(TexMapBits, TexMapWidth, X, Y) \
   TexMapBits[(Y * TexMapWidth) + X]

#define SC_INDEX 0x3c4
#define MAP_MASK 0x02

/* Describes the current location and stepping, in both the source and
   the destination, of an edge. Mirrors structure in DRAWTEXP.C. */
typedef struct {
   int Direction;    /* through edge list; 1 for a right edge (forward
                        through vertex list), -1 for a left edge (backward
                        through vertex list) */
   int RemainingScans;  /* height left to scan out in dest */
   int CurrentEnd;      /* vertex # of end of current edge */
   Fixedpoint SourceX;     /* current X location in source for this edge */
   Fixedpoint SourceY;     /* current Y location in source for this edge */
   Fixedpoint SourceStepX; /* X step in source for Y step in dest of 1 */
   Fixedpoint SourceStepY; /* Y step in source for Y step in dest of 1 */
                        /* variables used for all-integer Bresenham's-type
                           X stepping through the dest, needed for precise
                           pixel placement to avoid gaps */
   int DestX;           /* current X location in dest for this edge */
   int DestXIntStep;    /* whole part of dest X step per scan-line Y step */
   int DestXDirection;  /* -1 or 1 to indicate which way X steps
                           (left/right) */
   int DestXErrTerm;    /* current error term for dest X stepping */
   int DestXAdjUp;      /* amount to add to error term per scan line move */
   int DestXAdjDown;    /* amount to subtract from error term when the
                           error term turns over */
} EdgeScan;

int StepEdge(EdgeScan *);
int SetUpEdge(EdgeScan *, int);
void ScanOutLine(EdgeScan *, EdgeScan *);
int GetImagePixel(char *, int, int, int);

/* Statics to save time that would otherwise be spent passing them to
   subroutines. */
int DestY;
static int MaxVert, NumVerts;
static POINT_2D * VertexPtr;
static POINT_2D * TexVertsPtr;
char * TexMapBits;
int TexMapWidth;

/* Draws a texture-mapped polygon, given a list of destination polygon
   vertices, a list of corresponding source texture polygon vertices, and a
   pointer to the source texture's descriptor. */
void DrawTexturedPolygon(POINT_2D *polygon, int numvertices, POINT_2D *TexVerts, BITMAP *TexMap)
{
   int MinY, MaxY, MinVert, i;
   EdgeScan LeftEdge, RightEdge;

   NumVerts = numvertices;
   VertexPtr = polygon;
   TexVertsPtr = TexVerts;
   TexMapBits = TexMap->data;
   TexMapWidth = TexMap->width;

   /* Nothing to draw if less than 3 vertices */
   if (NumVerts < 3) {
      return;
   }

   /* Scan through the destination polygon vertices and find the top of
      the left and right edges, taking advantage of our knowledge that
      vertices run in a clockwise direction (else this polygon wouldn't
      be visible due to backface removal) */
   MinY = 32767;
   MaxY = -32767;
   for (i=0; i<NumVerts; i++) {
      if (VertexPtr[i].y < MinY) {
         MinY = VertexPtr[i].y;
         MinVert = i;
      }
      if (VertexPtr[i].y > MaxY) {
         MaxY = VertexPtr[i].y;
         MaxVert = i;
      }
   }

   /* Reject flat (0-pixel-high) polygons */
   if (MinY >= MaxY) {
      return;
   }

   /* The destination Y coordinate is not edge specific; it applies to
      both edges, since we always step Y by 1 */
   DestY = MinY;

   /* Set up to scan the initial left and right edges of the source and
      destination polygons. We always step the destination polygon edges
      by one in Y, so calculate the corresponding destination X step for
      each edge, and then the corresponding source image X and Y steps */
   LeftEdge.Direction = -1;   /* set up left edge first */
   SetUpEdge(&LeftEdge, MinVert);
   RightEdge.Direction = 1;   /* set up right edge */
   SetUpEdge(&RightEdge, MinVert);

   /* Step down destination edges one scan line at a time. At each scan
      line, find the corresponding edge points in the source image. Scan
      between the edge points in the source, drawing the corresponding
      pixels across the current scan line in the destination polygon. (We
      know which way the left and right edges run through the vertex list
      because visible (non-backface-culled) polygons always have the vertices
      in clockwise order as seen from the viewpoint) */
   for (;;) {
      /* Draw the scan line between the two current edges */
      ScanOutLine(&LeftEdge, &RightEdge);

      /* Advance the source and destination polygon edges, ending if we've
         scanned all the way to the bottom of the polygon */
      if (!StepEdge(&LeftEdge)) {
         break;
      }
      if (!StepEdge(&RightEdge)) {
         break;
      }
      DestY++;
   }
}

/* Steps an edge one scan line in the destination, and the corresponding
   distance in the source. If an edge runs out, starts a new edge if there
   is one. Returns 1 for success, or 0 if there are no more edges to scan. */
int StepEdge(EdgeScan * Edge)
{
   /* Count off the scan line we stepped last time; if this edge is
      finished, try to start another one */
   if (--Edge->RemainingScans == 0) {
      /* Set up the next edge; done if there is no next edge */
      if (SetUpEdge(Edge, Edge->CurrentEnd) == 0) {
         return(0);  /* no more edges; done drawing polygon */
      }
      return(1);     /* all set to draw the new edge */
   }

   /* Step the current source edge */
   Edge->SourceX += Edge->SourceStepX;
   Edge->SourceY += Edge->SourceStepY;

   /* Step dest X with Bresenham-style variables, to get precise dest pixel
      placement and avoid gaps */
   Edge->DestX += Edge->DestXIntStep;  /* whole pixel step */
   /* Do error term stuff for fractional pixel X step handling */
   if ((Edge->DestXErrTerm += Edge->DestXAdjUp) > 0) {
      Edge->DestX += Edge->DestXDirection;
      Edge->DestXErrTerm -= Edge->DestXAdjDown;
   }

   return(1);
}

/* Sets up an edge to be scanned; the edge starts at StartVert and proceeds
   in direction Edge->Direction through the vertex list. Edge->Direction must
   be set prior to call; -1 to scan a left edge (backward through the vertex
   list), 1 to scan a right edge (forward through the vertex list).
   Automatically skips over 0-height edges. Returns 1 for success, or 0 if
   there are no more edges to scan. */
int SetUpEdge(EdgeScan * Edge, int StartVert)
{
   int NextVert, DestXWidth;
   Fixedpoint DestYHeight;

   for (;;) {
      /* Done if this edge starts at the bottom vertex */
      if (StartVert == MaxVert) {
         return(0);
      }

      /* Advance to the next vertex, wrapping if we run off the start or end
         of the vertex list */
      NextVert = StartVert + Edge->Direction;
      if (NextVert >= NumVerts) {
         NextVert = 0;
      } else if (NextVert < 0) {
         NextVert = NumVerts - 1;
      }

      /* Calculate the variables for this edge and done if this is not a
         zero-height edge */
      if ((Edge->RemainingScans =
            VertexPtr[NextVert].y - VertexPtr[StartVert].y) != 0) {
         DestYHeight = INT_TO_FIXED(Edge->RemainingScans);
         Edge->CurrentEnd = NextVert;
         Edge->SourceX = INT_TO_FIXED(TexVertsPtr[StartVert].x);
         Edge->SourceY = INT_TO_FIXED(TexVertsPtr[StartVert].y);
         Edge->SourceStepX = FixedDiv(INT_TO_FIXED(TexVertsPtr[NextVert].x) -
               Edge->SourceX, DestYHeight);
         Edge->SourceStepY = FixedDiv(INT_TO_FIXED(TexVertsPtr[NextVert].y) -
               Edge->SourceY, DestYHeight);

         /* Set up Bresenham-style variables for dest X stepping */
         Edge->DestX = VertexPtr[StartVert].x;
         if ((DestXWidth =
               (VertexPtr[NextVert].x - VertexPtr[StartVert].x)) < 0) {
            /* Set up for drawing right to left */
            Edge->DestXDirection = -1;
            DestXWidth = -DestXWidth;
            Edge->DestXErrTerm = 1 - Edge->RemainingScans;
            Edge->DestXIntStep = -(DestXWidth / Edge->RemainingScans);
         } else {
            /* Set up for drawing left to right */
            Edge->DestXDirection = 1;
            Edge->DestXErrTerm = 0;
            Edge->DestXIntStep = DestXWidth / Edge->RemainingScans;
         }
         Edge->DestXAdjUp = DestXWidth % Edge->RemainingScans;
         Edge->DestXAdjDown = Edge->RemainingScans;
         return(1);  /* success */
      }
      StartVert = NextVert;   /* keep looking for a non-0-height edge */
   }
}

void ScanOutLine(EdgeScan * LeftEdge, EdgeScan * RightEdge)
{
   Fixedpoint SourceX;
   Fixedpoint SourceY;
   int DestX = LeftEdge->DestX;
   int DestXMax = RightEdge->DestX;
   Fixedpoint DestWidth;
   Fixedpoint SourceStepX, SourceStepY;

   if ((DestXMax - DestX) <= 0) {
      return;  /* nothing to draw */
   }

   SourceX = LeftEdge->SourceX;
   SourceY = LeftEdge->SourceY;

   /* Width of destination scan line, for scaling. Note: because this is an
      integer-based scaling, it can have a total error of as much as nearly
      one pixel. For more precise scaling, also maintain a fixed-point DestX
      in each edge, and use it for scaling. If this is done, it will also
      be necessary to nudge the source start coordinates to the right by an
      amount corresponding to the distance from the the real (fixed-point)
      DestX and the first pixel (at an integer X) to be drawn) */
   DestWidth = INT_TO_FIXED(DestXMax - DestX);

   /* Calculate source steps that correspond to each dest X step (across
      the scan line) */
   SourceStepX = FixedDiv(RightEdge->SourceX - SourceX, DestWidth);
   SourceStepY = FixedDiv(RightEdge->SourceY - SourceY, DestWidth);

   /* Advance 1/2 step in the stepping direction, to space scanned pixels
      evenly between the left and right edges. (There's a slight inaccuracy
      in dividing negative numbers by 2 by shifting rather than dividing,
      but the inaccuracy is in the least significant bit, and we'll just
      live with it.) */
   SourceX += SourceStepX >> 1;
   SourceY += SourceStepY >> 1;

   /* Scan across the destination scan line, updating the source image
      position accordingly */

   for (; DestX<DestXMax; DestX++) {
   
       /* Get the currently mapped pixel out of the image and draw it to
          the screen */
       x_put_pix(DestX, DestY, HiddenPageOffs,
             GET_IMAGE_PIXEL(TexMapBits, TexMapWidth,
             ROUND_FIXED_TO_INT(SourceX), ROUND_FIXED_TO_INT(SourceY)) );

       /* Point to the next source pixel */
       SourceX += SourceStepX;
       SourceY += SourceStepY;
   }
}

