; Fixed point routines, written by Michael Abrash.
; Tested with TASM 3.0.

MUL_ROUNDING_ON equ     1       ;1 for rounding on multiplies,
                                ; 0 for no rounding. Not rounding is faster,
                                ; rounding is more accurate and generally a
                                ; good idea
DIV_ROUNDING_ON equ     0       ;1 for rounding on divides,
                                ; 0 for no rounding. Not rounding is faster,
                                ; rounding is more accurate, but because
                                ; division is only performed to project to
                                ; the screen, rounding quotients generally
                                ; isn't necessary

        IDEAL
        LOCALS
        P386N
        MODEL   LARGE,C

        CODESEG

        PUBLIC  FixedMul
        PUBLIC  FixedDiv
        PUBLIC  XformPt
        PUBLIC  XformVec
        PUBLIC  ConcatXforms

;=====================================================================
; Multiplies two fixed-point values together.
; C far-callable as:
;       Fixedpoint FixedMul(Fixedpoint M1, Fixedpoint M2);

PROC    FixedMul
        ARG     M1:DWORD,M2:DWORD

        mov     eax,[M1]
        imul    [dword ptr M2] ;multiply
if MUL_ROUNDING_ON
        add     eax,8000h       ;round by adding 2^(-17)
        adc     edx,0           ;whole part of result is in DX
endif ;MUL_ROUNDING_ON
        shr     eax,16          ;put the fractional part in AX

        ret

ENDP    FixedMul

;=====================================================================
; Divides one fixed-point value by another.
; C far-callable as:
;       Fixedpoint FixedDiv(Fixedpoint Dividend, Fixedpoint Divisor);

PROC    FixedDiv
        ARG     Dividend:DWORD,Divisor:DWORD

if DIV_ROUNDING_ON
        sub     cx,cx           ;assume positive result
        mov     eax,[Dividend]
        and     eax,eax         ;positive dividend?
        jns     FDP1            ;yes
        inc     cx              ;mark it's a negative dividend
        neg     eax             ;make the dividend positive
FDP1:   sub     edx,edx         ;make it a 64-bit dividend, then shift
                                ; left 16 bits so that result will be
                                ; in EAX
        rol     eax,16          ;put fractional part of dividend in
                                ; high word of EAX
        mov     dx,ax           ;put whole part of dividend in DX
        sub     ax,ax           ;clear low word of EAX
        mov     ebx,[dword ptr Divisor]
        and     ebx,ebx         ;positive divisor?
        jns     FDP2            ;yes
        dec     cx              ;mark it's a negative divisor
        neg     ebx             ;make divisor positive
FDP2:   div     ebx             ;divide
        shr     ebx,1           ;divisor/2, minus 1 if the divisor is
        adc     ebx,0           ; even
        dec     ebx
        cmp     ebx,edx         ;set Carry if the remainder is at least
        adc     eax,0           ; half as large as the divisor, then
                                ; use that to round up if necessary
        and     cx,cx           ;should the result be made negative?
        jz      FDP3            ;no
        neg     eax             ;yes, negate it
FDP3:
else ;!DIV_ROUNDING_ON
        mov     edx,[Dividend]
        sub     eax,eax
        shrd    eax,edx,16      ;position so that result ends up
        sar     edx,16          ; in EAX
        idiv    [dword ptr Divisor]
endif ;DIV_ROUNDING_ON
        shld    edx,eax,16      ;whole part of result in DX;
                                ; fractional part is already in AX

        ret

ENDP    FixedDiv

;=====================================================================
; Matrix multiplies Xform by SourcePt, and stores the result in
; DestPt. Multiplies a 4x4 matrix times a 4x1 matrix; the result
; is a 4x1 matrix. Cheats by assuming the W coord is 1 and the
; bottom row of the matrix is 0 0 0 1, and doesn't bother to set
; the W coordinate of the destination.
; C far-callable as:
;       void XformPt(Xform WorkingXform, Fixedpoint *SourcePt,
;               Fixedpoint *DestPt);
;
; This assembly code is equivalent to this C code:
;   int i;
;
;   for (i=0; i<3; i++)
;      DestPt[i] = FixedMul(WorkingXform[i][0], SourcePt[0]) +
;            FixedMul(WorkingXform[i][1], SourcePt[1]) +
;            FixedMul(WorkingXform[i][2], SourcePt[2]) +
;            WorkingXform[i][3];   /* no need to multiply by W = 1 */

PROC    XformPt
        ARG     WorkingXform:DWORD,SourcePt:DWORD,DestPt:DWORD

        push    si              ;preserve register variables
        push    di

        les     si,[WorkingXform]       ; es:si points to xform matrix
        lfs     bx,[SourcePt]           ; fs:bx points to source point
        lgs     di,[DestPt]             ; gs:di points to dest point

soff=0
doff=0
        REPT 3                  ;do once each for dest X, Y, and Z
        mov     eax,[es:si+soff]   ;column 0 entry on this row
        imul    [dword ptr fs:bx]  ;xform entry times source X entry
if MUL_ROUNDING_ON
        add     eax,8000h       ;round by adding 2^(-17)
        adc     edx,0           ;whole part of result is in DX
endif ;MUL_ROUNDING_ON
        shrd    eax,edx,16      ;shift the result back to 16.16 form
        mov     ecx,eax         ;set running total

        mov     eax,[es:si+soff+4] ;column 1 entry on this row
        imul    [dword ptr fs:bx+4] ;xform entry times source Y entry
if MUL_ROUNDING_ON
        add     eax,8000h       ;round by adding 2^(-17)
        adc     edx,0           ;whole part of result is in DX
endif ;MUL_ROUNDING_ON
        shrd    eax,edx,16      ;shift the result back to 16.16 form
        add     ecx,eax         ;running total for this row

        mov     eax,[es:si+soff+8] ;column 2 entry on this row
        imul    [dword ptr fs:bx+8] ;xform entry times source Z entry
if MUL_ROUNDING_ON
        add     eax,8000h       ;round by adding 2^(-17)
        adc     edx,0           ;whole part of result is in DX
endif ;MUL_ROUNDING_ON
        shrd    eax,edx,16      ;shift the result back to 16.16 form
        add     ecx,eax         ;running total for this row

        add     ecx,[es:si+soff+12] ;add in translation
        mov     [gs:di+doff],ecx   ;save the result in the dest vector
soff=soff+16
doff=doff+4
        ENDM

        pop     di              ;restore register variables
        pop     si
        ret

ENDP    XformPt

;=====================================================================
; Matrix multiplies Xform by SourceVec, and stores the result in
; DestVec. Multiplies a 4x4 matrix times a 4x1 matrix; the result
; is a 4x1 matrix. Cheats by assuming the W coord is 0 and the
; bottom row of the matrix is 0 0 0 1, and doesn't bother to set
; the W coordinate of the destination.
; C far-callable as:
;       void XformVec(Xform WorkingXform, Fixedpoint *SourceVec,
;               Fixedpoint *DestVec);
;
; This assembly code is equivalent to this C code:
;   int i;
;
;   for (i=0; i<3; i++)
;      DestVec[i] = FixedMul(WorkingXform[i][0], SourceVec[0]) +
;            FixedMul(WorkingXform[i][1], SourceVec[1]) +
;            FixedMul(WorkingXform[i][2], SourceVec[2]);

PROC    XformVec
        ARG     WorkingXform:DWORD,SourceVec:DWORD,DestVec:DWORD

        push    si              ;preserve register variables
        push    di

        les     si,[WorkingXform]       ; es:si points to xform matrix
        lfs     bx,[SourceVec]          ; fs:bx points to source vector
        lgs     di,[DestVec]            ; gs:di points to dest vector

soff=0
doff=0
        REPT 3                  ;do once each for dest X, Y, and Z
        mov     eax,[es:si+soff]   ;column 0 entry on this row
        imul    [dword ptr fs:bx]  ;xform entry times source X entry
if MUL_ROUNDING_ON
        add     eax,8000h       ;round by adding 2^(-17)
        adc     edx,0           ;whole part of result is in DX
endif ;MUL_ROUNDING_ON
        shrd    eax,edx,16      ;shift the result back to 16.16 form
        mov     ecx,eax         ;set running total

        mov     eax,[es:si+soff+4] ;column 1 entry on this row
        imul    [dword ptr fs:bx+4] ;xform entry times source Y entry
if MUL_ROUNDING_ON
        add     eax,8000h       ;round by adding 2^(-17)
        adc     edx,0           ;whole part of result is in DX
endif ;MUL_ROUNDING_ON
        shrd    eax,edx,16      ;shift the result back to 16.16 form
        add     ecx,eax         ;running total for this row

        mov     eax,[es:si+soff+8] ;column 2 entry on this row
        imul    [dword ptr fs:bx+8] ;xform entry times source Z entry
if MUL_ROUNDING_ON
        add     eax,8000h       ;round by adding 2^(-17)
        adc     edx,0           ;whole part of result is in DX
endif ;MUL_ROUNDING_ON
        shrd    eax,edx,16      ;shift the result back to 16.16 form
        add     ecx,eax         ;running total for this row

;        add     ecx,[es:si+soff+12] ;add in translation
        mov     [gs:di+doff],ecx   ;save the result in the dest vector
soff=soff+16
doff=doff+4
        ENDM

        pop     di              ;restore register variables
        pop     si
        ret

ENDP    XformVec

;=====================================================================
; Matrix multiplies SourceXform1 by SourceXform2 and stores the
; result in DestXform. Multiplies a 4x4 matrix times a 4x4 matrix;
; the result is a 4x4 matrix. Cheats by assuming the bottom row of
; each matrix is 0 0 0 1, and doesn't bother to set the bottom row
; of the destination.
; C far-callable as:
;       void ConcatXforms(Xform SourceXform1, Xform SourceXform2,
;               Xform DestXform)
;
; This assembly code is equivalent to this C code:
;   int i, j;
;
;   for (i=0; i<3; i++) {
;      for (j=0; j<3; j++)
;         DestXform[i][j] =
;               FixedMul(SourceXform1[i][0], SourceXform2[0][j]) +
;               FixedMul(SourceXform1[i][1], SourceXform2[1][j]) +
;               FixedMul(SourceXform1[i][2], SourceXform2[2][j]);
;      DestXform[i][3] =
;            FixedMul(SourceXform1[i][0], SourceXform2[0][3]) +
;            FixedMul(SourceXform1[i][1], SourceXform2[1][3]) +
;            FixedMul(SourceXform1[i][2], SourceXform2[2][3]) +
;            SourceXform1[i][3];
;   }

PROC    ConcatXforms
        ARG     SourceXform1:DWORD,SourceXform2:DWORD,DestXform:DWORD

        push    si              ;preserve register variables
        push    di

        les     si,[SourceXform1]
        lfs     bx,[SourceXform2]
        lgs     di,[DestXform]

roff=0                          ;row offset
        REPT 3                  ;once for each row
coff=0                          ;column offset
        REPT 3                  ;once for each of the first 3 columns,
                                ; assuming 0 as the bottom entry (no
                                ; translation)
        mov     eax,[es:si+roff]   ;column 0 entry on this row
        imul    [dword ptr fs:bx+coff] ;times row 0 entry in column
if MUL_ROUNDING_ON
        add     eax,8000h       ;round by adding 2^(-17)
        adc     edx,0           ;whole part of result is in DX
endif ;MUL_ROUNDING_ON
        shrd    eax,edx,16      ;shift the result back to 16.16 form
        mov     ecx,eax         ;set running total

        mov     eax,[es:si+roff+4] ;column 1 entry on this row
        imul    [dword ptr fs:bx+coff+16] ;times row 1 entry in col
if MUL_ROUNDING_ON
        add     eax,8000h       ;round by adding 2^(-17)
        adc     edx,0           ;whole part of result is in DX
endif ;MUL_ROUNDING_ON
        shrd    eax,edx,16      ;shift the result back to 16.16 form
        add     ecx,eax         ;running total

        mov     eax,[es:si+roff+8] ;column 2 entry on this row
        imul    [dword ptr fs:bx+coff+32] ;times row 2 entry in col
if MUL_ROUNDING_ON
        add     eax,8000h       ;round by adding 2^(-17)
        adc     edx,0           ;whole part of result is in DX
endif ;MUL_ROUNDING_ON
        shrd    eax,edx,16      ;shift the result back to 16.16 form
        add     ecx,eax         ;running total

        mov     [gs:di+coff+roff],ecx ;save the result in dest matrix
coff=coff+4                     ;point to next col in xform2 & dest
        ENDM
                                ;now do the fourth column, assuming
                                ; 1 as the bottom entry, causing
                                ; translation to be performed
        mov     eax,[es:si+roff]   ;column 0 entry on this row
        imul    [dword ptr fs:bx+coff] ;times row 0 entry in column
if MUL_ROUNDING_ON
        add     eax,8000h       ;round by adding 2^(-17)
        adc     edx,0           ;whole part of result is in DX
endif ;MUL_ROUNDING_ON
        shrd    eax,edx,16      ;shift the result back to 16.16 form
        mov     ecx,eax         ;set running total

        mov     eax,[es:si+roff+4] ;column 1 entry on this row
        imul    [dword ptr fs:bx+coff+16] ;times row 1 entry in col
if MUL_ROUNDING_ON
        add     eax,8000h       ;round by adding 2^(-17)
        adc     edx,0           ;whole part of result is in DX
endif ;MUL_ROUNDING_ON
        shrd    eax,edx,16      ;shift the result back to 16.16 form
        add     ecx,eax         ;running total

        mov     eax,[es:si+roff+8] ;column 2 entry on this row
        imul    [dword ptr fs:bx+coff+32] ;times row 2 entry in col
if MUL_ROUNDING_ON
        add     eax,8000h       ;round by adding 2^(-17)
        adc     edx,0           ;whole part of result is in DX
endif ;MUL_ROUNDING_ON
        shrd    eax,edx,16      ;shift the result back to 16.16 form
        add     ecx,eax         ;running total

        add     ecx,[es:si+roff+12] ;add in translation

        mov     [gs:di+coff+roff],ecx ;save the result in dest matrix
coff=coff+4                     ;point to next col in xform2 & dest

roff=roff+16                    ;point to next col in xform2 & dest
        ENDM

        pop     di              ;restore register variables
        pop     si
        ret

ENDP    ConcatXforms

        END

