/*  gfx_c.c
 *
 *  Some additional graphics routines written in C for simplicity.
 *
 *  Written and copyright (c) 1994 by Steve Madsen.
 */

#include <malloc.h>
#include <stdlib.h>
#include <conio.h>
#include <xlib_all.h>
#include "project.h"
#include "math.h"
#include "gfx.h"
#include "vgl.h"

word screenheight = 200;
POINT eye = { 0, 0, 0 };
POINT light = { 0, 0, 0 };
byte background = BLACK, page, backcolor = BLACK;
RECT dirty[3];
static PALETTE curpal;
PALETTE defpal;

/*  DrawBackground()
 *
 *  Updates the background on the currently hidden page.  Only redraws the
 *  part of the screen which is enclosed by the dirty rectangle for that
 *  page.
 */

void DrawBackground(void)
{
  int mod;

  if (background == NONE)
    return;

  if ((mod = dirty[page].ul.x % 4) != 0)
    dirty[page].ul.x -= mod;
  if ((mod = dirty[page].ul.y % 4) != 0)
    dirty[page].ul.y -= mod;
  if ((mod = dirty[page].lr.x % 4) != 0)
    dirty[page].lr.x += (4 - mod);
  if ((mod = dirty[page].lr.y % 4) != 0)
    dirty[page].lr.y += (4 - mod);
  if (background == GIF)
    x_cp_vid_rect(dirty[page].ul.x, dirty[page].ul.y, dirty[page].lr.x + 1, dirty[page].lr.y + 1,
                  dirty[page].ul.x, dirty[page].ul.y, NonVisual_Offs, HiddenPageOffs, 320, 320);
  else
    x_rect_fill(dirty[page].ul.x, dirty[page].ul.y, dirty[page].lr.x + 1, dirty[page].lr.y + 1, HiddenPageOffs, backcolor);
  dirty[page].ul.x = 320;
  dirty[page].ul.y = screenheight;
  dirty[page].lr.x = 0;
  dirty[page].lr.y = 0;
}

/*  ExtendRectangle()
 *
 *  Extends the dirty rectangle for the currently hidden page.
 */

void ExtendRectangle(POINT_2D *p)
{
  if (p->x < dirty[page].ul.x)
    dirty[page].ul.x = p->x;
  if (p->x > dirty[page].lr.x)
    dirty[page].lr.x = p->x;
  if (p->y < dirty[page].ul.y)
    dirty[page].ul.y = p->y;
  if (p->y > dirty[page].lr.y)
    dirty[page].lr.y = p->y;
}

/*  FadeIn()
 *
 *  Fades the entire 256-color palette from the current palette to a
 *  palette struct.  This gives nice fades from (say) black to the palette
 *  for a graphic image.  Very professional looking. :-)
 */

void FadeIn(PALETTE *topalette)
{
  PALETTE showpal;
  int index, done = FALSE;

  GetPalette(&showpal);
  while (!done) {
    done = TRUE;
    for (index = 0; index < 256; index++) {
      showpal.color[index].red += sign(topalette->color[index].red - showpal.color[index].red);
      showpal.color[index].green += sign(topalette->color[index]. green - showpal.color[index].green);
      showpal.color[index].blue += sign(topalette->color[index].blue - showpal.color[index].blue);
      if (showpal.color[index].red != topalette->color[index].red || showpal.color[index].green != topalette->color[index].green ||
      showpal.color[index].blue != topalette->color[index].blue)
        done = FALSE;
    }
    SetPalette(&showpal);
  }
}

/*  FadeOut()
 *
 *  Fades the entire 256-color palette from whatever it is currently set to,
 *  to a single color triplet.  This gives nice fades from your graphic
 *  image to (say) black.  Also very professional looking. :-)
 */

void FadeOut(COLOR *tocolor)
{
  PALETTE showpal;
  int index, done = FALSE;

  GetPalette(&showpal);
  while (!done) {
    done = TRUE;
    for (index = 0; index < 256; index++) {
      showpal.color[index].red += sign(tocolor->red - showpal.color[index].red);
      showpal.color[index].green += sign(tocolor-> green - showpal.color[index].green);
      showpal.color[index].blue += sign(tocolor->blue - showpal.color[index].blue);
      if (showpal.color[index].red != tocolor->red || showpal.color[index].green != tocolor->green ||
      showpal.color[index].blue != tocolor->blue)
        done = FALSE;
    }
    SetPalette(&showpal);
  }
}

/*  GraphicsDone()
 *
 *  Shuts down the graphics subsystem and returns to text mode.
 */

void GraphicsDone(void)
{
  x_remove_vsync_handler();
  x_text_mode();
}

/*  GraphicsInit()
 *
 *  Sets up the entire graphics subsystem.
 */

void GraphicsInit(void)
{
  byte index;

  x_set_mode(X_MODE_320x200, 320);
  screenheight = 200;
  x_install_vsync_handler(1);
  x_set_triplebuffer(screenheight);
  if (HiddenPageOffs == Page0_Offs)
    page = 0;
  else
    if (HiddenPageOffs == Page1_Offs)
      page = 1;
    else
      page = 2;
  for (index = 0; index < 3; index++) {
    dirty[index].ul.x = 320;
    dirty[index].ul.y = screenheight;
    dirty[index].lr.x = 0;
    dirty[index].lr.y = 0;
  }
  GetPalette(&defpal);
}

/*  Line()
 *
 *  A 3d interface for the x_line() function.
 */

void Line(POINT *from, POINT *to, int color)
{
  POINT_2D from2d, to2d;

  Project(from, &from2d);
  from2d.x += 160;
  from2d.y += screenheight >> 1;
  Project(to, &to2d);
  to2d.x += 160;
  to2d.y += screenheight >> 1;
  x_line(from2d.x, from2d.y, to2d.x, to2d.y, color, HiddenPageOffs);
  ExtendRectangle(&from2d);
  ExtendRectangle(&to2d);
}

/*  Line_2d()
 *
 *  A 2d interface to the x_line() function.
 */

void Line_2d(POINT_2D *from, POINT_2D *to, int color)
{
  POINT_2D extend;

  x_line(from->x + 160, from->y + (screenheight >> 1),
         to->x + 160, to->y + (screenheight >> 1), color, HiddenPageOffs);
  extend = *from;
  extend.x += 160;
  extend.y += screenheight >> 1;
  ExtendRectangle(&extend);
  extend = *to;
  extend.x += 160;
  extend.y += screenheight >> 1;
  ExtendRectangle(&extend);
}

/*  LoadBackground()
 *
 *  Sets up the background picture.
 */

void LoadBackground(char *picture, int fade)
{
  char *buffer;
  PALETTE pal;
  int width, height, x, y;

  if (picture == NULL) {
    background = BLACK;
    x_rect_fill(0, 0, 320, screenheight, Page0_Offs, backcolor);
    x_rect_fill(0, 0, 320, screenheight, Page1_Offs, backcolor);
    x_rect_fill(0, 0, 320, screenheight, Page2_Offs, backcolor);
  } else {
    buffer = malloc(320 * screenheight);
    vglGif(picture, buffer, (char *)&pal, &width, &height);
    for (y = 0; y < height; y++)
      for (x = 0; x < width; x++)
        x_put_pix(x, y, NonVisual_Offs, buffer[y*320 + x]);
    x_cp_vid_rect(0, 0, 320, screenheight, 0, 0, NonVisual_Offs, Page0_Offs, 320, 320);
    x_cp_vid_rect(0, 0, 320, screenheight, 0, 0, NonVisual_Offs, Page1_Offs, 320, 320);
    x_cp_vid_rect(0, 0, 320, screenheight, 0, 0, NonVisual_Offs, Page2_Offs, 320, 320);
    background = GIF;
    if (fade > 0)
      FadeIn(&pal);
    else
      SetPalette(&pal);
    free(buffer);
  }
}

/*  MakePoint()
 *
 *  Converts a coordinate triplet into something usable.
 */

POINT *MakePoint(float x, float y, float z, POINT *p)
{
  static POINT point;

  point.x = FLOAT_TO_FIXED(x);
  point.y = FLOAT_TO_FIXED(y);
  point.z = FLOAT_TO_FIXED(z);
  if (p != NULL) {
    p->x = FLOAT_TO_FIXED(x);
    p->y = FLOAT_TO_FIXED(y);
    p->z = FLOAT_TO_FIXED(z);
  }
  return &point;
}

/*  MoveEye()
 *
 *  Computes the vector used for visible surface determination.
 */

void MoveEye(POINT *p)
{
  eye.x = p->x;
  eye.y = p->y;
  eye.z = p->z;
}

/*  MoveLight()
 *
 *  Computes the vector used for light sourcing.
 */

void MoveLight(POINT *p)
{
  light.x = p->x;
  light.y = p->y;
  light.z = p->z;
}

/*  PageFlip()
 *
 *  Flips pages and keeps track of the current page.
 */

void PageFlip(void)
{
  COLOR black = { 0, 0, 0 };

  x_page_flip(0, 0);
  if (HiddenPageOffs == Page0_Offs) {
    page = 0;
    if (kbhit() && getch() == 27) {
      FadeOut(&black);
      GraphicsDone();
      exit(0);
    }
  } else
    if (HiddenPageOffs == Page1_Offs)
      page = 1;
    else
      page = 2;
  DrawBackground();
}

/*  Polygon()
 *
 *  A 3d interface for the x_polygon() function.
 */

void Polygon(POINT vertices[], int numvertices, int color)
{
  POINT_2D points[20];
  int vertex;

  for (vertex = 0; vertex < numvertices; vertex++) {
    Project(&vertices[vertex], &points[vertex]);
    points[vertex].x += 160;
    points[vertex].y += (screenheight >> 1);
    ExtendRectangle(&points[vertex]);
  }
  x_polygon((VERTEX far *)points, numvertices, color, HiddenPageOffs);
}

/*  Polygon_2d()
 *
 *  A 2d interface for the x_polygon() function.
 */

void Polygon_2d(POINT_2D vertices[], int numvertices, int color)
{
  int vertex;

  for (vertex = 0; vertex < numvertices; vertex++) {
    vertices[vertex].x += 160;
    vertices[vertex].y += (screenheight >> 1);
    ExtendRectangle(&vertices[vertex]);
  }
  x_polygon((VERTEX far *)vertices, numvertices, color, HiddenPageOffs);
}

/*  Project()
 *
 *  Projects a 3d point onto 2d screen coordinates.
 */

void Project(POINT *threeD, POINT_2D *twoD)
{
  twoD->x = (int)((FixedDiv(threeD->x - eye.x, -(threeD->z - eye.z)) << 8) >> 16);
  twoD->y = (int)((FixedDiv(threeD->y - eye.y, -(threeD->z - eye.z)) << 8) >> 16);
}

/*  Rectangle()
 *
 *  Nice way of drawing 4-vertex polygons without dealing with arrays.
 */

void Rectangle(POINT *p0, POINT *p1, POINT *p2, POINT *p3, int color)
{
  POINT_2D points[4];
  int index;

  Project(p0, &points[0]);
  Project(p1, &points[1]);
  Project(p2, &points[2]);
  Project(p3, &points[3]);
  for (index = 0; index < 4; index++) {
    points[index].x += 160;
    points[index].y += (screenheight >> 1);
    ExtendRectangle(&points[index]);
  }
  x_polygon((VERTEX far *)points, 4, color, HiddenPageOffs);
}

/*  ResyncPalette()
 *
 *  Resynchronizes the palette used when spinning.
 */

void ResyncPalette(void)
{
  GetPalette(&curpal);
}

/*  SetBackground
 *
 *  Sets the background color used to draw over parts of the screen.
 */

void SetBackground(byte color)
{
  backcolor = color;
}

/*  SetPalette()
 *
 *  Wrapper for assembler SetPalette().
 */

void _SetPalette(PALETTE *);  /* assembler version, does the real work */

void SetPalette(PALETTE *pal)
{
  int black, blackval, index;

  /* find the most black color and use that for our background */

  black = 0;
  blackval = pal->color[0].red + pal->color[0].green + pal->color[0].blue;
  for (index = 1; index < 256; index++) {
    if (pal->color[index].red + pal->color[index].green + pal->color[index].blue < blackval) {
      black = index;
      blackval = pal->color[index].red + pal->color[index].green + pal->color[index].blue;
      if (blackval == 0)  /* none more black :-) */
        break;
    }
  }
  SetBackground(black);
  _SetPalette(pal);
}

/*  ShadedPaletteInit()
 *
 *  Sets up the default palette for shaded objects.
 */

void ShadedPaletteInit(void)
{
  byte index;
  PALETTE pal;

  GetPalette(&pal);
  pal.color[BLACK].red = 0;
  pal.color[BLACK].green = 0;
  pal.color[BLACK].blue = 0;
  for (index = 1; index < SHADES; index++) {
    pal.color[RED_BASE + index - 1].red = (44 / SHADES) * index + 20;
    pal.color[RED_BASE + index - 1].green = 0;
    pal.color[RED_BASE + index - 1].blue = 0;
    pal.color[GREEN_BASE + index - 1].red = 0;
    pal.color[GREEN_BASE + index - 1].green = (44 / SHADES) * index + 20;
    pal.color[GREEN_BASE + index - 1].blue = 0;
    pal.color[BLUE_BASE + index - 1].red = 0;
    pal.color[BLUE_BASE + index - 1].green = 0;
    pal.color[BLUE_BASE + index - 1].blue = (44 / SHADES) * index + 20;
    pal.color[GREY_BASE + index - 1].red = (44 / SHADES) * index + 20;
    pal.color[GREY_BASE + index - 1].green = (44 / SHADES) * index + 20;
    pal.color[GREY_BASE + index - 1].blue = (44 / SHADES) * index + 20;
  }
  SetPalette(&pal);
}

/*  SpinPalette()
 *
 *  Spins the palette in either direction a given number of steps.
 */

void SpinPalette(int dir)
{
  int count = 0, sign, index;
  COLOR save;

  if (dir) {
    sign = dir / dir;
    while (count != dir) {
      if (sign > 0) {
        save = curpal.color[255];
        for (index = 255; index > 1; index--)
          curpal.color[index] = curpal.color[index - 1];
        curpal.color[1] = save;
      } else {
        save = curpal.color[1];
        for (index = 1; index < 255; index++)
          curpal.color[index] = curpal.color[index + 1];
        curpal.color[255] = save;
      }
      count += sign;
    }
    SetPalette(&curpal);
  }
}

