/*  object.c
 *
 *  Object manipulation and display module.  Handles complex objects composed
 *  of several surfaces (defined as convex polygons).
 *
 *  Written and copyright (c) 1994 by Steve Madsen.
 */

#include <stdlib.h>
#include <string.h>
#include <xlib_all.h>
#include "project.h"
#include "math.h"
#include "gfx.h"
#include "object.h"
#include "vgl.h"

/* private routines -- cannot be called external to this module */

static int DefinePoint(SURFACE *surface, POINT *point)
{
  OBJECT *object;

  object = surface->partof;
  object->points[object->num_points] = *point;
  return (object->num_points++);
}

static void ComputeSurfaceNormal(SURFACE *surface)
{
  OBJECT *object;
  Fixedpoint x1x2, x1x3, y1y2, y1y3, z1z2, z1z3;

  object = surface->partof;
  x1x2 = object->points[surface->points[0]].x - object->points[surface->points[1]].x;
  x1x3 = object->points[surface->points[0]].x - object->points[surface->points[2]].x;
  y1y2 = object->points[surface->points[0]].y - object->points[surface->points[1]].y;
  y1y3 = object->points[surface->points[0]].y - object->points[surface->points[2]].y;
  z1z2 = object->points[surface->points[0]].z - object->points[surface->points[1]].z;
  z1z3 = object->points[surface->points[0]].z - object->points[surface->points[2]].z;

  surface->normal.i = FixedMul(y1y2, z1z3) - FixedMul(z1z2, y1y3);
  surface->normal.j = -FixedMul(z1z2, x1x3) - FixedMul(x1x2, z1z3);
  surface->normal.k = FixedMul(x1x2, y1y3) - FixedMul(y1y2, x1x3);
  NormalizeVector(&surface->normal);
}

static Fixedpoint FarthestZOnSurface(SURFACE *surface)
{
  OBJECT *object;
  int farthest = 0, index;

  object = surface->partof;
  for (index = 1; index < surface->num_points; index++)
    if (object->cur_points[surface->points[index]].z < object->cur_points[surface->points[farthest]].z)
      farthest = index;
  return (object->cur_points[surface->points[farthest]].z + object->center.z);
}

static int ClosestPointOnSurface(SURFACE *surface)
{
  OBJECT *object;
  int closest = 0, index;

  object = surface->partof;
  for (index = 1; index < surface->num_points; index++)
    if (object->cur_points[surface->points[index]].z > object->cur_points[surface->points[closest]].z)
      closest = index;
  return (closest);
}

static void ComputeVector(POINT *from, POINT *to, VECTOR *vector)
{
  vector->i = to->x - from->x;
  vector->j = to->y - from->y;
  vector->k = to->z - from->z;
}

/* public routines -- these are prototyped in object.h */

void AddPoint(SURFACE *surface, POINT *point)
{
  int index, found = FALSE;
  OBJECT *object;

  if (surface->num_points == MAX_POINTS_PER_SURFACE) {
    x_text_mode();
    printf("Too many points defined for surface.\n");
    exit(1);
  }
  object = surface->partof;
  for (index = 0; index < object->num_points; index++)
    if (object->points[index].x == point->x && object->points[index].y == point->y && object->points[index].z == point->z) {
      found = TRUE;
      break;
    }
  if (found)
    surface->points[surface->num_points++] = index;
  else
    surface->points[surface->num_points++] = DefinePoint(surface, point);
  if (surface->num_points == 3)
    ComputeSurfaceNormal(surface);
}

void DestroyObject(OBJECT *object)
{
  SURFACE *surface, *next;

  surface = object->surfaces;
  while (surface != NULL) {
    next = surface->next;
    DestroySurface(surface);
    surface = next;
  }
  free(object);
}

void DestroySurface(SURFACE *surface)
{
  if (surface->bmap != NULL) {
    if (surface->bmap->data != NULL)
      free(surface->bmap->data);
    free(surface->bmap);
  }
  free(surface);
}

void DestroyWorld(WORLD *world)
{
  OBJECT *object, *next;

  object = world->objects;
  while (object != NULL) {
    next = object->next;
    DestroyObject(object);
    object = next;
  }
  free(world);
}

void DrawObject(OBJECT *object)
{
  SURFACE *zbuffer[MAX_SURFACES_PER_OBJECT], *surface;
  int index = 0, lowest;

  surface = object->surfaces;
  while (surface != NULL) {
    zbuffer[index++] = surface;
    surface = surface->next;
  }
  for (lowest = 0; lowest < object->num_surfaces - 1; lowest++)
    for (index = lowest + 1; index < object->num_surfaces; index++)
      if (FarthestZOnSurface(zbuffer[index]) < FarthestZOnSurface(zbuffer[lowest]))
        swap((long)zbuffer[index], (long)zbuffer[lowest]);
  for (index = 0; index < object->num_surfaces; index++)
    DrawSurface(zbuffer[index]);
}

void DrawSurface(SURFACE *surface)
{
  OBJECT *object;
  POINT_2D p[MAX_POINTS_PER_SURFACE], map[MAX_POINTS_PER_SURFACE];
  POINT p3[MAX_POINTS_PER_SURFACE];
  int index;
  VECTOR v1, v2;
  Fixedpoint dot;

  object = surface->partof;
  for (index = 0; index < surface->num_points; index++) {
    p3[index].x = object->cur_points[surface->points[index]].x + object->center.x;
    p3[index].y = object->cur_points[surface->points[index]].y - object->center.y;
    p3[index].z = object->cur_points[surface->points[index]].z + object->center.z;
    Project(&p3[index], &p[index]);
  }
  v1.i = p[1].x - p[0].x;
  v1.j = p[1].y - p[0].y;
  v2.i = p[2].x - p[0].x;
  v2.j = p[2].y - p[0].y;
  if (v1.i * v2.j - v2.i * v1.j > 0) {
    switch (surface->style) {
      case SOLID:
        Polygon_2d(p, surface->num_points, surface->color);
        break;

      case TEXTURED:
        map[0].x = 0;
        map[0].y = 0;
        map[1].x = surface->bmap->width - 1;
        map[1].y = 0;
        map[2].x = surface->bmap->width - 1;
        map[2].y = surface->bmap->height - 1;
        map[3].x = 0;
        map[3].y = surface->bmap->height - 1;
        for (index = 0; index < surface->num_points; index++) {
          p[index].x += 160;
          p[index].y += (screenheight >> 1);
          ExtendRectangle(&p[index]);
        }
        DrawTexturedPolygon(p, surface->num_points, map, surface->bmap);
        break;

      case SHADED_RED:
        ComputeVector(&object->cur_points[surface->points[ClosestPointOnSurface(surface)]], &light, &v1);
        NormalizeVector(&v1);
        dot = DotProduct(&v1, &surface->cur_normal);
        index = FIXED_TO_INT(dot * SHADES);
        if (index >= SHADES)
          index = SHADES - 1;
        if (--index >= 0)
          Polygon_2d(p, surface->num_points, RED_BASE + index);
        else
          Polygon_2d(p, surface->num_points, BLACK);
        break;

      case SHADED_GREEN:
        ComputeVector(&object->cur_points[surface->points[ClosestPointOnSurface(surface)]], &light, &v1);
        NormalizeVector(&v1);
        dot = DotProduct(&v1, &surface->cur_normal);
        index = FIXED_TO_INT(dot * SHADES);
        if (index >= SHADES)
          index = SHADES - 1;
        if (--index >= 0)
          Polygon_2d(p, surface->num_points, GREEN_BASE + index);
        else
          Polygon_2d(p, surface->num_points, BLACK);
        break;

      case SHADED_BLUE:
        ComputeVector(&object->cur_points[surface->points[ClosestPointOnSurface(surface)]], &light, &v1);
        NormalizeVector(&v1);
        dot = DotProduct(&v1, &surface->cur_normal);
        index = FIXED_TO_INT(dot * SHADES);
        if (index >= SHADES)
          index = SHADES - 1;
        if (--index >= 0)
          Polygon_2d(p, surface->num_points, BLUE_BASE + index);
        else
          Polygon_2d(p, surface->num_points, BLACK);
        break;

      case SHADED_GREY:
        ComputeVector(&object->cur_points[surface->points[ClosestPointOnSurface(surface)]], &light, &v1);
        NormalizeVector(&v1);
        dot = DotProduct(&v1, &surface->cur_normal);
        index = FIXED_TO_INT(dot * SHADES);
        if (index >= SHADES)
          index = SHADES - 1;
        if (--index >= 0)
          Polygon_2d(p, surface->num_points, GREY_BASE + index);
        else
          Polygon_2d(p, surface->num_points, BLACK);
        break;

      case WIREFRAME:
        for (index = 0; index < surface->num_points - 1; index++)
          Line_2d(&p[index], &p[index + 1], surface->color);
        Line_2d(&p[surface->num_points - 1], &p[0], surface->color);
        break;

      default:
        GraphicsDone();
        printf("ERROR: unknown surface style %d.\n", surface->style);
    }
  }
}

void DrawWorld(WORLD *world)
{
  OBJECT *zbuffer[MAX_OBJECTS_PER_WORLD], *object;
  int index = 0, lowest;

  object = world->objects;
  while (object != NULL) {
    zbuffer[index++] = object;
    object = object->next;
  }
  for (lowest = 0; lowest < world->num_objects - 1; lowest++)
    for (index = lowest + 1; index < world->num_objects; index++)
      if (zbuffer[index]->center.z < zbuffer[lowest]->center.z)
        swap((long)zbuffer[index], (long)zbuffer[lowest]);
  for (index = 0; index < world->num_objects; index++)
    DrawObject(zbuffer[index]);
}

void LoadGifBitmap(SURFACE *surface, char *filename, int fade)
{
  char *buffer;
  PALETTE pal;
  int width, height, x, y;
  SURFACE *slist;

  if (surface->bmap != NULL) {
    if (surface->bmap->data != NULL)
      free(surface->bmap->data);
    free(surface->bmap);
  }
  surface->bmap = (BITMAP *)malloc(sizeof(BITMAP));

  /* first check to see if this bitmap has already been loaded before */

  slist = surface->partof->surfaces;
  while (slist != NULL) {
    if (slist->bmap != NULL && strcmp(slist->bmap->fname, filename) == 0) {
      surface->bmap = slist->bmap;
      return;
    }
    slist = slist->next;
  }

  /* it isn't, so go read the image */

  if ((buffer = malloc(320 * screenheight)) == NULL) {
    x_text_mode();
    printf("Not enough memory for GIF buffer.\n");
    exit(1);
  }
  if (!vglGif(filename, buffer, (char *)&pal, &width, &height)) {
    x_text_mode();
    printf("Can't find file %s.\n", filename);
    exit(1);
  }
  surface->bmap->height = height;
  surface->bmap->width = width;
  if ((surface->bmap->data = malloc(width * height)) == NULL) {
    x_text_mode();
    printf("Not enough memory for GIF bitmap malloc.\n");
    exit(1);
  }
  for (y = 0; y < height; y++)
    for (x = 0; x < width; x++)
      surface->bmap->data[(y * width) + x] = buffer[(y * width) + x];
  if (fade != 0)
    FadeIn(&pal);
  else
    SetPalette(&pal);
  free(buffer);
}

void LoadPxmBitmap(SURFACE *surface, char *filename)
{
  FILE *f;
  int x, y;
  SURFACE *slist;

  if (surface->bmap != NULL) {
    if (surface->bmap->data != NULL)
      free(surface->bmap->data);
    free(surface->bmap);
  }
  surface->bmap = (BITMAP *)malloc(sizeof(BITMAP));

  /* first check to see if this bitmap has already been loaded before */

  slist = surface->partof->surfaces;
  while (slist != NULL) {
    if (slist->bmap != NULL && strcmp(slist->bmap->fname, filename) == 0) {
      surface->bmap = slist->bmap;
      return;
    }
    slist = slist->next;
  }

  /* it isn't, so go read the image */

  if ((f = fopen(filename, "rb")) == NULL)
    return;
  surface->bmap->width = getw(f);
  surface->bmap->height = getw(f);
  surface->bmap->data = malloc(surface->bmap->width * surface->bmap->height);
  for (y = 0; y < surface->bmap->height; y++)
    for (x = 0; x < surface->bmap->width; x++)
      surface->bmap->data[(y * surface->bmap->width) + x] = getc(f);
  fclose(f);
}

OBJECT *NewObject(WORLD *world)
{
  OBJECT *object, *last;

  if (world->num_objects == MAX_OBJECTS_PER_WORLD) {
    x_text_mode();
    printf("Too many objects defined in world.\n");
    exit(1);
  }
  object = (OBJECT *)malloc(sizeof(OBJECT));
  MakePoint(0, 0, 0, &object->center);
  object->num_surfaces = 0;
  object->surfaces = NULL;
  object->num_points = 0;
  object->xcangle = 0;
  object->ycangle = 0;
  object->zcangle = 0;
  object->xcdelta = 0;
  object->ycdelta = 0;
  object->zcdelta = 0;
  object->next = NULL;
  if (world->objects == NULL)
    world->objects = object;
  else {
    last = world->objects;
    while (last->next != NULL)
      last = last->next;
    last->next = object;
  }
  world->num_objects++;
  return (object);
}

SURFACE *NewSurface(OBJECT *object)
{
  SURFACE *surface, *last;

  if (object->num_surfaces == MAX_SURFACES_PER_OBJECT) {
    x_text_mode();
    printf("Too many surfaces defined on object.\n");
    exit(1);
  }
  surface = (SURFACE *)malloc(sizeof(SURFACE));
  surface->num_points = 0;
  surface->style = 0;
  surface->bmap = NULL;
  surface->partof = object;
  surface->next = NULL;
  if (object->surfaces == NULL)
    object->surfaces = surface;
  else {
    last = object->surfaces;
    while (last->next != NULL)
      last = last->next;
    last->next = surface;
  }
  object->num_surfaces++;
  return (surface);
}

WORLD *NewWorld(void)
{
  WORLD *world;

  world = (WORLD *)malloc(sizeof(WORLD));
  world->num_objects = 0;
  world->objects = NULL;
  return (world);
}

void UpdateObject(OBJECT *object)
{
  Xform matrix;
  int vertex;
  SURFACE *surface;
  Fixedpoint x1x2, x1x3, y1y2, y1y3, z1z2, z1z3;

  /* update the points rotating around object center */

  object->xcangle += object->xcdelta;
  object->ycangle += object->ycdelta;
  object->zcangle += object->zcdelta;
  RotationMatrix(object->xcangle, object->ycangle, object->zcangle, matrix);
  for (vertex = 0; vertex < object->num_points; vertex++)
    XformPt(matrix, (Fixedpoint *)&object->points[vertex], (Fixedpoint *)&object->cur_points[vertex]);

  /* update surface normals */

  surface = object->surfaces;
  while (surface != NULL) {
    x1x2 = object->cur_points[surface->points[0]].x - object->cur_points[surface->points[1]].x;
    x1x3 = object->cur_points[surface->points[0]].x - object->cur_points[surface->points[2]].x;
    y1y2 = object->cur_points[surface->points[0]].y - object->cur_points[surface->points[1]].y;
    y1y3 = object->cur_points[surface->points[0]].y - object->cur_points[surface->points[2]].y;
    z1z2 = object->cur_points[surface->points[0]].z - object->cur_points[surface->points[1]].z;
    z1z3 = object->cur_points[surface->points[0]].z - object->cur_points[surface->points[2]].z;

    surface->cur_normal.i = FixedMul(y1y2, z1z3) - FixedMul(z1z2, y1y3);
    surface->cur_normal.j = -FixedMul(z1z2, x1x3) - FixedMul(x1x2, z1z3);
    surface->cur_normal.k = FixedMul(x1x2, y1y3) - FixedMul(y1y2, x1x3);
    NormalizeVector(&surface->cur_normal);
    surface = surface->next;
  }
}

void UpdateWorld(WORLD *world)
{
  OBJECT *object;

  object = world->objects;
  while (object != NULL) {
    UpdateObject(object);
    object = object->next;
  }
}

