%{

#include <math.h>
#include <ctype.h>
#include <xlib.h>
#include "object.h"
#include "gfx.h"
 
int lineno = 1;
OBJECT *object;
SURFACE *surface;
WORLD *world;
extern char *yytext;
 
void Error(char *format, ...);
POINT *MakePoint(float, float, float, POINT *);
float strtofloat(char *);

%}
 
%union {
  float fval;
  int ival;
  char sval[33];
}
 
%token T_OBJECT,T_VERTEX,T_CENTER,T_EYE,T_LIGHT,T_STYLE,T_COLOR
%token T_RED,T_GREEN,T_BLUE,T_GREY,T_WHITE,T_SHADED,T_TEXTURED,T_INT
%token T_FLOAT,T_NAME,T_LBRACE,T_RBRACE,T_SURFACE,T_ROTATE
%token T_GIFFILE,T_PXMFILE,T_COMMA,T_BACKGROUND,T_FADE,T_STARS,T_NONE
%token T_WIRE
 
%type <fval> fp
%type <sval> filename
%type <sval> giffile
%type <ival> integer
%type <ival> hues
%type <ival> fade
 
%%
 
world: eye light background object object_list
       | error { printf("Nothing to parse!\n"); YYABORT; }
       ;
 
eye: T_EYE lbrace fp comma fp comma fp rbrace { MoveEye(MakePoint($3, $5, $7, NULL)); }
     | T_EYE error '\n' { Error("eye coordinate definition is incorrect"); }
     ;
 
light: T_LIGHT lbrace fp comma fp comma fp rbrace { MoveLight(MakePoint($3, $5, $7, NULL)); }
       | T_LIGHT error '\n' { Error("light coordinate definition is incorrect"); }
       |
       ;

background: T_BACKGROUND backs
            | { LoadBackground(NULL, 0); SeedStars(0); }
            ;

backs: T_NONE { LoadBackground(NULL, 0); SeedStars(0); }
       | giffile fade { LoadBackground($1, $2); SeedStars(0); }
       | T_STARS integer { LoadBackground(NULL, 0); SeedStars($2); }
       ;

fade: T_FADE { $$ = 1; }
      | { $$ = 0; }
      ;

giffile: T_GIFFILE { strcpy($$, yytext); }
         ;
 
object_list: object object_list
             |
             ;
 
object: T_OBJECT { object = NewObject(world); } name lbrace center rotations surface surface_list rbrace
        ;
 
name: T_NAME
      |
      ;
 
center: T_CENTER lbrace fp comma fp comma fp rbrace { MakePoint($3, $5, $7, &object->center); }
        | T_CENTER error '}' { Error("bad or missing center definition"); }
        ;
 
rotations: T_ROTATE lbrace integer comma integer comma integer rbrace { object->xcdelta = $3; object->ycdelta = $5; object->zcdelta = $7; }
           | T_ROTATE error '}' { Error("incorrect rotation definition"); }
           ;
 
surface: T_SURFACE name { surface = NewSurface(object); } lbrace style vertex vertex vertex vertex_list rbrace
         ;
 
surface_list: surface surface_list
              |
              ;
 
style: T_STYLE styles
       | T_STYLE error '\n' { Error("incorrect style definition"); }
       ;
 
styles: T_COLOR integer { surface->style = SOLID; surface->color = $2; }
        | T_SHADED hues { surface->style = $2; ShadedPaletteInit(); }
        | T_TEXTURED { surface->style = TEXTURED; } filename
        | T_WIRE T_COLOR integer { surface->style = WIREFRAME; surface->color = $3; }
        ;
 
hues: T_RED { $$ = SHADED_RED; }
      | T_GREEN { $$ = SHADED_GREEN; }
      | T_BLUE { $$ = SHADED_BLUE; }
      | T_GREY { $$ = SHADED_GREY; }
      | T_WHITE { $$ = SHADED_GREY; }
      ;
 
filename: giffile fade { LoadGifBitmap(surface, $1, $2); }
          | T_PXMFILE { LoadPxmBitmap(surface, yytext); }
          | error '\n' { Error("filename expected"); }
          ;
 
vertex: T_VERTEX lbrace fp comma fp comma fp rbrace { AddPoint(surface, MakePoint($3, $5, $7, NULL)); }
        | T_VERTEX error '}' { Error("incorrect vertex definition"); }
        ;
 
vertex_list: vertex vertex_list
             |
             ;
 
lbrace: T_LBRACE
        | error '}' { Error("{ expected"); }
        ;
 
rbrace: T_RBRACE
        | error '\n' { Error("} expected"); }
        ;
 
fp: T_FLOAT { $$ = strtofloat(yytext); }
    | error { Error("float expected"); }
    ;
 
integer: T_INT { $$ = atoi(yytext); }
         | error { Error("integer expected"); }
         ;
 
comma: T_COMMA
       | error { Error(", expected"); }
       ;
%%
#include <stdarg.h>
 
void Error(char *format, ...)
{
  va_list arg;
 
  GraphicsDone();
  printf("ERROR: ");
  va_start(arg, format);
  vprintf(format, arg);
  va_end(arg);
  printf(" on line %d.\n", lineno);
}

#include "lexyy.c"

