// Copyright 1996 by Janne Lf
#include "fbuffer.h"
#include <vga.h>
#include <strings.h>

FrameBuffer::FrameBuffer(int w, int h)
{
	sizex = w;
	sizey = h;
	size  = w*h;
	fbuf  = new unsigned char[size];
	fwd   = new Edge[sizey];
	bwd   = new Edge[sizey];
}
FrameBuffer::~FrameBuffer()
{
	if (fbuf) delete [] fbuf;
	if (fwd ) delete []  fwd;
	if (bwd ) delete []  bwd;
}

void FrameBuffer::WritePixel(const XYPoint &p, int color)
{
	if (p.x >= 0 && p.y >= 0 && p.x < sizex && p.y < sizey)
		fbuf[p.x + p.y * sizex ] = color;
}

void FrameBuffer::Clear(int color)
{
	memset(fbuf, color, size);
}
void FrameBuffer::Show(int x, int y)
{
	int offset = 0;
	while (offset < size) {
		vga_drawscansegment(fbuf+offset, x,y++, sizex);
		offset += sizex;
	}
}

void FrameBuffer::scanedge(Edge *edge, const XYPoint &a, const XYPoint &b)
{
	// clip
	int min_y = a.y >? 0;
	int max_y = b.y <? sizey;
	if (max_y <= min_y) return;
	// initalize
	int dx = ((b.x - a.x)<<16)/(b.y - a.y);
	int  x = (a.x<<16);
	if (a.y  < 0) {
		x -= a.y * dx;
	}
	// interpolate
	do {
		edge[min_y].x = (x>>16); x += dx;
	} while (++min_y < max_y);
}

void FrameBuffer::scanedge(Edge *edge, const XYPoint &a, const XYPoint &b, const XYPoint &ta, const XYPoint &tb)
{
	// clip
	int min_y = a.y >? 0;
	int max_y = b.y <? sizey;
	if (max_y <= min_y) return;
	// initialize
	int  dx = (( b.x -  a.x)<<16)/(b.y - a.y);
	int dtx = ((tb.x - ta.x)<<16)/(b.y - a.y);
	int dty = ((tb.y - ta.y)<<16)/(b.y - a.y);
	int  x = ( a.x<<16);
	int tx = (ta.x<<16);
	int ty = (ta.y<<16);
	if (a.y < 0) {
		x  -= a.y *  dx;
		tx -= a.y * dtx;
		ty -= a.y * dty;
	}
	// interpolate
	do {
		edge[min_y].x  = (x>>16); x +=  dx;
		edge[min_y].tx =  tx; tx += dtx;
		edge[min_y].ty =  ty; ty += dty;
	} while (++min_y < max_y);
}


void FrameBuffer::hline(unsigned char *dst, const Edge &min, const Edge &max, int color)
{
	// clip
	int min_x = min.x >? 0;
	int max_x = max.x <? sizex;
	if (max_x <= min_x) return;
	// draw
	memset(dst + min_x, color, max_x - min_x);
}

void FrameBuffer::hline(unsigned char *dst, const Edge &min, const Edge &max, const unsigned char *bmap)
{
	// clip
	int min_x = min.x >? 0;
	int max_x = max.x <? sizex;
	if (max_x <= min_x) return;
	// initialize
	int dtx = (max.tx - min.tx) / (max.x - min.x);
	int dty = (max.ty - min.ty) / (max.x - min.x);
	register int tx = min.tx;
	register int ty = min.ty;
	if (min.x < 0) {		// if left edge is clipped adjust
		tx -= min.x * dtx;
		ty -= min.x * dty;
	}
	register unsigned char *e = dst + max_x;	// end
	register unsigned char *d = dst + min_x;	// current position
	// draw
	do {
		#if TEXMAP_SIZEX == 64 && TEXMAP_SIZEY == 64
		*d = bmap[ ( (tx>>16) & 63 ) + ( (ty>>10) & (63<<6) ) ];
		#elif TEXMAP_SIZEX == 256 && TEXMAP_SIZEY == 256
		*d = bmap[ ( (tx>>16) & 0xff ) + ( (ty>>8) & 0xff00 ) ];
		#else
		*d = bmap[ ((tx>>16)%TEXMAP_SIZEX) + ((ty>>16)%TEXMAP_SIZEY) * TEXMAP_SIZEX ];
		#endif
		tx += dtx;
		ty += dty;
	} while (++d < e);
}


/*
	draw ambient polygon (polygon must be convex)
		point		list of polygon corner points
		cnt		number of points
		color		polygon color
*/
void FrameBuffer::FillPoly(const XYPoint *point, int cnt, int color)
{
	// #1 edges
	//	scan left edges
	//	scan right edges
	// #2 scanlines
	//	draw lines from left edges to right edges

	register int i;

	if (cnt < 3) return;	// degenerate

	// find top and bottom of the polygon
	int min_y = point[0].y, min_i = 0;
	int max_y = point[0].y, max_i = 0;
	for (i=1; i<cnt; i++) {
		if (point[i].y < min_y) min_y = point[min_i = i].y;
		if (point[i].y > max_y) max_y = point[max_i = i].y;
	}
	if (min_y == max_y) return;	// degenerate

	// scan forward
	i = min_i;
	while (i != max_i) {
		int n = (i+1)%cnt;
		scanedge(fwd, point[i], point[n]);
		i = n;
	}
	// scan backward
	i = min_i;
	while (i != max_i) {
		int n = (i-1+cnt)%cnt;
		scanedge(bwd, point[i],point[n]);
		i = n;
	}

	int y = 0 >? min_y;
	int offset = y*sizex;
	while (y < max_y && y < sizey) {
		if (bwd[y].x < fwd[y].x)
			hline(fbuf + offset, bwd[y], fwd[y], color);
		else
			hline(fbuf + offset, fwd[y], bwd[y], color);
		y++; offset += sizex;
	}
}


/*
	draw textured polygon (polygon must be convex)
		point		list of polygon corner points
		tex_point	list of texture points
		cnt		number of points
		bmap		texture map data (64x64)
*/
void FrameBuffer::FillPoly(const XYPoint *point, const XYPoint *tex_point, int cnt, const unsigned char *bmap)
{
	// #1 edges
	//	scan left edges
	//	scan right edges
	// #2 scanlines
	//	draw lines from left edges to right edges

	register int i;

	if (cnt < 3) return;	// degenerate

	// find top and bottom of the polygon
	int min_y = point[0].y, min_i = 0;
	int max_y = point[0].y, max_i = 0;
	for (i=1; i<cnt; i++) {
		if (point[i].y < min_y) min_y = point[min_i = i].y;
		if (point[i].y > max_y) max_y = point[max_i = i].y;
	}
	if (min_y == max_y) return;	// degenerate

	// scan forward
	i = min_i;
	while (i != max_i) {
		int n = (i+1)%cnt;
		scanedge(fwd, point[i], point[n], tex_point[i], tex_point[n]);
		i = n;
	}
	// scan backward
	i = min_i;
	while (i != max_i) {
		int n = (i-1+cnt)%cnt;
		scanedge(bwd, point[i], point[n], tex_point[i], tex_point[n]);
		i = n;
	}

	int y = 0 >? min_y;
	int offset = y*sizex;
	while (y < max_y && y < sizey) {
		if (bwd[y].x < fwd[y].x)
			hline(fbuf + offset, bwd[y], fwd[y], bmap);
		else
			hline(fbuf + offset, fwd[y], bwd[y], bmap);
		y++; offset += sizex;
	}
}
