/****************************************************************
** UCF.C
**
** Universal Configuration File (UCF) Library Version 0.2 source file
**
**  Copyright (C) 1999 Blint Balogh <balintb@iname.com>
**
**  This library is free software; you can redistribute it and/or modify it
**  under the terms of the GNU Library General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or (at
**  your option) any later version.
**
**  This library is distributed in the hope that it will be useful, but
**  WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
**  Library General Public License for more details.
**
**  You should have received a copy of the GNU Library General Public License
**  along with this library (the file "COPYING.LIB"); if not, write to the
**  Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
**----------------------------------------------------------------
**
** Created by Balint, 12-13 Feb 1999, version 0.1
** Modified by Balint, 14 Feb 1999
**          - bugfix: _AdjustParents() added
**          - some other minor bugfixes
** Modified by Balint, 19 Feb 1999, version 0.2
**          - major change in the structure: ID's can only be strings
**          - reorganization of the way of handling dynamically allocated ID's
**          - major bugfix: the whole part of handling numbers in _GetToken()
**            is rewritten
**          - bugfix: handling hex and oct chars in strings in _GetToken()
**/

#include <stdlib.h>
#include <string.h>
#include "ucf.h"

/* If SAFE_REALLOC==1, ChangeUCFVariable(), UCFCreateBlock() and
 * UCFSelectBlockForce() won't delete the fields of the current context even
 * if there's no more memory available but they need more memory to have two
 * buffers at the same time.
 */
#define SAFE_REALLOC    1

#define MAXFIELDNUMDELTA 8
#define FIELDNUMINC 8
#define FIELDNUMSTART 4

#define STRINGLENINC 16
#define STRINGLENSTART 8

#define MAXSTACKDELTA 32
#define STACKLENINC 32
#define STACKLENSTART 16
#define STACKLENPLUS 4

#if STACKLENPLUS>=MAXSTACKDELTA
    #error "STACKLENPLUS >= MAXSTACKDELTA"
#endif

/****************************************************************/
/* Clears everything */
void InitUCFContext( UCFContext *C )
{
    if ( C==NULL ) return;
    C->Vars.FieldNum=C->Vars.ArraySize=0;
    C->Vars.Fields=NULL;
    C->Vars.Parent=NULL;
    C->Current=&C->Vars;
}

/*--------------------------------------------------------------*/
/* Deletes one field's content
 * Handles blocks and strings correctly.
 */
static int _DeleteVariable( UCFVariable *V )
{
    int i,isOK;

    if ( V->ID ) free( V->ID );
    if ( V->V.Type==UCFTstring )
    {
        if ( V->V.v.stringv ) free( V->V.v.stringv );
        return 1;
    }
    if ( V->V.Type!=UCFTblock ) return 1;

    isOK=((V->V.v.blockv.ArraySize==0) ^ (V->V.v.blockv.Fields!=NULL)) &&
        (V->V.v.blockv.ArraySize>=V->V.v.blockv.FieldNum);

    for ( i=0; i<V->V.v.blockv.FieldNum; i++ )
        if ( !_DeleteVariable( &V->V.v.blockv.Fields[i] ) ) isOK=0;
    free( V->V.v.blockv.Fields );
    return isOK;
}

/****************************************************************/
/* Deletes whole context
 * Returns: 1: OK
 *          0: error
 */
int QuitUCFContext( UCFContext *C )
{
    int i,isOK;

    if ( C==NULL ) return 0;
    isOK=((C->Vars.ArraySize==0) ^ (C->Vars.Fields!=NULL)) &&
        (C->Vars.ArraySize>=C->Vars.FieldNum);

    if ( C->Vars.Fields )
    {
        for ( i=0; i<C->Vars.FieldNum; i++ )
            if ( !_DeleteVariable( &C->Vars.Fields[i] ) ) isOK=0;
        free( C->Vars.Fields );
    }
    InitUCFContext( C );

    return isOK;
}

/****************************************************************/
/* Finds variable in current block
 *  In: UCFContext
 *  Out:UCFContext unmodified
 *      return value is 1 if variable found, 0 if not
 *      value in V (if found, unmodified if not found)
 */
int GetUCFVariable( UCFContext *C, const char *ID, UCFValue *V )
{
    int i;

    if ( (C==NULL) || (C->Current==NULL) || (ID==NULL) || (V==NULL) )
        return 0;
    for ( i=0; i<C->Current->FieldNum; i++ )
    {
        if ( strcmp( C->Current->Fields[i].ID, ID )==0 )
        {
            *V=C->Current->Fields[i].V;
            return 1;
        }
    }
    return 0;
}

/*--------------------------------------------------------------*/
static void _AdjustParents( UCFBlock *B )
{
    int i,j;
    UCFBlock *C;

    for ( i=0; i<B->FieldNum; i++ )
        if ( B->Fields[i].V.Type==UCFTblock )
        {
            C=&B->Fields[i].V.v.blockv;
            for ( j=0; j<C->FieldNum; j++ )
                if ( C->Fields[j].V.Type==UCFTblock )
                    C->Fields[j].V.v.blockv.Parent=C;
        }
}

/*--------------------------------------------------------------*/
#define _MAKEIDCOPY(ID,STR) {                                           \
                                if ( ((ID)=malloc(strlen(STR)+1))       \
                                    ==NULL )                            \
                                    return 0;                           \
                                strcpy( (ID), STR );                    \
                            }

/****************************************************************/
/* Changes a variable in the current context or adds it if not defined
 * already, i.e. its type is "empty".
 * Both C and V remain unmodified.
 * Return 1 if OK, 0 on error;
 * Variables can be deleted in two ways:
 *  1. make V=NULL
 *  2. make V.Type=UCFTempty
 * This function should be used also if an empty block is needed to be
 * created.
 */
int ChangeUCFVariable( UCFContext *C, const char *ID, UCFValue *V )
{
    int i;

    if ( (C==NULL) || (C->Current==NULL) || (ID==NULL) ) return 0;
    for ( i=0; i<C->Current->FieldNum; i++ )
    {
        if ( strcmp( C->Current->Fields[i].ID, ID )==0 )
        {
            _DeleteVariable( &C->Current->Fields[i] );
            if ( V && (V->Type!=UCFTempty) )
            {
                _MAKEIDCOPY( C->Current->Fields[i].ID, ID )
                C->Current->Fields[i].V=*V;
            }
            else
            {
            /* Deleting a variable */
                if (C->Current->FieldNum>1)
                {
                    if (i!=C->Current->FieldNum-1)
                        C->Current->Fields[i]=
                            C->Current->Fields[--C->Current->FieldNum];
                /* Decreasing array size */
                    if ( C->Current->ArraySize-C->Current->FieldNum>
                        MAXFIELDNUMDELTA )
                    {
                #if SAFE_REALLOC
                        UCFVariable *F;

                        if ( (F=malloc( sizeof(UCFVariable)*
                            C->Current->FieldNum ))==NULL )
                            return 0;
                        memcpy( F, C->Current->Fields,
                            sizeof(UCFVariable)*C->Current->FieldNum );
                        free( C->Current->Fields );
                        C->Current->Fields=F;
                #else
                        if ( (C->Current->Fields=realloc( C->Current->Fields,
                            sizeof(UCFVariable)*C->Current->FieldNum ))
                            ==NULL )
                        {
                        /* Unfortunately all fields have been deleted */
                            C->Current->ArraySize=C->Current->FieldNum=0;
                            return 0;
                        }
                #endif
                        C->Current->ArraySize=C->Current->FieldNum;
                        _AdjustParents( C->Current );
                    }
                }
                else
                {
                /* Deleting the whole field array */
                    C->Current->ArraySize=C->Current->FieldNum=0;
                    free( C->Current->Fields );
                    C->Current->Fields=NULL;
                }
            }
            return 1;
        }
    }
    /* Adding a new variable */
    if ( C->Current->ArraySize==0 )
    {
        if ( (C->Current->Fields=
            malloc( sizeof(UCFVariable)*FIELDNUMSTART ))==NULL )
            return 0;
        C->Current->ArraySize=FIELDNUMSTART;
/*      C->Current->FieldNum=0;   Not needed */
    }
    else if ( !(C->Current->ArraySize>C->Current->FieldNum) )
    {
#if SAFE_REALLOC
        UCFVariable *F;

        C->Current->ArraySize+=FIELDNUMINC;
        if ( (F=malloc( sizeof(UCFVariable)*C->Current->ArraySize ))==NULL )
        {
            C->Current->ArraySize-=FIELDNUMINC;
            return 0;
        }
        memcpy( F, C->Current->Fields,
            sizeof(UCFVariable)*C->Current->FieldNum );
        free( C->Current->Fields );
        C->Current->Fields=F;
#else
        C->Current->ArraySize+=FIELDNUMINC;
        if ( (C->Current->Fields=realloc( C->Current->Fields,
            sizeof(UCFVariable)*C->Current->ArraySize ))==NULL )
        {
        /* Unfortunately all fields have been deleted */
            C->Current->ArraySize=C->Current->FieldNum=0;
            return 0;
        }
#endif
        _AdjustParents( C->Current );
    }
    _MAKEIDCOPY( C->Current->Fields[C->Current->FieldNum].ID, ID )
    C->Current->Fields[C->Current->FieldNum++].V=*V;
    return 1;
}

/****************************************************************/
int ConvertUCFValueToString( UCFValue *V )
{
    char S[33];
    int n,i;

    if (V==NULL) return 0;
    switch ( V->Type )
    {
        case UCFTblock:
            for ( i=0; i<V->v.blockv.FieldNum; i++ )
                _DeleteVariable( &V->v.blockv.Fields[i] );
            free( V->v.blockv.Fields );
            V->Type=UCFTempty;

            if ( (V->v.stringv=malloc( 8 ))==NULL ) return 0;
            strcpy( V->v.stringv, "<BLOCK>" );
            break;

        case UCFTstring:
            return 1;

        case UCFTempty:
            if ( (V->v.stringv=malloc( 8 ))==NULL ) return 0;
            strcpy( V->v.stringv, "<EMPTY>" );
            break;

        case UCFTint:
        case UCFTfloat:
            if ( V->Type==UCFTint )
                n=sprintf( S, "%i", V->v.intv );
            else
                n=sprintf( S, "%g", V->v.floatv );
            V->Type=UCFTempty;
            if ( (V->v.stringv=malloc( n+1 ))==NULL ) return 0;
            strcpy( V->v.stringv, S );
            break;

    }
    V->Type=UCFTstring;
    return 1;
}

/****************************************************************/
/* Creates a new block in the current block and sets current=new block
 * If a field of the same ID is already defined then it will be deleted first.
 * Returns:
 *  1: OK
 *  0: failed
 */
int UCFCreateBlock( UCFContext *C, const char *ID )
{
    int i;

    if ( (C==NULL) || (C->Current==NULL) || (ID==NULL) ) return 0;
    for ( i=0; i<C->Current->FieldNum; i++ )
    {
        if ( strcmp( C->Current->Fields[i].ID, ID )==0 )
        {
            _DeleteVariable( &C->Current->Fields[i] );
            _MAKEIDCOPY( C->Current->Fields[i].ID, ID )
            C->Current->Fields[i].V.Type=UCFTblock;
            C->Current->Fields[i].V.v.blockv.Fields=NULL;
            C->Current->Fields[i].V.v.blockv.FieldNum=
                C->Current->Fields[i].V.v.blockv.ArraySize=0;
            C->Current->Fields[i].V.v.blockv.Parent=C->Current;
            C->Current=&C->Current->Fields[i].V.v.blockv;
            return 1;
        }
    }
    /* Adding a new variable */
    if ( C->Current->ArraySize==0 )
    {
        if ( (C->Current->Fields=
            malloc( sizeof(UCFVariable)*FIELDNUMSTART ))==NULL )
            return 0;
        C->Current->ArraySize=FIELDNUMSTART;
/*      C->Current->FieldNum=0;   Not needed */
    }
    else if ( !(C->Current->ArraySize>C->Current->FieldNum) )
    {
#if SAFE_REALLOC
        UCFVariable *F;

        C->Current->ArraySize+=FIELDNUMINC;
        if ( (F=malloc( sizeof(UCFVariable)*C->Current->ArraySize ))==NULL )
        {
            C->Current->ArraySize-=FIELDNUMINC;
            return 0;
        }
        memcpy( F, C->Current->Fields,
            sizeof(UCFVariable)*C->Current->FieldNum );
        free( C->Current->Fields );
        C->Current->Fields=F;
#else
        C->Current->ArraySize+=FIELDNUMINC;
        if ( (C->Current->Fields=realloc( C->Current->Fields,
            sizeof(UCFVariable)*C->Current->ArraySize ))==NULL )
        {
        /* Unfortunately all fields have been deleted */
            C->Current->ArraySize=C->Current->FieldNum=0;
            return 0;
        }
#endif
        _AdjustParents( C->Current );
    }
    _MAKEIDCOPY( C->Current->Fields[C->Current->FieldNum].ID, ID )
    C->Current->Fields[C->Current->FieldNum].V.Type=UCFTblock;
    C->Current->Fields[C->Current->FieldNum].V.v.blockv.Fields=NULL;
    C->Current->Fields[C->Current->FieldNum].V.v.blockv.FieldNum=
        C->Current->Fields[C->Current->FieldNum].V.v.blockv.ArraySize=0;
    C->Current->Fields[C->Current->FieldNum].V.v.blockv.Parent=C->Current;
    C->Current=&C->Current->Fields[C->Current->FieldNum++].V.v.blockv;
    return 1;
}

/****************************************************************/
/* Selects block of *ID, if founds it or creates a new block and selects it
 * if a variable of the same ID is not defined already.
 */
int UCFSelectBlock( UCFContext *C, const char *ID )
{
    int i;

    if ( (C==NULL) || (C->Current==NULL) || (ID==NULL) ) return 0;
    for ( i=0; i<C->Current->FieldNum; i++ )
    {
        if ( strcmp( C->Current->Fields[i].ID, ID )==0 )
        {
            if ( C->Current->Fields[i].V.Type!=UCFTblock ) return 0;
            C->Current=&C->Current->Fields[i].V.v.blockv;
            return 1;
        }
    }
    return UCFCreateBlock( C, ID );
}

/****************************************************************/
/* Selects block of *ID, if founds it or creates a new block and selects it
 */
int UCFSelectBlockForce( UCFContext *C, const char *ID )
{
    int i;

    if ( (C==NULL) || (C->Current==NULL) || (ID==NULL) ) return 0;
    for ( i=0; i<C->Current->FieldNum; i++ )
    {
        if ( strcmp( C->Current->Fields[i].ID, ID )==0 )
        {
            if ( C->Current->Fields[i].V.Type!=UCFTblock )
            {
                _DeleteVariable( &C->Current->Fields[i] );
                _MAKEIDCOPY( C->Current->Fields[i].ID, ID )
                C->Current->Fields[i].V.Type=UCFTblock;
                C->Current->Fields[i].V.v.blockv.Fields=NULL;
                C->Current->Fields[i].V.v.blockv.FieldNum=
                    C->Current->Fields[i].V.v.blockv.ArraySize=0;
                C->Current->Fields[i].V.v.blockv.Parent=C->Current;
            }
            C->Current=&C->Current->Fields[i].V.v.blockv;
            return 1;
        }
    }

    /* Adding a new variable */
    if ( C->Current->ArraySize==0 )
    {
        if ( (C->Current->Fields=
            malloc( sizeof(UCFVariable)*FIELDNUMSTART ))==NULL )
            return 0;
        C->Current->ArraySize=FIELDNUMSTART;
/*      C->Current->FieldNum=0;   Not needed */
    }
    else if ( !(C->Current->ArraySize>C->Current->FieldNum) )
    {
#if SAFE_REALLOC
        UCFVariable *F;

        C->Current->ArraySize+=FIELDNUMINC;
        if ( (F=malloc( sizeof(UCFVariable)*C->Current->ArraySize ))==NULL )
        {
            C->Current->ArraySize-=FIELDNUMINC;
            return 0;
        }
        memcpy( F, C->Current->Fields,
            sizeof(UCFVariable)*C->Current->FieldNum );
        free( C->Current->Fields );
        C->Current->Fields=F;
#else
        C->Current->ArraySize+=FIELDNUMINC;
        if ( (C->Current->Fields=realloc( C->Current->Fields,
            sizeof(UCFVariable)*C->Current->ArraySize ))==NULL )
        {
        /* Unfortunately all fields have been deleted */
            C->Current->ArraySize=C->Current->FieldNum=0;
            return 0;
        }
#endif
        _AdjustParents( C->Current );
    }
    _MAKEIDCOPY( C->Current->Fields[C->Current->FieldNum].ID, ID )
    C->Current->Fields[C->Current->FieldNum].V.Type=UCFTblock;
    C->Current->Fields[C->Current->FieldNum].V.v.blockv.Fields=NULL;
    C->Current->Fields[C->Current->FieldNum].V.v.blockv.FieldNum=
        C->Current->Fields[C->Current->FieldNum].V.v.blockv.ArraySize=0;
    C->Current->Fields[C->Current->FieldNum].V.v.blockv.Parent=C->Current;
    C->Current=&C->Current->Fields[C->Current->FieldNum++].V.v.blockv;
    return 1;
}

/****************************************************************/
int UCFSelectParent( UCFContext *C )
{
    if ( (C==NULL) || (C->Current==NULL) ) return 0;
    if ( C->Current->Parent )
    {
        C->Current=C->Current->Parent;
        return 1;
    }
    else
        return 0;
}

/****************************************************************/
int UCFSelectRoot( UCFContext *C)
{
    if ( C==NULL ) return 0;
    C->Current=&C->Vars;
    return 1;
}

/****************************************************************/
unsigned int UCFNumberOfFields( UCFContext *C )
{
    if ( C && C->Current )
        return C->Current->FieldNum;
    else
        return 0;
}

/****************************************************************/
int InitFindUCFVariable( UCFContext *C, UCFFNContext *N )
{
    if ( (C==NULL) || (N==NULL) || ((N->Block=C->Current)==NULL) )
        return 0;
    N->Index=0;
    return 1;
}

/****************************************************************/
int FindNextUCFVariable( UCFContext *C, UCFFNContext *N, char **ID,
    UCFValue *V )
{
    if ( (C==NULL) || (N==NULL) || (N->Block==NULL) ) return 0;
    if ( !(N->Index<N->Block->FieldNum) )
    {
        N->Block=NULL;
        return 0;
    }
    if ( ID ) *ID=N->Block->Fields[N->Index].ID;
    if ( V ) *V=N->Block->Fields[N->Index++].V;
    return 1;
}

/*--------------------------------------------------------------*/
static int _StrAddCh( char **S, unsigned int *n, unsigned int *len, char c )
{
    if ( *S )
    {
        if ( !(*n>*len) )
            if ( (*S=realloc( *S, (*n+=STRINGLENINC)+1 ))==NULL ) return 0;
    }
    else
    {
        if ( (*S=malloc( (*n=STRINGLENSTART)+1 ))==NULL ) return 0;
    }
    (*S)[(*len)++]=c;
    return 1;
}

/*::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
enum
{
    tkt_empty,
    tkt_str,
    tkt_idstr,
    tkt_int,
    tkt_float,
    tkt_spec
};

typedef struct
{
    int Type;
    union
    {
        char *stringv;
        int intv;
        double floatv;
        int specv;
    } v;
} Token;

/*--------------------------------------------------------------*/
/* Returns next token in *tk
 * Return value:
 *  1: OK
 *  0: Error
 */
static int _GetToken( FILE *F, Token *tk )
{
    char *S;
    unsigned int n,len,state,comm,nline,base,wasp;
    int ch,num,dnum;

#define __SAC(c)    {                                               \
                        if ( !_StrAddCh( &S, &n, &len, (c) ) )      \
                            return 0;                               \
                    }

    enum
    {
        s_normal,
        s_idstring,
        s_num,
        s_string,
        s_command,
        s_genc,
        s_linec,
        s_normc,
        s_nestc
    };

    S=NULL;
    n=len=0;
    state=s_normal;
    ch=getc( F );
    for (;;)
    {
        if ( (state!=s_string) && (ch=='\\') )
        {
            if ( (ch=getc( F ))=='\n' )
            {
                ch=getc( F );
                continue;
            }
            else
            {
                if ( S ) free( S );
                return 0;
            }
        }
        switch ( state )
        {
            case s_normal:
                if ( ch=='`' )      /* comment */
                {
                    state=s_genc;
                }
                else if ( ch=='$' )
                {
                    state=s_command;
                    ch=getc( F );
                    if ( ch=='{' )
                    {
                        comm=1;
                        nline=0;
                    }
                    else
                    {
                        comm=0;
                        continue;
                    }
                }
                else if ( ch=='\"' )
                {
                    state=s_string;
                }
                else if ( ((ch>='A') && (ch<='Z')) || (ch=='_') )
                {
                    __SAC( ch )
                    state=s_idstring;
                }
                else if ( (ch>='a') && (ch<='z') )
                {
                    __SAC( ch-'a'+'A' )
                    state=s_idstring;
                }
                else if ( (ch>='0') && (ch<='9') )
                {
                    __SAC( ch )
                    wasp=0;
                    state=s_num;
                }
                else if ( (ch==' ') || (ch=='\r') || (ch=='\t') || (ch=='\f')
                    || (ch==0) )
                {
                }
                else
                {
                    if ( S ) free( S );
                    tk->Type=tkt_spec;
                    tk->v.specv=ch;
                    return 1;
                }
                break;      /* case s_normal */

            case s_genc:
                if ( ch=='<' )
                {
                    state=s_normc;
                    nline=0;
                }
                else if ( ch=='{' )
                {
                    state=s_nestc;
                    comm=1;
                    nline=0;
                }
                else if ( ch==EOF )
                {
                    if ( S ) free( S );
                    tk->Type=tkt_spec;
                    tk->v.specv=EOF;
                    return 1;
                }
                else
                {
                    state=s_linec;
                }
                break;      /* case s_genc */

            case s_linec:
                if ( ch=='\n' )
                {
                    state=s_normal;
                    continue;
                }
                else if ( ch==EOF )
                {
                    if ( S ) free( S );
                    tk->Type=tkt_spec;
                    tk->v.specv=EOF;
                    return 1;
                }
                break;      /* case s_linec */

            case s_normc:
                if ( ch=='>' )
                {
                    ch=getc( F );
                    if ( ch=='`' )
                    {
                        state=s_normal;
                        if ( nline )
                            ungetc( '\n', F );
                    }
                }
                else if ( ch=='\n' )
                {
                    nline=1;
                }
                else if ( ch==EOF )
                {
                    if ( S ) free( S );
                    tk->Type=tkt_spec;
                    tk->v.specv=EOF;
                    return 1;
                }
                break;      /* case s_normc */

            case s_nestc:
                if ( ch=='}' )
                {
                    if ( (ch=getc( F ))=='`' )
                        if (--comm==0)
                        {
                            state=s_normal;
                            if ( nline )
                                ungetc( '\n', F );
                        }
                }
                else if ( ch=='`' )
                {
                    if ( (ch=getc( F ))=='{' )
                        comm++;
                }
                else if ( ch=='\n' )
                {
                    nline=1;
                }
                else if ( ch==EOF )
                {
                    if ( S ) free( S );
                    tk->Type=tkt_spec;
                    tk->v.specv=EOF;
                    return 1;
                }
                break;      /* case s_nestc */

            case s_command:
                if ( ch==EOF )
                {
                    if ( S ) free( S );
                    tk->Type=tkt_spec;
                    tk->v.specv=EOF;
                    return 1;
                }
                else if ( comm==0 )
                {
                    if ( (ch==' ') || (ch=='\r') || (ch=='\t') || (ch=='\f')
                        || (ch==0) )
                    {
                        state=s_normal;
                        if ( nline )
                            ungetc( '\n', F );
                    }
                    else if ( ch=='\n' )
                    {
                        state=s_normal;
                        continue;
                    }
                }
                else
                {
                    if ( ch=='$' )
                    {
                        ch=getc( F );
                        if ( ch=='}' )
                            comm--;
                        else if ( ch=='{' )
                            comm++;
                    }
                    else if ( ch=='\n' )
                    {
                        nline=1;
                    }
                }
                break;      /* case s_command */

            case s_string:
                if ( ch=='\\' )
                {
                    ch=getc( F );
                    switch ( ch )
                    {
                        case '\n':
                            ch=getc( F );
                            continue;
                        case 'a':
                            ch=7;
                            break;
                        case 'b':
                            ch=8;
                            break;
                        case 'f':
                            ch=12;
                            break;
                        case 'n':
                            ch=10;
                            break;
                        case 'r':
                            ch=13;
                            break;
                        case 't':
                            ch=9;
                            break;
                        case 'v':
                            ch=11;
                            break;
                        case 'x':
                            dnum=0;
                            num=0;
                            for (;;)
                            {
                                ch=getc( F );
                                if ( dnum==2 ) break;
                                if ( (ch>='0') && (ch<='9') )
                                {
                                    num<<=4;
                                    num+=ch-'0';
                                }
                                else if ( (ch>='A') && (ch<='F') )
                                {
                                    num<<=4;
                                    num+=ch-'A'+10;
                                }
                                else if ( (ch>='a') && (ch<='f') )
                                {
                                    num<<=4;
                                    num+=ch-'a'+10;
                                }
                                else break;
                                dnum++;
                            }
                            if ( num==0 )
                            {
                                if ( S ) free( S );
                                return 0;
                            }
                            __SAC( num )
                            continue;       /* - case 'x' */
                        case EOF:
                            ungetc( EOF, F );
                            if ( S ) free( S );
                            return 0;
                        default:
                            if ( ( ch>='0' ) && ( ch<='7' ) )
                            {
                                dnum=1;
                                num=ch-'0';
                                for (;;)
                                {
                                    ch=getc( F );
                                    if ( dnum==3 ) break;
                                    if ( ( ch>='0' ) && ( ch<='7' ) )
                                    {
                                        num<<=3;
                                        num+=ch-'0';
                                    }
                                    else break;
                                    dnum++;
                                }
                                if ( num==0 )
                                {
                                    if ( S ) free( S );
                                    return 0;
                                }
                                __SAC( num )
                                continue;
                            }
                            break;      /* - default */
                    }       /* - switch ( ch ) */
                    __SAC( ch )
                }   /* if ( ch=='\\' ) */
                else if ( ch=='\"' )
                {
                    tk->Type=tkt_str;
                    S[len]=0;
                    tk->v.stringv=S;
                    return 1;
                }
                else if ( (ch==EOF) || (ch=='\n') )
                {
                    ungetc( ch, F );
                    if ( S ) free( S );
                    return 0;
                }
                else
                {
                    __SAC( ch )
                }
                break;      /* case s_string */

            case s_idstring:
                if ( ((ch>='A') && (ch<='Z')) || ((ch>='0') && (ch<='9'))
                    || (ch=='_') )
                {
                    __SAC( ch )
                }
                else if ( (ch>='a') && (ch<='z') )
                {
                    __SAC( ch-'a'+'A' )
                }
                else
                {
                    ungetc( ch, F );
                    tk->Type=tkt_idstr;
                    S[len]=0;
                    tk->v.stringv=S;
                    return 1;
                }
                break;      /* case s_idstring */

            case s_num:
                if ( ((ch>='0') && (ch<='9')) || ((ch>='A') && (ch<='Z')) ||
                    ((ch>='a') && (ch<='z')) || (ch=='_') || (ch=='.') )
                {
                    if ( (ch=='e') || (ch=='E') )
                    {
                        ch='e';
                        wasp=1;
                    }
                    else
                        wasp=0;
                    __SAC( ch )
                }
                else if ( ((ch=='+') || (ch=='-')) && (wasp || (len==0)) )
                {
                    __SAC( ch );
                    wasp=0;
                }
                else
                {
                    char *P;

                    ungetc( ch, F );

                    S[len]=0;
                    base=10;
                    wasp=0;
                    if (S[0]=='0')
                        switch ( S[1] )
                        {
                            case 'b':
                            case 'B':
                                base=2;
                                wasp=1;
                                S[1]='0';
                                break;
                            case 'o':
                            case 'O':
                                base=8;
                                wasp=1;
                                S[1]='0';
                                break;
                            case 'd':
                            case 'D':
                                wasp=1;
                                S[1]='0';
                                break;
                            case 'x':
                            case 'X':
                                base=16;
                                wasp=1;
                                S[1]='0';
                                break;
                        }
                    if ( !wasp )
                        switch ( S[len-1] )
                        {
                            case 'b':
                            case 'B':
                                base=2;
                                S[len-1]=0;
                                break;
                            case 'o':
                            case 'O':
                                base=8;
                                S[len-1]=0;
                                break;
                            case 'd':
                            case 'D':
                                S[len-1]=0;
                                break;
                            case 'h':
                            case 'H':
                                base=16;
                                S[len-1]=0;
                                break;
                        }

                    if ( (base==10) && ( (strchr( S, '.' )!=NULL) ||
                        (strchr( S, 'e' )!=NULL) ) )
                    {
                        tk->v.floatv=strtod( S, &P );
                        tk->Type=tkt_float;
                    }
                    else
                    {
                        tk->v.intv=strtol( S, &P, base );
                        tk->Type=tkt_int;
                    }

                    if ( *P )
                    {
                        free( S );
                        return 0;
                    }
                    return 1;
                }
                break;      /* case s_num */

        }   /* switch ( state ) */
        ch=getc( F );
    }   /* for (;;) */

#undef __SAC

}   /* _GetToken() */

/*::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
typedef struct
{
    int State;
    UCFBlock *Current;
} StackType;

/*--------------------------------------------------------------*/
static int _Push( StackType **stack, unsigned int *size, unsigned int *len,
    StackType *v )
{
    if ( *stack )
    {
        if ( !(*size>*len) )
            if ( (*stack=realloc( *stack,
                sizeof(StackType)*(*size+=STACKLENINC) ))==NULL )
                return 0;
    }
    else
    {
        if ( (*stack=malloc( sizeof(StackType)*(*size=STACKLENSTART) ))
            ==NULL )
            return 0;
    }
    (*stack)[(*len)++]=*v;
    return 1;
}

/*--------------------------------------------------------------*/
static int _Pop( StackType **stack, unsigned int *size, unsigned int *len,
    StackType *v )
{
    if ( len )
    {
        *v=(*stack)[--(*len)];
        if ( *size-*len>MAXSTACKDELTA )
        {
            if ( len )
            {
                if ( (*stack=realloc( *stack,
                    sizeof(StackType)*( (*size=*len)+STACKLENPLUS ) ))==NULL )
                    return 0;
            }
            else
            {
                free( *stack );
                *stack=NULL;
                *size=0;
            }
        }
        return 1;
    }
    else
    {
        if ( *stack ) free( *stack );
        return 0;
    }
}

/****************************************************************/
/* UCFReadConfigFile(): Reads configuration file and adds new variables
 * Out: 1 -> succeeded
 *      0 -> failed
 *
 * Syntax:
 *
 * def={ ID [ . ID [...] ]
 *          {
 *                { '='
 *                  { ['#'] VAL [ ['#'] VAL [...] ] } |
 *                  { '.' [ num ]
 *                    '{' valdef [ { ';' | ',' | '\n' } valdef [...] ] '}' }
 *                } |
 *                { '.' '{' [ def [ { ';' | '\n' } def [...] ] '}' }
 *          }
 *  }
 *
 * var.var=#val val
 * var.var=val #val
 * var.var=. [<NL>] { #val val, #val val
 *              val #val }
 * var.var=. [<NL>] { val #val }
 * var.var=.3 [<NL>] { #val val }
 * var.var. [<NL>] {
 *      var.var=val #val
 * }
 *
 * variablename=value
 * variablename:
 *  "string"
 *  233 0d233 233d 0xE9 0E9h 0b11101001 11101001b 0o351 351o 0233
 *  4566.32 4.56632e3
 *  -4
 *  specstring:  A..Z,a..z,_,0..9
 *
 * Or:
 * var=val1 val2 #val3 val4
 *
 * All specstrings and constants are upcase'd
 *
 * Predefined constants:
 *  true,yes,on,enabled=1
 *  false,no,off,disabled=0
 *
 * Special commands: (not ipmlemented)
 *  $xxxxx ${xxx  ... xxxx .... $}xxx
 *
 * Comments:
 * ` until the end of the current line
 * `<
 * >`
 * `{ `{ nested }` }`
 *
 */

int UCFReadConfigFile( UCFContext *C, FILE *F )
{
    UCFBlock *Root;
    unsigned int stacksize=0,stacklen=0;
    StackType st, *stack=NULL;
    int state,valid,wasvalid,arrindex;
    Token tk;
    char *ID;
    UCFValue V;

    typedef struct
    {
        char *Str;
        int Num;
    } ConstantType;

    static ConstantType Constants[]=
    {
        { "TRUE", 1 },
        { "FALSE", 0 },
        { "YES", 1 },
        { "NO", 0 },
        { "ON", 1 },
        { "OFF", 0 },
        { "ENABLE", 1 },
        { "DISABLE", 0 },
        { "ENABLED", 1 },
        { "DISABLED", 0 }
    };

#define CONSTANT_NUM 10

#define __ERR       {                                                   \
                        goto __label_error;                             \
                    }

#define __PUSH(v)   {                                                   \
                        if ( !_Push(&stack,&stacksize,&stacklen,&(v)) ) \
                            __ERR                                       \
                    }

#define __POP(v)    {                                                   \
                        if ( !_Pop(&stack,&stacksize,&stacklen,&(v)) )  \
                            __ERR                                       \
                    }

#define __PUSHC     {                                                   \
                        st.State=state;                                 \
                        st.Current=C->Current;                          \
                        __PUSH( st )                                    \
                    }

#define __POPC      {                                                   \
                        __POP( st )                                     \
                        state=st.State;                                 \
                        C->Current=st.Current;                          \
                    }

#define __SETID     {                                                   \
                        ID=tk.v.stringv;                                \
                        tk.v.stringv=NULL;                              \
                    }

#define __SETV      {                                                   \
                        if ( tk.Type==tkt_idstr )                       \
                        {                                               \
                            int i;                                      \
                            for ( i=0; i<CONSTANT_NUM; i++)             \
                            {                                           \
                                if ( strcmp( Constants[i].Str,          \
                                    tk.v.stringv )==0 )                 \
                                {                                       \
                                    free( tk.v.stringv );               \
                                    tk.Type=tkt_int;                    \
                                    tk.v.intv=Constants[i].Num;         \
                                    break;                              \
                                }                                       \
                            }                                           \
                        }                                               \
                        switch ( tk.Type )                              \
                        {                                               \
                            case tkt_str:                               \
                            case tkt_idstr:                             \
                                V.Type=UCFTstring;                      \
                                V.v.stringv=tk.v.stringv;               \
                                tk.v.stringv=NULL;                      \
                                break;                                  \
                            case tkt_int:                               \
                                V.Type=UCFTint;                         \
                                V.v.intv=tk.v.intv;                     \
                                break;                                  \
                            case tkt_float:                             \
                                V.Type=UCFTfloat;                       \
                                V.v.floatv=tk.v.floatv;                 \
                                break;                                  \
                        }                                               \
                    }

    enum
    {
        s_norm,
        s_id,
        s_eq,
        s_val,
        s_arr
    };

    if ( (C==NULL) || (C->Current==NULL) || (F==NULL) ) return 0;

    Root=C->Current;
    state=s_norm;
    ID=NULL;

    for (;;)
    {
        if ( !_GetToken( F, &tk ) )
            __ERR
        switch ( state )
        {
            case s_norm:
                if ( (tk.Type==tkt_idstr) || (tk.Type==tkt_str) )
                {
                    __PUSHC
                    __SETID
                    state=s_id;
                }
                else if ( tk.Type==tkt_spec )
                {
                    if ( (tk.v.specv=='}') && (C->Current!=Root) )
                    {
                        __POPC
                    }
                    else if ( (tk.v.specv=='\n') || (tk.v.specv==';') )
                    {
                    }
                    else if ( tk.v.specv==EOF )
                    {
                        if ( stack ) free( stack );
                        C->Current=Root;
                        return 1;
                    }
                    else
                        __ERR
                }
                else
                    __ERR
                break;      /* case s_norm */

            case s_id:
                if ( ( (tk.Type==tkt_idstr) || (tk.Type==tkt_str) ) &&
                    (ID==NULL) )
                {
                    __SETID
                }
                else if ( tk.Type==tkt_spec )
                {
                    if ( (tk.v.specv=='.') && (ID!=NULL) )
                    {
                        if ( !UCFSelectBlockForce( C, ID ) )
                            __ERR
                        free( ID );
                        ID=NULL;
                    }
                    else if ( ((tk.v.specv=='{') || (tk.v.specv=='\n'))
                        && (ID==NULL) )
                    {
                        while ( (tk.Type==tkt_spec) && (tk.v.specv=='\n') )
                            if ( !_GetToken( F, &tk ) )
                                __ERR
                        state=s_norm;
                    }
                    else if ( (tk.v.specv=='=') && (ID!=NULL) )
                    {
                        state=s_eq;
                    }
                    else
                        __ERR
                }
                else
                    __ERR
                break;      /* case s_id */

            case s_eq:
                if ( tk.Type==tkt_spec )
                {
                    if ( tk.v.specv=='.' )
                    {
                        if ( !UCFSelectBlockForce( C, ID ) )
                            __ERR
                        free( ID );
                        if ( !_GetToken( F, &tk ) )
                            __ERR
                        if (tk.Type==tkt_int)
                        {
                            if ( tk.v.intv<0 )
                                __ERR
                            arrindex=tk.v.intv;
                            if ( !_GetToken( F, &tk ) )
                                __ERR
                        }
                        else
                            arrindex=0;
                        while ( (tk.Type==tkt_spec) && (tk.v.specv=='\n') )
                            if ( !_GetToken( F, &tk ) )
                                __ERR
                        if ( (tk.Type==tkt_spec) && (tk.v.specv=='{') )
                        {
                            if ( (ID=malloc(33))==NULL )
                                __ERR
                            sprintf( ID, "%i", arrindex );
                            state=s_arr;
                            __PUSHC
                            state=s_val;
                            valid=0;
                            wasvalid=0;
                        }
                        else
                            __ERR
                    }
                    else if ( tk.v.specv=='#' )
                    {
                        state=s_val;
                        V.Type=UCFTempty;
                        valid=1;
                        wasvalid=0;
                    }
                    else
                        __ERR
                }
                else if ( (tk.Type==tkt_idstr) || (tk.Type==tkt_str) ||
                    (tk.Type==tkt_int) || (tk.Type==tkt_float) )
                {
                    state=s_val;
                    __SETV
                    valid=0;
                    wasvalid=1;
                }
                else
                    __ERR
                break;      /* case s_eq */

            case s_val:
                if ( (tk.Type==tkt_idstr) || (tk.Type==tkt_str) ||
                    (tk.Type==tkt_int) || (tk.Type==tkt_float) )
                {
                    if ( (valid&&(wasvalid!=2)) || (wasvalid==0) )
                    {
                        __SETV
                        if ( valid )
                            wasvalid=2;
                        else
                            wasvalid=1;
                    }
                }
                else /* tk.Type==tkt_spec */
                {
                    if ( tk.v.specv=='#' )
                        valid=1;
                    else if ( ( (tk.v.specv==';') || (tk.v.specv==',') ||
                        (tk.v.specv=='\n') || (tk.v.specv=='}') ||
                        (tk.v.specv==EOF) ) && (V.Type!=UCFTempty) )
                    {
                        if ( (wasvalid!=0) &&
                            !ChangeUCFVariable( C, ID, &V ) )
                            __ERR
                        __POPC
                        if ( state!=s_arr ) free( ID );
                        if ( tk.v.specv=='}' )
                        {
                            if ( state==s_arr )
                                free( ID );
                            __POPC
                        }
                        else if ( state==s_arr )
                        {
                            if ( wasvalid!=0 )
                            {
                                arrindex++;
                                sprintf( ID, "%i", arrindex );
                            }
                            __PUSHC
                            state=s_val;
                            valid=0;
                            wasvalid=0;
                        }
                        else if ( tk.v.specv==',' )
                            __ERR
                    }
                }
                break;      /* case s_val */

        }   /* switch ( state ) */
        if ( ((tk.Type==tkt_str) || (tk.Type==tkt_idstr)) && tk.v.stringv )
            free( tk.v.stringv );
    }   /* for (;;) */

__label_error:
    if ( stack ) free( stack );
    if ( ((tk.Type==tkt_str) || (tk.Type==tkt_idstr)) && tk.v.stringv )
        free( tk.v.stringv );
    if ( ID ) free( ID );
    C->Current=Root;
    return 0;

#undef CONSTANT_NUM
#undef __ERR
#undef __PUSH
#undef __POP
#undef __PUSHC
#undef __POPC
#undef __SETID
#undef __SETV

}  /* UCFReadConfigFile() */


/*##############################################################*/
/* End of file: UCF.C */

