/****************************************************************
** UCFNORM.C
**
** Universal Configuration File (UCF) Normalizer
**
**  Copyright (C) 1999 Blint Balogh <balintb@iname.com>
**
**  This program is free software; you can redistribute it and/or modify it
**  under the terms of the GNU General Public License as published by the
**  Free Software Foundation; either version 2 of the License, or (at your
**  option) any later version.
**
**  This program is distributed in the hope that it will be useful, but
**  WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
**  Public License for more details.
**
**  You should have received a copy of the GNU General Public License along
**  with this program (the file "COPYING"); if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**/

#include <stdio.h>
#include "ucf.h"

static UCFContext C;
static FILE *InF, *OutF;
static char ErrCannotOpen[]="Cannot open file: %s\n";

static void PrintString( const char *String )
{
    int i;

    fputc( '\"', OutF );
    for ( i=0; String[i]; i++ )
        if ( String[i]=='\n' )
            fprintf( OutF, "\\n" );
        else if ( String[i]=='\r' )
            fprintf( OutF, "\\r" );
        else if ( String[i]=='\t' )
            fprintf( OutF, "\\t" );
        else if ( (String[i]<' ') || (String[i]==127) )
            fprintf( OutF, "\\x%X%X", String[i]>>4, String[i]&0xF );
        else if ( String[i]=='\"' )
            fprintf( OutF, "\\\"" );
        else
                fputc( String[i], OutF );
    fputc( '\"', OutF );
}

static void PrintBlock( int Space )
{
    UCFFNContext N;
    UCFValue V;
    char *ID;

    if ( !InitFindUCFVariable( &C, &N ) )
    {
        fprintf( stderr, "InitFindUCFVariable() failed\n" );
        return;
    }
    while ( FindNextUCFVariable( &C, &N, &ID, &V ) )
    {
        if ( Space ) fprintf( OutF, "%*c", Space, ' ' );
        PrintString( ID );
        if ( V.Type!=UCFTblock )
        {
            fputc( '=', OutF );
            switch ( V.Type )
            {
                case UCFTstring:
                    PrintString( V.v.stringv );
                    fputc( '\n', OutF );
                    break;
                case UCFTint:
                    fprintf( OutF, "%i\n", V.v.intv );
                    break;
                case UCFTfloat:
                    fprintf( OutF, "%g\n", V.v.floatv );
                    break;
            }
        }
        else
        {
            if ( !UCFSelectBlock( &C, ID ) )
                fprintf( stderr, "UCFSelectBlock() failed\n" );
            else
            {
                fprintf( OutF, ".\n" );
                if ( Space ) fprintf( OutF, "%*c", Space, ' ' );
                fprintf( OutF, "{\n" );
                PrintBlock( Space+2 );
                if ( Space ) fprintf( OutF, "%*c", Space, ' ' );
                fprintf( OutF, "}\n" );
                if ( !UCFSelectParent( &C ) )
                    fprintf( stderr, "UCFSelectParent() failed\n" );
            }
        }
    }
}

int main( int argc, char *argv[] )
{

    printf( "\nUCF Normalizer (C) 1999 Balint Balogh <balintb@iname.com>\n\n"
".  This program comes with ABSOLUTELY NO WARRANTY; this is free software,\n"
".  and you are welcome to redistribute it under the terms of the GNU General\n"
".  Public License as published by the Free Software Foundation;\n"
".  see the file \"COPYING\" for details.\n\n" );

    if ( argc>1 )
    {
        if ( (InF=fopen( argv[1], "rt" ))==NULL )
        {
            fprintf( stderr, ErrCannotOpen, argv[1] );
            return 1;
        }
    }
    else
        InF=stdin;

    if ( argc>2 )
    {
        if ( (OutF=fopen( argv[2], "wt" ))==NULL )
        {
            fprintf( stderr, ErrCannotOpen, argv[2] );
            return 2;
        }
    }
    else
        OutF=stdout;

    InitUCFContext( &C );

    if ( !UCFReadConfigFile( &C, InF ) )
    {
        fprintf( stderr, "UCFReadConfigFile() failed\n" );
    }
    if ( argc>1 ) fclose( InF );

    PrintBlock( 0 );

    if ( argc>2 ) fclose( OutF );

    return 0;
}

