{////////////////////////////////////////////////////////////////////////////
 // My purpose here is not really to teach you to code,  but just to maybe //
 // help you out a bit with the VGA registers and give you a few tricks.   //
 // You can get all this information in a file called vgadoc4b.zip at:     //
 // x2ftp.oulu.fi  /pub/msdos/programming/docs                             //
 //                                                                        //
 // And now to begin...                                                    //
 //                                                                        //
 // I bet some of you guys out there didn't know that since the beginning  //
 // of time,  it has been possible to address 128Kb of linear video RAM on //
 // the standard VGA (0A0000 - 0BFFFF).  Some of you who have been around  //
 // a while would probably have seen that on the old Trident VGA cards     //
 // (8900 ranges) you were able to do this without messing around with VGA //
 // registers,  even though other VGA cards only made 0A0000 - 0AFFFF (ie. //
 // the first 64Kb) addressable.                                           //
 //                                                                        //
 // Anyway,  it's quite easy to set the VGA into 128Kb address mode - just //
 // set bits 2-3 of VGA port 03CEh (index 6) to 0.  Like this:             //
 //    mov  dx,3ceh       ;port 03CEh - Graphics register set              //
 //    mov  al,6          ;index 06h  - Miscellaneous register             //
 //    out  dx,al                                                          //
 //    inc  dx                                                             //
 //    in   al,dx                                                          //
 //    and  al,11110011b  ;clear bits 2-3                                  //
 //    out  dx,al         ;load register                                   //
 //                                                                        //
 // This of course opens up a few new doors.  You can have just over two   //
 // pages of video in standard BIOS VGA mode 13h (320x200x256). Or you can //
 // set bits 0-4, index 9, port 03D4h to 00000 and give yourself packed-   //
 // pixel memory model 320x400x256 mode (like mode 13h but 400 lines).  Or //
 // you can be creative and begin to mess around with some CRTC registers  //
 // to get your own resolutions.                                           //
 //                                                                        //
 // BTW: tweak16b.zip contains a file called vga.txt i think? which tells //
 //      you all this stuff.  If you want to start making your own modes,  //
 //      mess around with tweak16b after getting to know the basics from   //
 //      this code.  A nice idea would be to incorporate Finn's mode Q     //
 //      (256x256x256) and the 128Kb idea to get exactly 2 pages.  Hmm.... //
 //                                                                        //
 // There are a few nice routines here,  like pixel-granularity video      //
 // buffer positioning, setting vertical retrace, display and blank bounds //
 // and an accurate refresh rate measure.  Feel free to modify and/or use  //
 // this code for whatever you want - you can even greet me if you like.   //
 // <deep breath> BUT IF YOU CHANGE THIS FILE IN ANY WAY, PLEASE CLEARLY   //
 // MARK WHAT PARTS HAVE BEEN CHANGED OR ADDED OR REMOVED AND KEEP YOUR    //
 // COMMENTS SEPARATE FROM THIS BOX!  (whew.  dont like shouting...)       //
 //                                                                        //
 // Also,  it is possible to damage hardware (like your screen) by messing //
 // with the VGA regs,  but I will not be held responsible if that happens.//
 // (But send me a message if you manage such a great feat - I think we can//
 // appreciate software that is able to physically damage something 8-D ) //
 //                                                                        //
 // Anyway,  this code is pretty much self-explanatory so get on with it.  //
 //                                                                        //
 // IMPORTANT:  About ME...         ;)                                     //
 //     Name: Mark Webster                                                 //
 //    Alias: lsd                                                          //
 //    Group: MeltDown  (South Africa)                                     //
 // Position: lead coder                                                   //
 //   e-mail: centri@iafrica.com    (be sure to direct it to Mark or lsd)  //
 //    snail: 32 Frere Road                                                //
 //           Glenwood                                                     //
 //           4001                                                         //
 //           South Africa                                                 //
 // (Send flames, advice, infinite gratitude, grovelling pleas, etc)       //
 //                                                                        //
 // Oh yes... I also have a sneaky way of using almost all the VGA RAM in  //
 //     several modes by doing sneaky things with VBE calls. Contact me if //
 //     you want to know - I might post some more code to x2ftp.oulu.fi    //
 ////////////////////////////////////////////////////////////////////////////
}


uses dos,crt;

type rgb=record r,g,b:byte; end;

const tickfreq=1000;

var a,b,c   : word;
    d       : longint;
    pal     : array[0..255] of rgb;
    old8    : procedure;
    maxtick,
    ticks   : word;
    ch      : char;

    vtotal,            {Vertical total}
    vrstart,           {Vertical retrace start}
    vdend,             {Vertical display end}
    vbstart  : word;   {Vertical blank start}
    vrend,             {Vertical retrace end}
    vbend    : byte;   {Vertical blank end}
    hz       : word;   {Refresh rate}
    clock,             {Clock  0-3}
    lines,             {Display lines 0-3}
    scans    : byte;   {Scan line repeat 0-63}
    LXRes    : word;   {Logical X resolution}



procedure int8; interrupt; assembler;
asm mov  al,20h
    out  20h,al
    inc  ticks
end;


procedure waitvr; assembler;
asm
    mov  dx,3dah
@v1:in   al,dx
    test al,8
    jnz  @v1
@v2:in   al,dx
    test al,8
    jz   @v2
end;


Procedure Enable128Kb; assembler;
asm
    mov  dx,3ceh
    mov  al,6
    out  dx,al            ;{ Graphics Miscellaneous Register }
    inc  dx
    in   al,dx
    and  al,11110011b     ;{ Mask out Memory Mapping bits }
    out  dx,al            ;{ Issue new value }
end;


Procedure SetClock(clock:byte); assembler;
{ IBM define the clocks as such: 0 - 25Mhz,      1 - 28Mhz,
                                 2 - Reserved,   3 - Reserved
  But most VGA cards have defined 2 and 3 to be sometimes pleasantly
  surprising  (on clock 3 you can usually get 100hz refresh in mode 13h }
asm
    mov  dx,3cch      ;{ 03cch is where you read 03c2h (Misc Output Reg) }
    in   al,dx
    and  al,11110011b ;{ Mask off clock bits }
    mov  ah,clock
    and  ah,00000011b
    shl  ah,2
    or   al,ah        ;{ Calc new value for the reg }
    mov  dx,3c2h
    out  dx,al        ;{ Misc Output Register   -  set the clock! }
end;

function GetClock:byte; assembler;
asm
    mov  dx,3cch      ;{ 03cch is where you read 03c2h (Misc Output Reg) }
    in   al,dx
    and  al,00001100b ;{ Get clock bits }
    shr  al,2
end;


Procedure SetDisplayLines(lines:byte); assembler;
{ IBM define the lines on display as such:
        0 - Reserved,  1 - 400,  2 - 350,  3 - 480
  but dont be scared to use 0 because your VGA card has probably defined it
  as something nice }
asm
    mov  dx,3cch      ;{ 03cch is where you read 03c2h (Misc Output Reg) }
    in   al,dx
    and  al,00111111b ;{ Mask off display-lines bits }
    mov  ah,lines
    and  ah,00000011b
    shl  ah,6
    or   al,ah        ;{ Calc new value for the reg }
    mov  dx,3c2h
    out  dx,al        ;{ Misc Output Register }
end;

function GetDisplayLines:byte; assembler;
asm
    mov  dx,3cch      ;{ 03cch is where you read 03c2h (Misc Output Reg) }
    in   al,dx
    shr  al,6         ;{ Get display-lines bits }
end;


Procedure SetLogicalXRes(XRes:byte); assembler;
{In case you didn't know,  you can have a logical X resolution which can be
 greater than the displayed resolution.  ie. you may have display of 320, but
 each line may actually be up to 2040 pixels wide! Good for horiz. scrolling }
asm
    mov  dx,3d4h      ;{ CRTC }
    mov  al,13h       ;{ Logical line width (officially: Offset Register) }
    mov  ah,XRes
    out  dx,ax
    shr  ax,8         ;{Assuming we're in dword mode (normal for mode 13h), }
    shl  ax,3         ;{ we need XRes will be QWord granularity, so we need }
    mov  LXRes,ax     ;{ to multiply by 8 (shl by 3) to get actual XRes     }
end;

function GetLogicalXRes:byte; assembler;
asm
    mov  dx,3d4h      ;{ CRTC }
    mov  al,13h       ;{ Logical line width (officially: Offset Register) }
    out  dx,al
    inc  dx
    in   al,dx
end;


Procedure SetDisplayStart(offs:longint); assembler;
{ This procedure will set the display buffer start to an offset within the
  first 64K of Video RAM }
asm
    db 66h; mov  bx,[word ptr offs] ;{ mov ebx,[dword ptr offs] }
    db 66h; shr  bx,2               ;{ shr ebx,2 }

    mov  dx,3d4h              ;{ 03d4h - CRT Controller }
    mov  al,0dh               ;{ 0dh - Lower 8 bits of display start }
    mov  ah,bl
    out  dx,ax

    mov  al,0ch
    mov  ah,bh                ;{ 0ch - Upper 8 bits of display start }
    out  dx,ax

    call waitvr

    mov  ah,[byte ptr offs]
    and  ah,00000011b
    shl  ah,1

    mov  dx,3dah
    in   al,dx
    mov  dx,3c0h         ;{ 03c0h - Attribute }
    mov  al,013h + 20h   ;{ 13h - Horizontal PEL panning register }
    out  dx,al
    mov  al,ah
    out  dx,al
end;

function GetDisplayStart:longint; assembler;
asm
    db 66h; xor bx,bx       ;{ xor ebx,ebx }

    mov  dx,3d4h              ;{ 03d4h - CRT Controller }
    mov  al,0dh               ;{ 0dh - Lower 8 bits of display start }
    out  dx,al
    inc  dx
    in   al,dx
    mov  bl,al

    dec  dx
    mov  al,0ch
    out  dx,al                ;{ 0ch - Upper 8 bits of display start }
    inc  dx
    in   al,dx
    mov  bh,al

    db 66h; shl bx,2        ;{ shl ebx,2 }

    mov  dx,3dah
    in   al,dx
    mov  dx,3c0h         ;{ 03c0h - Attribute }
    mov  al,013h + 20h   ;{ 13h - Horizontal PEL panning register }
    out  dx,al
    inc  dx
    in   al,dx
    shr  al,1
    and  al,00000011b
    or   bl,al

    mov  ax,bx
    db 66h; shr bx,16       ;{ shr ebx,16 }
    mov  dx,bx
end;


Procedure SetScanLineRepeat(scans:byte); assembler;
{Here you can tell the VGA to repeat scan lines.  eg. mode 13h is actually a
 400 line mode,  but each scan line is repeated twice to get 200 lines.
 "scans" can range from 0 to 63 }
asm
    and  scans,00111111b
    mov  ah,scans
    test ah,00100000b
    jz   @ok
{    dec  ah}
    shr  ah,1
    or   ah,10000000b

@ok:mov  dx,3d4h
    mov  al,09           ;{ 09h - Scan repeat register                      }
    out  dx,al           ;{ note: bit 7 causes all scan lines to be doubled }
    inc  dx
    in   al,dx
    and  al,01100000b
    or   al,ah
    out  dx,al
end;

function GetScanLineRepeat:byte; assembler;
asm
    mov  dx,3d4h
    mov  al,09           ;{ 09h - Scan repeat register                      }
    out  dx,al           ;{ note: bit 7 causes all scan lines to be doubled }
    inc  dx
    in   al,dx
    and  al,10011111b
    test al,10000000b
    jz   @ok

    and  al,00011111b
    shl  al,1
@ok:
end;


Procedure SetVerticalBounds(vtotal,vrstart,vrend,vbstart,vbend,vdend:word); assembler;
{Mess a bit with these values to get custom Y resolutions (use also in con-
 junction with SetDisplayLines and SetClock).  Sorry,  no tweaking of X resolutions
 here... }
asm
    cli
    mov  dx,3d4h
    mov  al,6
    mov  ah,[byte ptr vtotal]
    out  dx,ax

    mov  al,10h
    mov  ah,[byte ptr vrstart]
    out  dx,ax

    mov  al,11h
    out  dx,al
    inc  dx
    in   al,dx
    and  al,11110000b
    and  vrend,00001111b
    or   ax,vrend
    out  dx,al
    dec  dx

    mov  al,12h
    mov  ah,[byte ptr vdend]
    out  dx,ax

    mov  al,15h
    mov  ah,[byte ptr vbstart]
    out  dx,ax

    mov  al,16h
    mov  ah,[byte ptr vbend]
    out  dx,ax

    mov  dx,3d4h

    mov  al,7
    out  dx,al
    inc  dx
    in   al,dx
    and  al,00010000b
    mov  bx,vtotal
    mov  bl,bh
    and  bl,00000001b
    or   al,bl
    and  bh,00000010b
    shl  bh,4
    or   al,bh

    mov  bx,vdend
    mov  bl,bh
    and  bl,00000001b
    shl  bl,1
    or   al,bl
    and  bh,00000010b
    shl  bh,5
    or   al,bh

    mov  bx,vrstart
    mov  bl,bh
    and  bl,00000001b
    shl  bl,2
    or   al,bl
    and  bh,00000010b
    shl  bh,6
    or   al,bh

    mov  bx,vbstart
    and  bh,1
    shl  bh,3
    or   al,bh
    out  dx,al
    sti
end;



function GetRefreshRate:word;
{Gets the screen refresh rate}
  var a:word;
begin
  a:=round(1193180/tickfreq);
  maxtick:=round(1193180/a) *1; { *1 = 1 second }
  asm mov  al,36h
      out  43h,al
      mov  ax,a
      out  40h,al
      mov  al,ah
      out  40h,al
      cli
  end;              ;{set timer frequency ( int 8 -> irq0 ) }

  getintvec(8,@Old8); setintvec(8,@int8);
  waitvr;
  asm sti end;
  ticks:=0; repeat until ticks=1; ticks:=0;
  a:=0;             {Synchronize retrace and timer}

  repeat
    waitvr;
    if ticks<maxtick then inc(a);
  until ticks>=maxtick;   {Wait one second and count the retraces}

  asm cli end;
  setintvec(8,@Old8);
  asm mov  al,36h
      out  43h,al
      xor  al,al
      out  40h,al
      out  40h,al
      sti
  end;             ;{ Put timer back to 18.2065 ticks and restore int8 }

  GetRefreshRate:=a;
end;



begin
  asm mov  ax,13h
      int  10h      ;{320x200x256 Chained}
                    ;{Feel free to use different modes (even VESA modes
                      if you know how - contact me if you want a decent
                      VESA .tpu }

      call Enable128Kb

      mov  bx,0a000h
      mov  es,bx
      xor  bx,bx
   @1:mov  [es:bx],bl
      inc  bx
      jnz  @1       ;{fill 1st 64Kb with some junk...}

      mov  bx,0b000h
      mov  es,bx
      xor  bx,bx
      xor  ax,ax
      xor  cx,cx

   @2:mov  [es:bx],al
      dec  cx
      js   @3
   @4:inc  bx
      jnz  @2
      jmp  @5

   @3:mov  cx,319
      inc  al
      jmp  @4       ;{fill 2nd 64Kb with some horizontal junk...}

   @5:
  end;

  directvideo:=false;

;{This next bunch of code gets the current values of the vertical retrace,
  display, and blank registers }
  asm mov  dx,3d4h
      mov  al,11h
      out  dx,al
      inc  dx
      in   al,dx
      and  al,01111111b
      out  dx,al

      mov  dx,3d4h
      mov  al,6
      out  dx,al
      inc  dx
      in   al,dx

      mov  bl,al
      dec  dx
      mov  al,7
      out  dx,al
      inc  dx
      in   al,dx
      and  al,00100001b
      mov  ah,al
      shr  ah,4
      and  al,1
      or   ah,al
      mov  al,bl

      mov  vtotal,ax

      mov  dx,3d4h
      mov  al,10h
      out  dx,al
      inc  dx
      in   al,dx
      mov  bl,al
      dec  dx
      mov  al,7
      out  dx,al
      inc  dx
      in   al,dx
      and  al,10000100b
      shr  al,2
      mov  ah,al
      shr  ah,4
      and  al,1
      or   ah,al
      mov  al,bl
      mov  vrstart,ax

      mov  dx,3d4h
      mov  al,11h
      out  dx,al
      inc  dx
      in   al,dx
      and  al,00001111b
      mov  vrend,al

      mov  dx,3d4h
      mov  al,12h
      out  dx,al
      inc  dx
      in   al,dx
      mov  bl,al
      dec  dx
      mov  al,7
      out  dx,al
      inc  dx
      in   al,dx
      and  al,01000010b
      shr  al,1
      mov  ah,al
      shr  ah,1
      and  al,1
      or   ah,al
      mov  al,bl
      mov  vdend,ax

      mov  dx,3d4h
      mov  al,15h
      out  dx,al
      inc  dx
      in   al,dx
      mov  bl,al
      dec  dx
      mov  al,7
      out  dx,al
      inc  dx
      in   al,dx
      and  al,00001000b
      shr  al,3
      mov  ah,al
      mov  al,bl
      mov  vbstart,ax

      mov  dx,3d4h
      mov  al,16h
      out  dx,al
      inc  dx
      in   al,dx
      mov  vbend,al
      out  dx,al
  end;


  lines:=GetDisplayLines;
  clock:=GetClock;
  scans:=GetScanLineRepeat;

  hz:=GetRefreshRate;

  repeat
    gotoxy(1,1);
    writeln('key  function value');
    writeln('q/a  vtotal:  ',vtotal:4);
    writeln('w/s  vrstart: ',vrstart:4);
    writeln('e/d  vrend:   ',vrend:4);
    writeln('r/f  vbstart: ',vbstart:4);
    writeln('t/g  vbend:   ',vbend:4);
    writeln('y/h  vdend:   ',vdend:4);
    writeln;
    writeln('c    clock:   ',clock:4);
    writeln('l    lines:   ',lines:4);
    writeln('+/-  scans:   ',scans:4);
    writeln('m    refresh rate: ',hz,'Hz ');
    ch:=upcase(readkey);
    case ch of
     'Q' : inc(vtotal);
     'A' : dec(vtotal);
     'W' : inc(vrstart);
     'S' : dec(vrstart);
     'E' : inc(vrend);
     'D' : dec(vrend);
     'R' : inc(vbstart);
     'F' : dec(vbstart);
     'T' : inc(vbend);
     'G' : dec(vbend);
     'Y' : inc(vdend);
     'H' : dec(vdend);
     'C' : clock:=(clock+1) and 3;
     'L' : lines:=(lines+1) and 3;
     '+' : if scans<64 then inc(scans);
     '-' : if scans>0 then dec(scans);
     'M' : begin
             gotoxy(6,12); writeln('measuring....   ');
             hz:=GetRefreshRate;
           end;
    end;
    vtotal:=vtotal and 1023;
    vrstart:=vrstart and 1023;
    vrend:=vrend and 15;
    vbstart:=vbstart and 511;
    vbend:=vbend and 255;
    vdend:=vdend and 1023;
    waitvr;
    SetVerticalBounds(vtotal, vrstart, vrend, vbstart, vbend, vdend);
    SetClock(clock);
    SetDisplayLines(lines);
    SetScanLineRepeat(scans);
  until ch=#27;

  asm mov ax,3
      int 10h
  end;
end.