// Routines for PowerGlove cursor load, movement and
// manipulation interface.  Used in combination with GLOVEPTR.C

/* Original written by Dave Stampe, Aug. 1992 */

/*
 This code is part of the VR-386 project, created by Dave Stampe.
 VR-386 is a desendent of REND386, created by Dave Stampe and
 Bernie Roehl.  Almost all the code has been rewritten by Dave
 Stampre for VR-386.

 Copyright (c) 1994 by Dave Stampe:
 May be freely used to write software for release into the public domain
 or for educational use; all commercial endeavours MUST contact Dave Stampe
 (dstampe@psych.toronto.edu) for permission to incorporate any part of
 this software or source code into their products!  Usually there is no
 charge for under 50-100 items for low-cost or shareware products, and terms
 are reasonable.  Any royalties are used for development, so equipment is
 often acceptable payment.

 ATTRIBUTION:  If you use any part of this source code or the libraries
 in your projects, you must give attribution to VR-386 and Dave Stampe,
 and any other authors in your documentation, source code, and at startup
 of your program.  Let's keep the freeware ball rolling!

 DEVELOPMENT: VR-386 is a effort to develop the process started by
 REND386, improving programmer access by rewriting the code and supplying
 a standard API.  If you write improvements, add new functions rather
 than rewriting current functions.  This will make it possible to
 include you improved code in the next API release.  YOU can help advance
 VR-386.  Comments on the API are welcome.

 CONTACT: dstampe@psych.toronto.edu
*/


#include <stdio.h>
#include <dos.h>

#include "vrconst.h"
#include "pointint.h"
#include "vr_api.h"
#include "intmath.h"
#include "pcdevice.h"
#include "segment.h"


/***************** GLOVE CURSOR SUPPORT *************/

/* also used to manip. objects */

static SEGMENT *glove_seg; /* wrist twist for presentation */
static SEGMENT *glove_joints[20]; /* joints on hand */

#define SELECT_LENGTH 20	/* distance along last index finger segment */
/* to selection point */

static int glove_update(PDRIVER *d, POINTER *p) /* read glove, update positions */
{
  int c;

  c = pointer_read(d, p);

  if ((c & (PNEW_POS | PNEW_ROT | PNEW_FLEX)) == 0) return 0;

  abs_move_segment(wrist_seg, p->x, p->y + Y_PTR_OFFSET, p->z + PTR_DIST);

  if (d->pdata->type&P_IS6DG)
    abs_rot_segment(wrist_seg, p->rx, p->ry, p->rz, RYXZ);
  else    // fake rotation: looks good
    abs_rot_segment(glove_seg, -2000L*p->y, p->ry, -4000L*p->x, RYXZ);

  abs_rot_segment(glove_joints[1], 0L,30*65536L+18061L*p->flex[0],0L, RYXZ);
  abs_rot_segment(glove_joints[2], -20*65536L,90*65536L,-5*65536L+38700L*p->flex[1], RYZX);
  abs_rot_segment(glove_joints[3], 38700*p->flex[2],0L,0L, RYXZ);
  abs_rot_segment(glove_joints[4], 38700*p->flex[3],0L,0L, RYXZ);
  abs_rot_segment(glove_joints[5], 38700*p->flex[4],0L,0L, RYXZ);
  abs_rot_segment(glove_joints[6], 38700*p->flex[5],0L,0L, RYXZ);
  abs_rot_segment(glove_joints[7], 38700*p->flex[6],0L,0L, RYXZ);
  abs_rot_segment(glove_joints[8], 38700*p->flex[7],0L,0L, RYXZ);
  abs_rot_segment(glove_joints[9], 38700*p->flex[8],0L,0L, RYXZ);
  abs_rot_segment(glove_joints[10],38700*p->flex[9],0L,0L, RYXZ);

  world_changed++;
  update_object(wrist_seg);
  return 1;
}


	// sets up the 3D/6D manip system
void set_3D_manip_data(PDRIVER *ptr, SEGMENT *move, SEGMENT *sel, POSE *seloffset);


static int load_glove_cursor(PDRIVER *gd, char *glove_fname)
{
  int i;
  FILE *in;
  OBJECT *obj;
  POSE pgo = { 0,0,SELECT_LENGTH,0,0,0};  // glove to select point

  if ((in = fopen(glove_fname, "r")) == NULL)
    {
      errprintf("Cannot read glove object file %s", glove_fname);
      return -1;
    }
  if ((gd->pdata->type & P_IS6DG) == 0)
	abs_rot_segment(wrist_seg, 55*65536L, 0, 0, RYXZ); /* glove wrist pose */

  glove_seg = load_figure_as_object(in, default_objlist, glove_joints, 20, 1, 1, 1);
  attach_object(glove_seg, wrist_seg, 0);
  fclose(in);

  add_objlist_to_world(default_objlist);

  if (seg_error(NULL) || (glove_seg == NULL))
    {
      errprintf("%s loading glove object", seg_error(NULL));
      return -2;
    }

  do_for_all_child_objects(glove_seg, make_object_unselectable);

	// register 3D manipulation data
  set_3D_manip_data(gd, wrist_seg, glove_joints[4], &pgo);

  return 0;
}

static PDRIVER *glvptr;


static void gloveptr_quit()
{
  pointer_quit(glvptr);
}


PDRIVER *glove_initialize(char *glvdev, char *glvobj, POSE *s)
{
  PDRIVER *p;
  POINTER x;

  p = pointer_init(P_IS3DG | P_IS6DG, glvdev); /* setup glove device */
  if (p == NULL) return NULL;

  init_pointer(&x); /* so that defaults are OK */
	/* use abs. glove motion */
  pointer_abscale(p, s->x, s->y, s->z, s->rx, s->ry, s->rz);
  set_mouse_limits(p, screeninfo->xmax, screeninfo->ymax);
  pointer_read(p, &x);
  pointer_read(p, &x); /* save current position */
  mouse_read(p, NULL, NULL, NULL);

  glvptr = p;
  atexit(gloveptr_quit);

  load_glove_cursor(p, glvobj);

  return p;
}




	// read, handle glove gestures
void glove_process()
{
  POINTER gp;
  int g;

  glove_update(glvptr, &gp);
  switch(gp.gesture)
    {
      case G_FIST :
	g = GRASP_DO;
	break;
      case G_PINCH:
	g = ROTATE_DO;
	break;
      case G_POINT:
	g = SELECT_DO;
	break;
      default:
	g = FREE_DO;
	break;
   }
 do_3D_manip(g);
}



static char *gest[] = {
	"FLAT", "THUMB_IN", "INDEX_IN", "MIDDLE_IN",
	"RING_IN", "PINCH", "FIST", "THUMB_OUT", "POINT",
	"BADFINGER", "RING_OUT", "??????" };


char *get_glove_gesture_name()
{
  if (glvptr) /* prevent pre-init call */
    {
      int sl;
      POINTER p;

      last_pointer(glvptr, &p);
      sl = p.gesture;
      if (sl<0 || sl>G_UNKNOWN) sl = G_UNKNOWN;
      return gest[sl];
    }
  return NULL;
}