// This file: Reads joystick devices thru pointer drivers
// in joyptrs.c.  Adds and supports drivers.
// Uses joystick data to "move" a POSE structure for user

// ALL code in this file by Dave Stampe, 12/24/93

/*
 This code is part of the VR-386 project, created by Dave Stampe.
 VR-386 is a desendent of REND386, created by Dave Stampe and
 Bernie Roehl.  Almost all the code has been rewritten by Dave
 Stampre for VR-386.

 Copyright (c) 1994 by Dave Stampe:
 May be freely used to write software for release into the public domain
 or for educational use; all commercial endeavours MUST contact Dave Stampe
 (dstampe@psych.toronto.edu) for permission to incorporate any part of
 this software or source code into their products!  Usually there is no
 charge for under 50-100 items for low-cost or shareware products, and terms
 are reasonable.  Any royalties are used for development, so equipment is
 often acceptable payment.

 ATTRIBUTION:  If you use any part of this source code or the libraries
 in your projects, you must give attribution to VR-386 and Dave Stampe,
 and any other authors in your documentation, source code, and at startup
 of your program.  Let's keep the freeware ball rolling!

 DEVELOPMENT: VR-386 is a effort to develop the process started by
 REND386, improving programmer access by rewriting the code and supplying
 a standard API.  If you write improvements, add new functions rather
 than rewriting current functions.  This will make it possible to
 include you improved code in the next API release.  YOU can help advance
 VR-386.  Comments on the API are welcome.

 CONTACT: dstampe@psych.toronto.edu
*/


#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <bios.h>
#include <conio.h>

#include "pointer.h"
#include "vr_api.h"
#include "pcdevice.h"  /* for low-level joystick and keyboard */
#include "intmath.h"
#include "vrconst.h"


static unsigned lastarrowkey = 0;
extern int use_keyjoy;


BOOL process_motion_keys(unsigned c)
{
  BOOL handled = 1;
  int i;

  key_set_move(0,0,0,0);        // reset key nav. joystick device

  switch (c)
	{
	case LEFT:
	case SHLEFT:
	case ASHLEFT:
		if(use_keyjoy) break;
		key_set_move(-1,0,0,0);
		lastarrowkey = c;
		break;
	case RIGHT:
	case SHRIGHT:
	case ASHRIGHT:
		if(use_keyjoy) break;
		key_set_move(1,0,0,0);
		lastarrowkey = c;
		break;
	case UP:
		if(use_keyjoy) break;
		key_set_move(0,-1,0,0);
		lastarrowkey = c;
		break;
	case DOWN:
		if(use_keyjoy) break;
		key_set_move(0,1,0,0);
		lastarrowkey = c;
		break;
	case SHDOWN:
	case ASHDOWN:
		if(use_keyjoy) break;
		key_set_move(0,-1,1,0);
		lastarrowkey = c;
		break;
	case SHUP:
	case ASHUP:
		if(use_keyjoy) break;
		key_set_move(0,1,1,0);
		lastarrowkey = c;
		break;
	case LSHLEFT:
	case ALSHLEFT:
		if(use_keyjoy) break;
		key_set_move(1,0,0,1);
		lastarrowkey = c;
		break;
	case LSHRIGHT:
	case ALSHRIGHT:
		if(use_keyjoy) break;
		key_set_move(-1,0,0,1);
		lastarrowkey = c;
		break;
	case LSHUP:
	case ALSHUP:
		if(use_keyjoy) break;
	case PGUP:
		key_set_move(0,-1,0,1);
		lastarrowkey = c;
		break;
	case LSHDOWN:
	case ALSHDOWN:
		if(use_keyjoy) break;
	case PGDN:
		key_set_move(0,1,0,1);
		lastarrowkey = c;
		break;
	case CTRLPGUP:
		key_set_move(1,0,1,1);
		lastarrowkey = c;
		break;
	case CTRLPGDN:
		key_set_move(-1,0,1,1);
		lastarrowkey = c;
		break;
	case 'R':                   // repeat (benchmark)
		if (lastarrowkey)
		  for (i = 0; i < 100; i++)
		    {
		      process_motion_keys(lastarrowkey);
		      joystick_process();
		      update_body_links();
		      screen_refresh(current_camera);
		      if(kbhit()) break;
		    }
		position_changed++;
		break;

	default: handled = 0;
	}
  return handled;
}

static int qset = 0;

// up to 9 joystick devices supported

static PDRIVER *joydevs[10] = {NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL};

static void joy_quit()
{
  PDRIVER **p = joydevs;
  while(*p)
    {
      pointer_quit(*p);
      p++;
    }
}


BOOL add_joy_device(char *name)
{
  PDRIVER **ps, **pa = joydevs;
  PDRIVER *p = pointer_init(P_IS2DP,name);
  POINTER x;

  if(!p) return 1;	// can't create
  init_pointer(&x); /* so that defaults are OK */
  pointer_tscale(p, 50, 50, 50); /* scale motion */
  pointer_read(p, &x);
  pointer_read(p, &x); /* save current position */

  do { ps = pa; }  while((*pa++)!=NULL);
  *ps = p;
  if(!qset++) atexit(joy_quit);

  return 0;
}


/*************** JOYSTICK NAVIGATION ***************/


/// This reads all pointer (joystick), changes body pose to match

static int b_spinmap[4] = { 1, 1, 1, 3 };


BOOL do_joy_navigate(POSE * body_pose, BOOL spinmode, COORD sstep, ANGLE astep, BOOL screendir)
{
  PDRIVER **pa = joydevs;
  PDRIVER *p;
  POINTER px;
  int c;
  int x,y,b;
  long dist;
  long ang;
  int sscale;
  long mx=0, my=0, mz=0;
  MATRIX n;
  int changed = 0;

  while((p=*pa++)!=NULL)	// for each device...
   {
     c = pointer_read(p,&px);
     x = px.x;
     y = px.y;
     if(x==0 && y==0) continue;
     b = px.buttons & 3;
     dist = sstep;
     ang = astep;
     sscale = last_render_time();
     if(spinmode) b=b_spinmap[b];

     switch (b)	// set up move by buttons
       {
	 case 0:/* no buttons down */
		body_pose->ry += (x * ang)/2000*sscale;
		mz = -(y * dist * sscale)/400L;
		changed++;
		break;

	 case 1:/* first button down */
		body_pose->rx -= (y * ang)/1200*sscale;
		body_pose->ry += (x * ang)/1000*sscale;
		changed++;
		break;

	 case 2:/* second button down */
		mx = -(x * dist * sscale)/400L;
		my = -(y * dist * sscale)/400L;
		changed++;
		break;

	 case 3:/* both buttons down */
		body_pose->rz += (x * ang)/3000*sscale;
		if (stereo_type == MONOSCOPIC)           // none yet for stereo
		  {
		    SCALE z = get_camera_zoom(current_camera);
		    z += (y*65536L) /8000*sscale;
		    if(z<65536L*0.5) z = 65536L*0.5;
		    if(z>65536L*16.0) z = 65536L*16.0;
		    set_camera_zoom(current_camera,z);
		  }
		changed++;
		break;

	 default:
		break;
       }
    if(b!=1)            // map to screen orientation
      {
	if(screendir)
	  std_matrix(n,body_pose->rx,body_pose->ry,body_pose->rz, 0, 0, 0);
	else
	  std_matrix(n, 0, body_pose->ry, 0, 0, 0, 0);
	matrix_point(n, &mx, &my, &mz);
	body_pose->x += mx;
	body_pose->y += my;
	body_pose->z += mz;
	changed++;
      }
  }

  return changed;
}


	// passes command to all joystick drivers
	// currently used to control mouse
	// and to toggle acceleration mode
void joy_set_mode(int mode)
{
  PDRIVER **pa = joydevs;
  PDRIVER *p;

  while((p=*pa++)!=NULL)	// for each device...
   {
     device_command(p, mode);
   }
}


