                                             
/*
**  Dialogue / Display Functions.
**
**  Copyright 1993-2000 by Paul D. Burgin. All rights reserved.
*/

#include "stdio.h"
#include "conio.h"
#include "bios.h"
#include "time.h"
#include "stdarg.h"
#include "ctype.h"
#include "string.h"
#include "alloc.h"
#include "stdlib.h"
#include "dir.h"
#include "process.h"
#include "io.h"
#include "dos.h"
#include "setjmp.h"

#include "build.h"
#include "types.h"
#include "extern.h"
#include "6809cpu.h"
#include "6809regs.h"
#include "key.h"
#include "snapshot.h"
#include "doscart.h" /* DOSEMU */

#include "lfn.h"
#include "lfn2.h"    /* DOSEMU */

/* Prototype for hard reset jump buffer. */
extern jmp_buf hard_reset;

extern clock_t last_time, this_time;	/* Speed calculation intervals. */
double percent = -1.0;					/* Last speed % calculated.     */
double should_be_seconds;				/* Number of seconds it should  */
										/* take for 10000 IRQ's.        */

const unsigned char	for_reading_text[] = "     FOR READING";
const unsigned char	for_writing_text[] = "     FOR WRITING";
const unsigned char	for_editing_text[] = "     FOR EDITING";

unsigned char	saved_ff22;
signed char		saved_new_vmode;
unsigned int  	saved_new_screen_base;
unsigned char	saved_new_screen_size;
int				wx1, wy1, wx2, wy2;

/* Screen buffers. */
unsigned char	*norm_screen;
unsigned char	*debug_screen;
unsigned char	*info_screen;

boolean	info_init = FALSE;
unsigned char	os[] = { 45, 33, 0 };
unsigned char	vermajor = VER_MAJOR;
unsigned char	verminor = VER_MINOR;

/* Variables for memory to disk swapping. */
unsigned char	swapping = 1;
boolean			swapped;
unsigned char	out_of_memory[] = "Out Of Memory";

void beep(void)
{
   sound(650);
   delay(61);
   nosound();
}

/* Write a section of far memory to disk. */
void farwrite(unsigned char far *from, unsigned int size, FILE *swp)
{
	register unsigned char val;

	while (size > 0)
	{
		val = from[--size];
		checksum += val;
		fputc(val, swp);
	}
	farfree(from);
}

/* Swap memory out to disk. */
void swap_out(void)
{
	FILE	*swap;

	swapped = ((swap = lfn_fopen("pcd.swp","wb")) != NULL);
	if (swapped)
	{
		fprintf(swap, "PC-DRAGON SWAP FILE");
		fputc(arch, swap);
		checksum = 0;
		if (arch == dragon64)
			farwrite(rom2, 0x4000, swap);
		if (arch != dragon32)
			farwrite(hiram, 0x8000, swap);
		farwrite(memory, MEMORY_SIZE+1, swap);
		fputc(checksum, swap);
		free(norm_screen);
		fwrite(debug_screen,4000,1,swap);
		free(debug_screen);
		fwrite(info_screen,4000,1,swap);
		free(info_screen);
		fclose(swap);
	}
}

/* Read a section of far memory from disk. */
void farread(unsigned char far *to, unsigned int size, FILE *swp)
{
	while (size > 0)
		checksum += (to[--size] = fgetc(swp));
}

/* Swap memory in from disk. */
void swap_in(void)
{
	FILE	*swap;
	int		header_count;

	if (!swapped || ((swap = lfn_fopen("pcd.swp","rb")) == NULL))
		quit("Swap File Not Found",1);
	else
	{
		checksum = 0;
		for (header_count = 0; header_count < 20; header_count++)
			arch = fgetc(swap);
		if ((info_screen = (unsigned char *)malloc(4000)) == NULL)
			quit(out_of_memory,1);
		if ((debug_screen = (unsigned char *)malloc(4000)) == NULL)
			quit(out_of_memory,1);
		if ((norm_screen = (unsigned char *)malloc(2000)) == NULL)
			quit(out_of_memory,1);
		if ((memory = (unsigned char far *)farmalloc(MEMORY_SIZE+1)) == NULL)
			quit(out_of_memory,1);
		if (arch != dragon32)
		{
			if ((hiram = (unsigned char far *)farmalloc(0x8000)) == NULL)
				quit(out_of_memory,1);
		}
		if (arch == dragon64)
		{
			if ((rom2 = (unsigned char far *)farmalloc(0x4000)) == NULL)
				quit(out_of_memory,1);
			farread(rom2, 0x4000, swap);
		}
		if (arch != dragon32)
			farread(hiram, 0x8000, swap);
		farread(memory, MEMORY_SIZE+1, swap);
		if (checksum != fgetc(swap))
			quit("Checksum Failure On Swap File",1);
		fread(debug_screen,4000,1,swap);
		fread(info_screen,4000,1,swap);
		fclose(swap);
	}
}

/* View reference manual by calling TextView. */
void view_manual(void)
{
	char local_header[80];

	sprintf(local_header," PC-DRAGON V%d.%02d REFERENCE MANUAL ",
		VER_MAJOR, VER_MINOR);
	if (textview("docs\\pcdragon.txt",local_header,BLUE) == 0)
		textview("pcdragon.txt",local_header,BLUE);
}

/* Perform system command with optional swapping. */
void sys(unsigned char *sys_command)
{
	if (swapping >= 3)
		swap_out();
	system(sys_command);
	if (swapping >= 3)
		swap_in();
}

/* Shell escape commands. */
void shell_escape(unsigned char shell_function)
{
	if (new_int9_set)
		Set_Old_Int9();

	/* Go back to text mode. */
	temp_textmode();

	switch (shell_function)
	{
		case 0:	sys("DIR /-P /ON CASSETTE\\GAMES\\*.CAS > pcd.tmp");
				textview("pcd.tmp"," GAMES DIRECTORY ",RED);
				break;

		case 1:	sys("DIR /-P /ON CASSETTE\\ARCHIVE\\*.CAS > pcd.tmp");
				textview("pcd.tmp"," ARCHIVE DIRECTORY ",RED);
				break;

		case 2:	sys("DIR /-P /ON CASSETTE\\*.CAS > pcd.tmp");
				textview("pcd.tmp"," CASSETTE DIRECTORY ",RED);
				break;

		case 3: if (swapping >= 1)
					swap_out();
				setcursortype(_NORMALCURSOR);
				textattr((BLACK << 4) + DARKGRAY);
				cprintf("\n\rType EXIT to return to PC-DRAGON.");
				textattr((BLACK << 4) + LIGHTGRAY);
				set_directory(0);
				system("COMMAND");
				set_directory(1);
				if (swapping >= 1)
					swap_in();
				break;

		case 4:	view_manual();
				break;

		case 5:	sys("DIR /-P /A-D /ON SNAPSHOT\\*.* > pcd.tmp");
				textview("pcd.tmp"," SNAPSHOT DIRECTORY ",RED);
				break;

		case 6:	sys("DIR /-P /ON CARTRIGE\\*.DGN > pcd.tmp");
				textview("pcd.tmp"," CARTRIDGE DIRECTORY ",RED);
				break;

		case 7:	sys("DIR /-P /A-D /ON *.* > pcd.tmp");
				textview("pcd.tmp"," BASE DIRECTORY ",RED);
				break;

		case 8:	sys("DIR /-P /A-D /ON VIRTDISK\\*.* > pcd.tmp"); /* DOSEMU */
				textview("pcd.tmp"," VIRTUAL DISK DIRECTORY ",RED);
				break;

	}

	/* Go back into appropriate Dragon video mode & refresh. */
	restore_vmode();

	if (new_int9_set)
		Set_New_Int9();

	this_time = 0;
}

/* Prepare to draw a dialogue box. */
void box_setup(void)
{
	saved_ff22				= memory[0xff22];
	saved_new_vmode			= new_vmode;
	saved_new_screen_base	= new_screen_base;
	saved_new_screen_size	= new_screen_size;

	/* Return to text screen display. */
	if (vmode >= 0)
	{
		new_screen_size = 0;
		new_screen_base = last_text_base;
		memory[0xff22] &= 0x0f;
		execute_vmode(TRUE);
	}
	setcursortype(_NOCURSOR);
}

/* Draw border into a window of specified size. */
void border(unsigned char breadth, unsigned char depth)
{
	unsigned char border_count;

	putch(201);
	for(border_count = 2; border_count < breadth; border_count++)
		putch(205);
	putch(187);
	for(border_count = 2; border_count < depth; border_count++)
	{
		gotoxy(1, border_count);
		putch(186);
		gotoxy(breadth, border_count);
		putch(186);
	}
	gotoxy(1, depth);
	putch(200);
	for(border_count = 2; border_count < breadth; border_count++)
		putch(205);
	putch(188);
}

/* General dialogue box drawing function. */
void create_box(unsigned char box_lines, unsigned char box_width)
{
	unsigned char border_buffer[96];
	unsigned char btemp;

	box_setup();

	/* Calculate box co-ordinates. */
	wx1 = (debug_disp ? 40 : 20) - (box_width>>1);
	wx2 = wx1 + box_width;
	wy1 = 12 - (box_lines>>1);
	wy2 = wy1 + box_lines;

	/* Preserve screen contents. */
	gettext(wx1 - 2, wy1 - 2, wx2 + 3, wy2 + 2, norm_screen);

	/* Draw box shadow. */
	if (debug_disp)
	{
		gettext(wx1, wy2 + 2, wx2 + 1, wy2 + 2, border_buffer);
		for (btemp = 0; btemp < (box_width + 2); btemp++)
			border_buffer[1+(btemp<<1)] = DARKGRAY;
		puttext(wx1, wy2 + 2, wx2 + 1, wy2 + 2, border_buffer);
	}
	else
	{
		gettext(wx1 - 1, wy2 + 2, wx2 + 1, wy2 + 2, border_buffer);
		for (btemp = 0; btemp < (box_width + 3); btemp++)
			border_buffer[1+(btemp<<1)] = BLACK;
		puttext(wx1 - 1, wy2 + 2, wx2 + 1, wy2 + 2, border_buffer);
	}

	gettext(wx2 + 2, wy1 - 1, wx2 + 3, wy2 + 3, border_buffer);
	for (btemp = 0; btemp < (box_lines + 4); btemp++)
	{
		border_buffer[1+(btemp<<2)] = (debug_disp ? DARKGRAY : BLACK);
		if (debug_disp)
			border_buffer[3+(btemp<<2)] = DARKGRAY;
	}
	puttext(wx2 + 2, wy1 - 1, wx2 + 3, wy2 + 3, border_buffer);

	/* Draw box. */
	window(wx1 - 2, wy1 - 2, wx2 + 1, wy2 + 1);
	textattr((RED << 4) + WHITE);
	clrscr();
	border(box_width + 4, box_lines + 4);

	/* Prepare to draw text in the centre of the box. */
	window(wx1, wy1, wx2, wy2);
	textattr((RED << 4) + YELLOW);
}

/* Remove dialogue box/restore dragon screen. */
void remove_box(void)
{
	if (wx1 > 0)
		puttext(wx1 - 2, wy1 - 2, wx2 + 3, wy2 + 2, norm_screen);
	else
	{
		new_window(WIN_DRAGON);
		refresh_video();
	}
	new_window(WIN_DRAGON);
	setcursortype(cursortype);

	memory[0xff22]	= saved_ff22;
	new_vmode		= saved_new_vmode;
	new_screen_base	= saved_new_screen_base;
	new_screen_size	= saved_new_screen_size;

	if (new_int9_set)
		Set_New_Int9();

	this_time = 0;
}

/* Generic menu/selection box function. */
unsigned char selection_box(
	unsigned char num_lines,
	unsigned char width,
	unsigned char num_opt,
	boolean yes_no,
	boolean allow_ls,
	boolean raw_return,
	unsigned char default_opt, ...)
{
	unsigned char	selection				= 0;
	unsigned char	kin;
	boolean			got_selection			= FALSE;
	va_list			va_ap;					/* DOSEMU * was global */
	unsigned char	*txt;					/* DOSEMU * was global */
	unsigned char	opt_count;				/* DOSEMU * was global */

	/* Draw and fill box. */
	create_box(num_lines, width);
	va_start(va_ap, default_opt);
	for (opt_count = 0; opt_count < num_lines; opt_count++)
	{
		txt = va_arg(va_ap, unsigned char *);
		if ((txt[0] == '') && (txt[1] == '!'))
		{
			struct text_info txti;

			gettextinfo(&txti);
			textattr((txt[2] << 4) + txt[3]);
			cprintf("%s\n\r", &txt[4]);
			textattr(txti.attribute);
		}
		else
			cprintf("%s\n\r", txt);
	}
	va_end(va_ap);

	/* Restore original INT9 handler before doing this. */
	if (new_int9_set)
		Set_Old_Int9();

	/* Get user input. */
	while (!got_selection)
	{
		kin = getch2(TRUE);

		if (raw_return)
		{
			selection = kin;
			got_selection = TRUE;
		}
		else
		{
			/* Allow special case of 'L' and 'S' for snapshots. */
			if (allow_ls)
			{
				switch (toupper(kin))
				{
					case 'L':	kin = '2'; break;
					case 'S':	kin = '3'; break;
				}
			}

			/* Translate number keys into numeric selection number. */
			if ((!yes_no) && (kin >= '1') && ((kin - '0') <= num_opt))
			{
				selection		= kin - '0';
				got_selection	= TRUE;
			}
			else
			{
				switch(kin)
				{
					case 32:
					case 13:	/* SPACE/RETURN = default */
								if (default_opt != 0)
								{
									selection		= default_opt;
									got_selection	= TRUE;
								}
								else
									beep(); /* no default available */
								break;

					case 3:
					case 0x3b:
					case 0x3c:
					case 0x3d:
					case 0x3e:
					case 0x3f:
					case 0x40:
					case 0x41:
					case 0x42:
					case 0x85:
					case 0x86:	/* Any BREAK or F-key = quit menu */
								got_selection = TRUE;
								break;

					case 'Y':
					case 'y':
					case 'N':
					case 'n':	/* YES/NO type boxes. */
								if (yes_no)
								{
									selection = toupper(kin);
									got_selection = TRUE;
									break;
								}

					default:	/* Unexpected key - beep. */
								beep();
								break;
				}
			}
		}
	}

	if ((num_lines > 11) || (width > 27))
	{
		window(wx1 - 2, wy1 - 2, wx2 + 1, wy2 + 1);
		textattr(BLACK << 4);
		clrscr();
	}
	remove_box();

	return(selection);
}

/* Generic input box function. */
boolean input_box(boolean is_fname,
	unsigned char *dest_ptr,
	unsigned char num_lines,
	unsigned int width, ...)
{
	unsigned char	kin;
	unsigned int	line_len;
	unsigned int	box_width;
	unsigned char	*local_ptr;
	va_list			va_ap;					/* DOSEMU * was global */
	unsigned char	*txt;					/* DOSEMU * was global */
	unsigned char	opt_count;				/* DOSEMU * was global */

	/* Make local copy of default string. */
	if ((local_ptr = strdup(dest_ptr)) == NULL)
		return FALSE;

	if ((box_width = width) >= 21)
		box_width = 21;

	/* Draw and fill box. */
	create_box(num_lines + 1, box_width);
	va_start(va_ap, width);
	for (opt_count = 0; opt_count < num_lines; opt_count++)
	{
		txt = va_arg(va_ap, unsigned char *);
		cprintf("%s\n\r", txt);
	}
	va_end(va_ap);

	/* Print default string. */
	line_len = strlen(dest_ptr);
	textattr((LIGHTGRAY << 4) + BLUE);
	if (line_len > box_width)
		cprintf(&dest_ptr[line_len - box_width]);
	else
		cprintf(dest_ptr);
	textattr((RED << 4) + YELLOW);

	endkey = FALSE;
	putch(' ');
	if (wherex() == 1)
		gotoxy(box_width + 1, wherey());
	else
		gotoxy(wherex() - 1, wherey());
	setcursortype(_NORMALCURSOR); /* DOSEMU * moved */
	switch (kin = getch2(TRUE))
	{
		case 8:		if (line_len > 0)
						dest_ptr[--line_len] = '\0';
					else
						beep(); /* nothing to delete */
					break;

		case 3:
		case 9:
		case 10:
		case 94:
		case 13:	break;

		default:	line_len = 0;
					if (kin == ' ')
					{
						if ((!is_fname) || (!lfn_is95()))
							kin = '_';
					}
					if (isprint(kin))
						dest_ptr[line_len++] = kin;
					dest_ptr[line_len] = '\0';
					break;
	}
	gotoxy(1,wherey());
	for (opt_count = 0; opt_count <= box_width; opt_count++)
		putch(' ');
	if (line_len > box_width)
		cprintf(&dest_ptr[line_len - box_width]);
	else
		cprintf(dest_ptr);

	/* Get rest of input. */
	while ((kin != 13) && ((kin != 3) || endkey))
	{
		endkey = FALSE;
		setcursortype(_NORMALCURSOR); /* DOSEMU * duplicated */
		kin = getch2(TRUE);
		if (isprint(kin)) /* Printable characters. */
		{
			if (line_len < width)
			{
				if (line_len >= box_width)
				{
					gotoxy(1,wherey());
					cprintf(&dest_ptr[1 + (line_len - box_width)]);
				}
				if (kin == ' ')
				{
					if ((!is_fname) || (!lfn_is95()))
						kin = '_';
				}
				putch(kin);
				dest_ptr[line_len++] = kin;
				dest_ptr[line_len] = '\0';
			}
			else
				beep(); /* line too long */
		}
		else if (kin == 8) /* Delete. */
		{
			if (line_len > 0)
			{
				dest_ptr[--line_len] = '\0';
				if (line_len >= box_width)
				{
					gotoxy(1,wherey());
					cprintf(&dest_ptr[line_len - box_width]);
				}
				else
				{
					gotoxy(wherex() - 1, wherey());
					putch(' ');
					gotoxy(wherex() - 1, wherey());
				}
			}
			else
				beep(); /* nothing to delete */
		}
		else if (kin == 12) /* Home key. */
		{
			dest_ptr[line_len = 0] = '\0';
			gotoxy(1,wherey());
			for (opt_count = 0; opt_count <= box_width; opt_count++)
				putch(' ');
		}
	}
	remove_box();

	if (kin == 3)
		strcpy(dest_ptr, local_ptr);

	free(local_ptr);

	return(kin != 3);
}

/* display key list (F1,2) */
void display_key_list(void)
{
	if (selection_box(8,25,0,FALSE,FALSE,FALSE,99,
		"Press Ctrl-Alt Or F8",
		"         For Control Menu","",
		"Press F10",
		"    To Switch To Debugger","",
		"Press Ctrl-Q Or Ctrl-X",
		"         To Quit Emulator") > 0)
	{
		if (selection_box(8,25,0,FALSE,FALSE,FALSE,99,
			"  F1 - Help Menu",
			"  F2 - Virtual Disk Menu", /* DOSEMU */
			"  F3 - File Management",
			"  F4 - Emulator Options",
			"  F5 - Video Mode Menu",
			"  F7 - Function Keys",
			"  F8 - Control Menu",
			"  F9 - Assembler Menu") > 0)
		{
			if (selection_box(8,25,0,FALSE,FALSE,FALSE,99,
				"  Ctrl F1 - Pmode 1",
				"  Ctrl F2 - Pmode 2",
				"  Ctrl F3 - Pmode 3",
				"  Ctrl F4 - Pmode 4",
				"  Ctrl F5 - Palette 0/1",
				"  Ctrl F6 - 2/4 Colours",
				"  Ctrl F7 - Pixel Width",
				"  Ctrl F8 - Text Mode") > 0)
			{
				if (selection_box(8,25,0,FALSE,FALSE,FALSE,99,
					"Press Ctrl-F9",
					"        For Source Editor","",
					"Press Ctrl-F10",
					"          For Info Screen","",
					"Press Shift-F5 To F8",
					"       For Screen Up/Down") > 0)
				{
					if (selection_box(8,25,0,FALSE,FALSE,FALSE,99,
						"       PC-MODE ONLY",
						"       ============","",
						"Press Alt-A To Z",
						"       For BASIC Keywords","",
						"Press Alt-F1 To F10",
						"       For User Functions","") > 0)
					{
						selection_box(8,25,0,FALSE,FALSE,FALSE,99,
							"",
							"Press Ctrl-Return",
							"       For History Buffer",
							"","",
							"Press Shift-F1 To F4",
							"    For Shift+Cursor Keys",
							"");
					}
				}
			}
		}
	}
}

/* directory menu (F1,3) */
void directory_menu(void)
{
	switch (selection_box(9,25,7,FALSE,FALSE,FALSE,99,
						"      DIRECTORY MENU","",
						" 1 - Disk Directory", /* DOSEMU */
						" 2 - Cassette Directory",
						" 3 - Archive Directory",
						" 4 - Games Directory",
						" 5 - Snapshot Directory",
						" 6 - Cartridge Directory",
						" 7 - Base Directory"))
	{
		case 1:	shell_escape(8); break; /* DOSEMU */
		case 2:	shell_escape(2); break;
		case 3:	shell_escape(1); break;
		case 4:	shell_escape(0); break;
		case 5:	shell_escape(5); break;
		case 6:	shell_escape(6); break;
		case 7:	shell_escape(7); break;
	}
}

/* Information screen. */
void inf_screen(void)
{
	double		real_seconds;
	unsigned	int cnt = (80 * 24);

	temp_textmode();

	if (!info_init)
	{
		textattr((LIGHTGRAY << 4) + DARKGRAY);
		while (cnt-- > 0)
		{
			putch(177);
			if ((cnt % 80) == 0)
				putch('\n');
		}
		textattr((LIGHTGRAY << 4) + RED);
		cprintf("                       Press ESC To Return To The Emulator                      ");

		/* Title. */
		window(3,2,78,2);
		textattr((LIGHTGRAY << 4) + BLUE);
		clrscr();
		cprintf("         --= PC-DRAGON V%d.%02d INFORMATION & CONFIGURATION SCREEN =--",VER_MAJOR,VER_MINOR);

		textattr((GREEN << 4) + LIGHTGREEN);
		window(4,4,77,23);
		clrscr();
		border(74,20);

		gettext(1,1,80,25,info_screen);
		info_init = TRUE;
	}
	else
		puttext(1,1,80,25,info_screen);

	textattr((GREEN << 4) + BLACK);
	window(7,6,80,25);
	cprintf("Architecture:    ");
	switch (arch)
	{
		case dragon32:	cprintf("Dragon 32");
						break;

		case dragon64:	if (mapmode1)
							cprintf("Dragon 64 (Map 1)");
						else
							cprintf("Dragon 64 (Map 0)");
						break;

		case tandy:		if (mapmode1)
							cprintf("Tandy CoCo (Map 1)");
						else
							cprintf("Tandy CoCo (Map 0)");
						break;
	}
	cprintf("\n\rROM Breakpoints: %s",
		(install_breakpoints ? "Installed" : "None"));
	cprintf("\n\rSecond ROM:      ");
	if (arch == dragon64)
	{
		if (rom2loaded)
			cprintf("Loaded");
		else
			cprintf("Not Found");
	}
	else
		cprintf("Not Available");
	cprintf("\n\rCartridge Name:  %s", cartname);
	cprintf("\n\rCartridge Area:  $%04X-$%04X", cartstart, cartend);
	cprintf("\n\rComms Ports:     %dP, %dS", bootparallel, bootserial);
	cprintf("\n\rSerial Baud:     ");
	switch (comspeed)
	{
		case _COM_1200:	cprintf("1200"); break;
		case _COM_2400:	cprintf("2400"); break;
		case _COM_4800:	cprintf("4800"); break;
		case _COM_9600:	cprintf("9600"); break;
	}
	cprintf("\n\rPrinter Port:    ");
	if (printfile)
		cprintf("PRINTER.OUT");
	else
	{
		switch (lpt)
		{
			case LPT1:	cprintf("LPT1"); break;
			case LPT2:	cprintf("LPT2"); break;
			case COM1:	cprintf("COM1"); break;
			case COM2:	cprintf("COM2"); break;
			default:	cprintf("None"); break;
		}
	}
	cprintf("\n\rPrinter <CR>:    ");
	if (crlf)
		cprintf("<CR><LF>");
	else
		cprintf("<CR>");
	cprintf("\n\rPrinter Mode:    ");
	if (printsimple)
		cprintf("Basic");
	else
		cprintf("Advanced");
	cprintf("\n\rCycles Per IRQ:  %u",irq_rate);
	cprintf("\n\rIllegal Opcodes: %s",
		(ignore_illegal_opcodes ? "Ignore" : "Pause"));
	cprintf("\n\rSwapping:        ");
	switch (swapping)
	{
		case 0:	cprintf("None");			break;
		case 1:	cprintf("DOS Shell");		break;
		case 2:	cprintf("Editor");			break;
		case 3:	cprintf("Assembler");		break;
	}
	cprintf("\n\rArtifacting:     ");
	switch(get_artifact())
	{
		case 1:		cprintf("Blue Edge");	break;
		case 2:		cprintf("Red Edge");	break;
		default:	cprintf("Off");			break;
	}
	cprintf("\n\rKeyboard Mode:   %s",
		(new_int9_set ? "Hardware Mapped" : "ROM Breakpoint"));

	window(43,6,80,25);
	if ((last_time > 0) && (this_time > 0))
	{
		real_seconds = ((this_time - last_time) / CLK_TCK);
		percent = (should_be_seconds / real_seconds);
	}
	cprintf("Relative Speed:  ");
	if (percent >= 0.0)
		cprintf("%2.1f%%", percent);
	else
		cprintf("Unavailable");
	cprintf("\n\rText Mode:       LC=%s, Gfx=%s",
		(lower_case ? "On" : "Off"),
		(install_udg ? "On" : "Off"));
	cprintf("\n\rCursor:          ");
	if (cursortype == _NOCURSOR)
		cprintf("Off");
	else
	{
		if (cursortype == _NORMALCURSOR)
			cprintf("Underline");
		else
			cprintf("Solid");
	}
	cprintf("\n\rGame Adapter:    %s",
		(hasjoystick ? "Detected" : "Not Found"));
	cprintf("\n\rJoystick Device: %s",
		(realjoy ? "PC Joystick" : "Mouse"));
	cprintf("\n\rJoystick Type:   %s",
		(digital_joysticks ? "Digital" : "Analogue"));
	cprintf("\n\rJoystick Update: %d Reads", hardjoy_resolution);
	cprintf("\n\rMouse Status:    ");
	if (mouse_disabled)
		cprintf("Not Found");
	else
	{
		if (mouse_installed)
			cprintf("Installed");
		else
			cprintf("Not Initialised");
	}
	cprintf("\n\rCassette Status: ");
	if (cassette_disabled)
		cprintf("Failed");
	else if (file_mode != CLOSED)
		cprintf("Busy");
	else
		cprintf("Ready");
	cprintf("\n\rCassette Mode:   ");
	switch (file_mode)
	{
		case READING:	cprintf("Reading"); break;
		case WRITING:	cprintf("Writing"); break;
		case CLOSED:	cprintf("Closed");  break;
	}
	cprintf("\n\rFilename Prompt: %s%s",
		(ask_filenames ? "Yes" : "No"),
		(lfn_is95() ? " (LFN)" : ""));
	cprintf("\n\rSnapshot Boot:   %s", (snapboot ? "Yes" : "No"));
	cprintf("\n\rSnapshot Format: V1.%d%s",
		(version & 0x0f),
		(compress ? " (RLE)" : ""));
	cprintf("\n\rSnapshot Parts:  [");
	if (snap_load_options) putch('O');
	if (load_debug_state) putch('D');
	if (load_extensions) putch('X');
	cprintf("]\n\rSnapshot Save:   $0000-$%04X (",snap_length-1);
	switch (write_port)
	{
		case this_rom:		putch('N');	break;
		case any_dragon:	putch('D');	break;
		case any_rom:		putch('A');	break;
	}
	putch(')');

	window(13,22,80,25);
	textcolor(LIGHTCYAN);
	cprintf("Display Mode:  #%d @ $%04X-$%04X, ",
		new_vmode, new_screen_base, new_screen_end());
	if (new_vmode >= 0)
		cprintf("%d Colours (Screen 1", ((new_four_colours * 2) + 2));
	else
		cprintf("Text Mode (Screen 0");
	cprintf(",%d)", new_palette);

	/* Wait for ESC to be pressed again before returning. */
	if (new_int9_set)
		while (!keys[kESC]);
	else
		while ((getkey() & 0x00ff) != 0x1b);

	/* Set appropriate video mode & refresh. */
	restore_vmode();
	this_time = 0;
}

/* Display help screens (F1). */ /* DOSEMU new */
void help_screen(void)
{
	unsigned char	hstitle1[26+4];
	unsigned char	hstitle2[26+4];
	unsigned char	hscolour[] = {'', '!', BLUE, WHITE, '\0'};

	sprintf(hstitle1,"PC-DRAGON V%d.%02d%s", VER_MAJOR, VER_MINOR, BETA);
	sprintf(hstitle2,"%s%s",hscolour,"                         ");
	strncpy(4+hstitle2+((25>>1)-(strlen(hstitle1)>>1)),
		hstitle1,
		strlen(hstitle1));
	sprintf(hstitle1,"%s%s",hscolour," (C) 1993-99 PAUL BURGIN ");
	switch (selection_box(9,25,4,FALSE,FALSE,FALSE,99,
						hstitle2,
						hstitle1,"",
						"        HELP MENU","",
						" 1 - Reference Manual",
						" 2 - Control Key Summary",
						" 3 - Directory Listing",
						" 4 - Information Screen"))
	{
		case 1:	shell_escape(4);	break;
		case 2:	display_key_list();	break;
		case 3:	directory_menu();	break;
		case 4: inf_screen();		break;
	}
}

/* Generic procedure to enquire whether user */
/* would like to overwrite existing files.   */
outfile_status open_outfile(unsigned char *fname, FILE **fptr,
	unsigned char *dev)
{
	/* Check whether file exists. */
	if (lfn_access(fname,0) == 0)
	{
		if (selection_box(3,20,0,TRUE,FALSE,FALSE,99,
				"File already exists!","",
				"  Overwrite (y/n)?") != 'Y')
			return(OUT_DECLINED);
	}

	/* Try to open file. */
	if ((*fptr = lfn_fopen(fname,"wb")) == NULL)
	{
		if (dev != NULL)
			selection_box(2,14,0,FALSE,FALSE,FALSE,99,
				"Failed to open",
				dev);
		return(OUT_FAILED);
	}
	return(OUT_OPEN);
}

/* Generic function to input a name and open a file. */
boolean generic_open(
	f_m for_message,			 /* type of i/o to be attempted */
	unsigned char *extension,	 /* default extension */
	unsigned char *prompt,		 /* message to appear in input box */
	FILE **fiptr,				 /* pointer to FILE structure */
	unsigned char *devi,		 /* device identifier or fullpath buffer */
	unsigned char *name_default, /* default filename */
	boolean append_first,		 /* TRUE if extension is to be displayed */
	unsigned char *copy_name,	 /* buffer for entered filename */
	boolean must_exist)			 /* TRUE if files must exist */
{
	unsigned char	local_fname[LFN_MAXPATH+4+9] = "virtdisk\\";
	unsigned char	local_fbuff[LFN_MAXPATH+4];
	unsigned char	*for_msg_text = "     FOR DRIVE x"; /* DOSEMU */
	boolean			got_name_ok = TRUE, opened_ok;

	strcpy(local_fbuff,name_default);
	if (append_first)
		strcat(local_fbuff,extension);

	/* DOSEMU++ */
	switch (for_message)
	{
		case READING:	for_msg_text = (unsigned char *)for_reading_text;
						break;
		case WRITING:	for_msg_text = (unsigned char *)for_writing_text;
						break;
		case EDITING:	for_msg_text = (unsigned char *)for_editing_text;
						break;
		case DSKDRIVE1:
		case DSKDRIVE2:
		case DSKDRIVE3:
		case DSKDRIVE4:	for_msg_text[strlen(for_msg_text) - 1]
							= (for_message - DSKDRIVE1) + '1';
						break;
	}
	/* DOSEMU-- */

	while (got_name_ok)
	{
		if (input_box(TRUE,local_fbuff,3,LFN_MAXPATH-5,
			prompt,for_msg_text,""))
		{
			strcpy(&local_fname[9],local_fbuff);

			/* Append extension to filename if necessary. */
			if (strext(&local_fname[9]) == NULL)
				strcat(&local_fname[9],extension);

			/* Open the specified file. */
			switch(for_message)
			{
				case READING:
				case EDITING:

					if (must_exist)
					{
						*fiptr = lfn_fopen(&local_fname[9],"rb");
						opened_ok = (*fiptr != NULL);
						if (!opened_ok)
						{
							selection_box(2,14,0,FALSE,FALSE,FALSE,99,
								"Failed to open",devi);
						}
					}
					else
						opened_ok = TRUE;
					break;

				/* DOSEMU++ */
				case DSKDRIVE1:
				case DSKDRIVE2:
				case DSKDRIVE3:
				case DSKDRIVE4:

					devi[0] = '\0';
open_new_disk:		*fiptr = lfn_fopen(&local_fname[9],"rb");
					opened_ok = (*fiptr != NULL);
					if (opened_ok)
						_lfn_fullpath(devi,&local_fname[9],MAXPATH);
					else
					{
						*fiptr = lfn_fopen(local_fname,"rb");
						opened_ok = (*fiptr != NULL);
						if (opened_ok)
							_lfn_fullpath(devi,local_fname,MAXPATH);
						else
						{
							if (must_exist)
							{
								selection_box(2,14,0,FALSE,FALSE,FALSE,99,
									"Failed to open",
									"  disk file!");
							}
							else
							{
								/* Offer to create a new disk file. */
								if (selection_box(2,18,0,TRUE,FALSE,FALSE,99,
										"Create new virtual",
										" disk file (y/n)?") == 'Y')
								{
									*fiptr = lfn_fopen(&local_fname[9],"wb+");
									opened_ok = (*fiptr != NULL);
									if (opened_ok)
									{
										create_disk(*fiptr);
										fclose(*fiptr);
										must_exist = TRUE;
										goto open_new_disk;
									}
								}
							}
						}
					}
					break;
					/* DOSEMU-- */

				case WRITING:

					opened_ok = (open_outfile(&local_fname[9],fiptr,devi)
									== OUT_OPEN);
					break;
			}

			if (opened_ok)
			{
				if (copy_name != NULL)
					strcpy(copy_name, local_fbuff);
				return(TRUE);
			}
		}
		else
			got_name_ok = FALSE;
	}
	return(FALSE);
}

/* Various functions implementing the options menus.... */

void texterror(void)
{
	selection_box(2,13,0,FALSE,FALSE,FALSE,99,
		"Cannot change",
		"in text mode!");
}

void video_menu(void)
{
	unsigned char	optbox[5][26];
	unsigned char	menukey;
	boolean			double_wide;
	unsigned int	vdg_size;
	unsigned char	sizekey;

	do
	{
		sprintf(optbox[0],"1 - Number Of Colours...%s",
			((new_vmode < 0) ? "9" : (new_four_colours ? "4" : "2")));
		sprintf(optbox[1],"2 - Palette.............%d",new_palette);
		double_wide = (new_vmode < 8);
		sprintf(optbox[2],"3 - Pixels.........%s",
			((new_vmode < 0) ? "...N/A" : (double_wide ? "..Wide" : "Narrow")));
		vdg_size = vdg_sizes[new_vmode+1];
		sprintf(optbox[3],"4 - Memory.....%s%d bytes",
			((vdg_size < 1000) ? "." : ""),vdg_size);
		sprintf(optbox[4],"5 - Base Address....$%04X",new_screen_base);

		switch (menukey = selection_box(7,25,5,FALSE,FALSE,FALSE,99,
			"     VIDEO MODE MENU","",
			optbox[0],optbox[1],optbox[2],
			optbox[3],optbox[4]))
		{
			case 1:	if (new_vmode >= 0)
						toggle_2or4_colours();
					else
						texterror();
					break;

			case 2:	set_vmode(memory[0xff22] ^= 0x08);
					break;

			case 3:	if (new_vmode >= 0)
						toggle_pixel_width();
					else
						texterror();
					break;

			case 4:	if (new_vmode >= 0)
					{
						if (double_wide)
						{
							sizekey = selection_box(9,14,7,FALSE,FALSE,FALSE,99,
								" VIDEO MEMORY","",
								"1 - 512 bytes",
								"2 - 1024 bytes",
								"3 - 2048 bytes",
								"4 - 1536 bytes",
								"5 - 3072 bytes",
								"6 - 3072 bytes",
								"7 - 6144 bytes");
						}
						else
						{
							sizekey = selection_box(9,14,7,FALSE,FALSE,FALSE,99,
								" VIDEO MEMORY","",
								"1 - 512 bytes",
								"2 - 3072 bytes",
								"3 - 2048 bytes",
								"4 - 4608 bytes",
								"5 - 3072 bytes",
								"6 - 9216 bytes",
								"7 - 6144 bytes");
						}
						/* Any selection made? */
						if ((sizekey > 0) && (sizekey < 99))
						{
							new_screen_size = sizekey - 1;
							set_vmode(memory[0xff22]);
						}
					}
					else
						texterror();
					break;

			case 5:	value_box("  BASE ADDRESS", &new_screen_base, "$%04X", FALSE);
					new_screen_base &= 0xfe00;
					if (new_vmode < 0)
						execute_vmode(FALSE);
					break;
		}
	} while ((menukey != 0) && (menukey != 99));
}

void gen_opt(void)
{
	unsigned char optbox[5][26];
	unsigned char menukey;

	do
	{
		sprintf(optbox[0],"1 - Keyboard Mode....%s",(new_int9_set ? "Real" : "..PC"));
		sprintf(optbox[1],"2 - Filename Prompt...%s",(ask_filenames ? "Yes" : ".No"));
		sprintf(optbox[2],"3 - Illegal Ops....%s",(ignore_illegal_opcodes ? "Ignore" : ".Pause"));
		strcpy(optbox[3],"4 - Swapping.......");
		switch (swapping)
		{
			case 0:	strcat(optbox[3],"..None");	break;
			case 1:	strcat(optbox[3],".Shell");	break;
			case 2:	strcat(optbox[3],"Editor");	break;
			case 3:	strcat(optbox[3],".Assem");	break;
		}
		sprintf(optbox[4],"5 - Cycles Per IRQ..%05u",irq_rate);

		switch (menukey = selection_box(7,25,5,FALSE,FALSE,FALSE,99,
			"     GENERAL OPTIONS","",
			optbox[0],optbox[1],optbox[2],
			optbox[3],optbox[4]))
		{
			case 1:	if (new_int9_set)
						Set_Old_Int9();
					new_int9_set = !new_int9_set;
					if (new_int9_set)
						Set_New_Int9();
					break;

			case 2:	ask_filenames = !ask_filenames;
					break;

			case 3:	ignore_illegal_opcodes = !ignore_illegal_opcodes;
					break;

			case 4:	swapping = ((swapping == 3) ? 0 : (swapping + 1));
					break;

			case 5:	value_box(" CYCLES PER IRQ", &irq_rate, "#%d", FALSE);
					if (irq_rate == 0)
						irq_rate = 1;
					should_be_seconds = (double)irq_rate / 89.0;
					break;
		}
	} while ((menukey != 0) && (menukey != 99));
}

void disp_opt(void)
{
	unsigned char optbox[3][26];
	unsigned char menukey;
	unsigned char cur_art;

	do
	{
		sprintf(optbox[0],"1 - Lower Case........%s",(lower_case ? ".On" : "Off"));
		strcpy(optbox[1],"2 - PC Cursor...");
		switch(cursortype)
		{
			case _NORMALCURSOR:	strcat(optbox[1],"Underline");	break;
			case _NOCURSOR:		strcat(optbox[1],"......Off");	break;
			default:			strcat(optbox[1],"....Solid");	break;
		}
		strcpy(optbox[2],"3 - Artifacting......");
		switch(cur_art = get_artifact())
		{
			case 0:	strcat(optbox[2],".Off");	break;
			case 1:	strcat(optbox[2],"Blue");	break;
			case 2:	strcat(optbox[2],".Red");	break;
		}

		switch (menukey = selection_box(5,25,3,FALSE,FALSE,FALSE,99,
			"     DISPLAY OPTIONS","",optbox[0],optbox[1],optbox[2]))
		{
			case 1:	lower_case = !lower_case;
					if (vmode < 0)
						refresh_video();
					break;

			case 2:	switch (cursortype)
					{
						case _NOCURSOR:		cursortype = _NORMALCURSOR;	break;
						case _NORMALCURSOR:	cursortype = _SOLIDCURSOR;	break;
						default:			cursortype = _NOCURSOR; 	break;
					}
					if (vmode < 0)
						setcursortype(cursortype);
					break;

			case 3:	set_artifact((cur_art == 2) ? 0 : (cur_art + 1));
					break;
		}
	} while ((menukey != 0) && (menukey != 99));
}

void print_opt(void)
{
	unsigned char optbox[4][26];
	unsigned char menukey;

	do
	{
		strcpy(optbox[0],"1 - Printer Port.....");
		if (printfile)
			strcat(optbox[0], "File");
		else
		{
			switch (lpt)
			{
				case LPT1:	strcat(optbox[0], "LPT1");	break;
				case LPT2:	strcat(optbox[0], "LPT2");	break;
				case COM1:	strcat(optbox[0], "COM1");	break;
				case COM2:	strcat(optbox[0], "COM2");	break;
				default:	strcat(optbox[0], "None");	break;
			}
		}
		strcpy(optbox[1],"2 - Serial Baud...");
		switch (comspeed)
		{
			case _COM_1200:	strcat(optbox[1],"1200bps");	break;
			case _COM_2400:	strcat(optbox[1],"2400bps");	break;
			case _COM_4800:	strcat(optbox[1],"4800bps");	break;
			default:		strcat(optbox[1],"9600bps");	break;
		}
		sprintf(optbox[2],"3 - Print <CR>...%s",
			(crlf ? "<CR><LF>" : "....<CR>"));
		sprintf(optbox[3],"4 - Mode.........%s",
			(printsimple ? "...Basic" : "Advanced"));

		switch (menukey = selection_box(6,25,4,FALSE,FALSE,FALSE,99,
			"     PRINTER OPTIONS","",optbox[0],optbox[1],optbox[2],optbox[3]))
		{
			case 1:	if (printfile)
					{
						change_printfile(FALSE);
						change_lpt(NO_PORTS);
						numserial = numparallel = 0;
					}
					else
					{
						switch (lpt)
						{
							case NO_PORTS:	numparallel = bootparallel;
											numserial = bootserial;
											if (change_lpt(LPT1))
												break;
							case LPT1:		if (change_lpt(LPT2))
												break;
							case LPT2:		if (change_lpt(COM1))
												break;
							case COM1:		if (change_lpt(COM2))
												break;
							default:		change_printfile(TRUE);
											break;
						}
					}
					break;

			case 2:	switch (comspeed)
					{
						case _COM_1200:	set_comspeed(_COM_2400);	break;
						case _COM_2400:	set_comspeed(_COM_4800);	break;
						case _COM_4800:	set_comspeed(_COM_9600);	break;
						default:		set_comspeed(_COM_1200);	break;
					}
					break;

			case 3:	crlf = !crlf;
					break;

			case 4:	printsimple = !printsimple;
					break;
		}
	} while ((menukey != 0) && (menukey != 99));
}

void joy_opt(void)
{
	unsigned char	optbox[3][26];
	unsigned char	menukey;
	unsigned int	local_hardjoy;

	do
	{
		sprintf(optbox[0],"1 - Device....%s",(realjoy ? "PC Joystick" : "......Mouse"));
		sprintf(optbox[1],"2 - Type.........%s",(digital_joysticks ? ".Digital" : "Analogue"));
		sprintf(optbox[2],"3 - Update.......%02d Reads",hardjoy_resolution);

		switch (menukey = selection_box(5,25,3,FALSE,FALSE,FALSE,99,
			"     JOYSTICK OPTIONS","",optbox[0],optbox[1],optbox[2]))
		{
			case 1:	realjoy = !realjoy;
					if (realjoy && !hasjoystick)
						realjoy = FALSE;
					break;

			case 2:	digital_joysticks = !digital_joysticks;
					break;

			case 3: local_hardjoy = (unsigned int)hardjoy_resolution;
					if (value_box("JOYSTICK  UPDATE", &local_hardjoy, "#%d", FALSE))
					{
						if ((local_hardjoy > 0) && (local_hardjoy < 100))
							hardjoy_resolution = (unsigned char)local_hardjoy;
					}
					break;
		}
	} while ((menukey != 0) && (menukey != 99));
}

void snap_opt(void)
{
	unsigned char optbox[7][26];
	unsigned char menukey;

	do
	{
		sprintf(optbox[0],"1 - Format...........V1.%d", version & 0x0f);
		sprintf(optbox[1],"2 - Save Length.....$%04X", snap_length);
		sprintf(optbox[2],"3 - Save Compressed...%s", (compress ? "Yes" : ".No"));
		strcpy(optbox[3],"4 - Portability....");
		switch(write_port)
		{
			case this_rom:		strcat(optbox[3],"..None");	break;
			case any_dragon:	strcat(optbox[3],"Dragon");	break;
			case any_rom:		strcat(optbox[3],"...Any");	break;
		}
		sprintf(optbox[4],"5 - Load Options......%s", (snap_load_options ? "Yes" : ".No"));
		sprintf(optbox[5],"6 - Load Debugger.....%s", (load_debug_state ? "Yes" : ".No"));
		sprintf(optbox[6],"7 - Load Extensions...%s", (load_extensions ? "Yes" : ".No"));

		switch (menukey = selection_box(9,25,7,FALSE,FALSE,FALSE,99,
			"     SNAPSHOT OPTIONS","",
			optbox[0],optbox[1],optbox[2],optbox[3],
			optbox[4],optbox[5],optbox[6]))
		{
			case 1:	version = ((version == V1_4) ? V1_2 : (version + 1));
					break;

			case 2: value_box("SNAPSHOT  LENGTH", &snap_length, "$%04X", FALSE);
					if ((snap_length == 0) || (snap_length > 0xff00))
						snap_length = 0xff00;
					break;

			case 3:	compress = !compress;
					break;

			case 4:	switch(write_port)
					{
						case this_rom:		if (arch != tandy)
											{
												write_port = any_dragon;
												break;
											}
						case any_dragon:	write_port = any_rom;
											break;
						case any_rom:		write_port = this_rom;
											break;
					}
					break;

			case 5:	snap_load_options	= !snap_load_options;	break;
			case 6:	load_debug_state	= !load_debug_state;	break;
			case 7:	load_extensions		= !load_extensions;		break;
		}
	} while ((menukey != 0) && (menukey != 99));
}

void putoptcol(unsigned char thiscol, boolean isback,
	unsigned char thisdef, unsigned char thisoth)
{
	if ((thisoth | thiscol) == 0)
	{
		textattr(DARKGRAY);
	}
	else
	{
		if (isback)
			textattr((thiscol << 4) | thisoth);
		else
			textattr((thisoth << 4) | thiscol);
	}

	if (thiscol == thisdef)
		putch('*');
	else
		putch(' ');

	cprintf("Col%X", thiscol);

	if (thiscol == thisdef)
		putch('*');
	else
		putch(' ');

	textattr((RED << 4) | YELLOW);
}

unsigned char getoptcolour(unsigned char hiresopt, boolean is_background,
	unsigned char default_selection, unsigned char default_other)
{
	unsigned char	selection				= default_selection;
	unsigned char	kin;
	boolean			got_selection			= FALSE;
	unsigned char	opt_count;				/* DOSEMU * was global */

	/* Draw and fill box. */
	create_box(8, 24);
	switch (hiresopt)
	{
		case 0:	cprintf("   %sGROUND  COLOUR\n\r\n\r",
					(is_background ? "BACK" : "FORE"));
				break;

		case 1:	cprintf("     BORDER  COLOUR\n\r\n\r");
				break;

		case 2:
		case 3:
		case 4:
		case 5:
		case 6:
		case 7:	cprintf("      PIXEL COLOUR\n\r\n\r");
				break;
	}
	for (opt_count = 0; opt_count < 16; opt_count+=4)
	{
		putoptcol(opt_count, is_background, default_selection, default_other);
		putoptcol(opt_count + 1, is_background, default_selection, default_other);
		putoptcol(opt_count + 2, is_background, default_selection, default_other);
		putoptcol(opt_count + 3, is_background, default_selection, default_other);
		cprintf("\n\r");
	}
	cprintf("\n\r    Press 0-9 Or A-F");

	/* Restore original INT9 handler before doing this. */
	if (new_int9_set)
		Set_Old_Int9();

	/* Get user input. */
	while (!got_selection)
	{
		kin = tolower(getch2(TRUE));

		if ((kin >= '0') && (kin <= '9'))
		{
			selection = kin - '0';
			got_selection = TRUE;
		}
		else if ((kin >= 'a') && (kin <= 'f'))
		{
			selection = kin - 87;
			got_selection = TRUE;
		}
		else if ((kin == 3) || (kin == 13) || (kin == 32))
			got_selection = TRUE;
		else	/* Unexpected key - beep. */
			beep();
	}
	remove_box();

	return(selection);
}

void text_cols(void)
{
	unsigned char menukey;

	do
	{
		switch (menukey = selection_box(6,25,4,FALSE,FALSE,FALSE,99,
			"    TEXT MODE OPTIONS","",
			"1 - Screen 0,0 Foreground",
			"2 - Screen 0,0 Background",
			"3 - Screen 0,1 Foreground",
			"4 - Screen 0,1 Background"))
		{
			case 1:	fore_ary[0] = getoptcolour(0,FALSE,fore_ary[0],back_ary[0]);
					break;

			case 2:	back_ary[0] = getoptcolour(0,TRUE,back_ary[0],fore_ary[0]);
					break;

			case 3:	fore_ary[1] = getoptcolour(0,FALSE,fore_ary[1],back_ary[1]);
					break;

			case 4:	back_ary[1] = getoptcolour(0,TRUE,back_ary[1],fore_ary[1]);
					break;
		}
		if (new_vmode < 0)
			refresh_video();
	} while ((menukey != 0) && (menukey != 99));
}

void gethicolour(unsigned char hi_hiresopt,
	unsigned char *hi_default_selection, unsigned char hi_default_other)
{
	unsigned char hi_result = (*hi_default_selection & 0x0f);

	hi_result = getoptcolour(hi_hiresopt, FALSE,
		hi_result, hi_default_other);

	if (hi_result > 0x07)
		hi_result |= 0x30;

	*hi_default_selection = hi_result;
}

void hirescols(boolean hires4cols, unsigned char hirespal)
{
	unsigned char menukey;

	do
	{
		menukey = selection_box(8,19,6,FALSE,FALSE,FALSE,99,
			"   PIXEL COLOURS","",
			"1 - 00 Pixels",
			"2 - 01 Pixels (LHS)",
			"3 - 01 Pixels (RHS)",
			"4 - 10 Pixels (LHS)",
			"5 - 10 Pixels (RHS)",
			"6 - 11 Pixels");

		if ((menukey >= 1) && (menukey <= 6))
			gethicolour(menukey+1,&vmode_ary[hires4cols][hirespal][menukey-1],BLACK);
	} while ((menukey != 0) && (menukey != 99));
}

void chg_bordr(boolean hires4cols, unsigned char hirespal)
{
	gethicolour(1,&border_ary[hires4cols][hirespal],BLACK);
}

void graf_cols(boolean graf_4_cols)
{
	unsigned char menukey;

	do
	{
		switch (menukey = selection_box(6,21,4,FALSE,FALSE,FALSE,99,
			(graf_4_cols ? "   COLOUR GRAPHICS" : " MONOCHROME GRAPHICS"),"",
			"1 - Screen 1,0 Pixels",
			"2 - Screen 1,0 Border",
			"3 - Screen 1,1 Pixels",
			"4 - Screen 1,1 Border"))
		{
			case 1:	hirescols(graf_4_cols,0);	break;
			case 2:	chg_bordr(graf_4_cols,0);	break;
			case 3:	hirescols(graf_4_cols,1);	break;
			case 4:	chg_bordr(graf_4_cols,1);	break;
		}
	} while ((menukey != 0) && (menukey != 99));
}

void colour_opt(void)
{
	unsigned char menukey;

	do
	{
		switch (menukey = selection_box(5,21,3,FALSE,FALSE,FALSE,99,
			"    COLOUR OPTIONS","",
			"1 - Text Mode",
			"2 - 2 Colour Graphics",
			"3 - 4 Colour Graphics"))
		{
			case 1:	text_cols();		break;
			case 2:	graf_cols(FALSE);	break;
			case 3:	graf_cols(TRUE);	break;
		}
	} while ((menukey != 0) && (menukey != 99));
}

/* Options menu (F4). */
void opt_screen(void)
{
	unsigned char optkey;

	do
	{
		switch (optkey = selection_box(8,20,6,FALSE,FALSE,FALSE,99,
			"    OPTIONS MENU","",
			"1 - General Options",
			"2 - Display Options",
			"3 - Printer Options",
			"4 - Joystick Options",
			"5 - Snapshot Options",
			"6 - Colour Options"))
		{
			case 1:	gen_opt();		break;
			case 2:	disp_opt();		break;
			case 3:	print_opt();	break;
			case 4:	joy_opt();		break;
			case 5:	snap_opt();		break;
			case 6:	colour_opt();	break;
		}
	} while ((optkey > 0) && (optkey < 99));
}

/* Function key configuration (F7). */
void function_opt(void)
{
	unsigned char keynum;
	unsigned char keystr[6][26];
	unsigned char phase = 0;
	unsigned char ftmp;
	unsigned char strl;

	do
	{
		for (ftmp = 0; ftmp < 5; ftmp++)
		{
			strl = strlen(presets[ftmp+phase]);
			if (strl < 25)
			{
				memset(keystr[ftmp],' ',24);
				strcpy(&keystr[ftmp][12-(strl>>1)],presets[ftmp+phase]);
			}
			else
			{
				strncpy(keystr[ftmp],presets[ftmp+phase],25);
				keystr[ftmp][25] = '\0';
			}
		}

		sprintf(keystr[5]," FUNCTION KEYS ALT F%d-F%d",phase+1,phase+5);

		switch (keynum = selection_box(9,25,9,FALSE,FALSE,TRUE,99,
			keystr[5],"",
			keystr[0],keystr[1],keystr[2],keystr[3],keystr[4],"",
			"   Press F1-F10 Or A-Z"))
		{
			case '#':	keynum = 135;

			case 0x85:
			case 0x86:	keynum -= 0x42;

			case 0x3b:
			case 0x3c:
			case 0x3d:
			case 0x3e:
			case 0x3f:
			case 0x40:
			case 0x41:
			case 0x42:	keynum -= 0x14;

			case '0':	keynum += 10;
			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
			case '6':
			case '7':
			case '8':
			case '9':	input_box(FALSE,presets[keynum-'1'],2,
							PRESET_LEN," FUNCTION KEY STRING","");
						break;

			case 13:
			case 32:	phase = (5 - phase);
						break;

			default:	keynum = toupper(keynum);
						if ((keynum >= 'A') && (keynum <= 'Z'))
						{
							sprintf(keystr[5],"ALT-%c TEXT",keynum);
							input_box(FALSE,keywords[keynum-'A'],2,
								10,keystr[5],"");
						}
						else
						{
							selection_box(1,21,0,FALSE,FALSE,FALSE,99,
								"Key not programmable!");
						}
			case 3:		break;
		}
	} while (keynum != 3);
}

/* File management menu (F3). */
void file_menu(void)
{
	unsigned char	screenopts[2][26];
	unsigned char	screen_extension[5];

	if (new_vmode < 0)
		strcpy(screen_extension,".dtx");
	else
		sprintf(screen_extension,".d%02d",new_vmode);
	sprintf(screenopts[0],"4 - Load Screenshot [D%c%c]",
		toupper(screen_extension[2]),toupper(screen_extension[3]));
	sprintf(screenopts[1],"5 - Save%s",&screenopts[0][8]);

	switch (selection_box(9,25,7,FALSE,TRUE,FALSE,99,
		"     FILE MANAGEMENT","",
		"1 - Reset Cassette  [CAS]",
		"2 - Load Snapshot   [PAK]",
		"3 - Save Snapshot   [PAK]",
		screenopts[0],
		screenopts[1],
		"6 - Load Cartridge  [DGN]",
		"7 - Load Motorola   [S19]"))
	{
		case 1:	close_files(TRUE);
				break;

		case 2:	/* load snapshot */
				{
					boolean			got_snapname_ok = TRUE;
					boolean			lseb;
					unsigned char	local_snapnam[LFN_MAXPATH];

					if (!safe_to_exit())
					{
						if (exit_yes_no() != 'Y')
							break;
					}
					strcpy(local_snapnam,snapshot_name);
					while (got_snapname_ok)
					{
						if (input_box(TRUE,local_snapnam,3,
							LFN_MAXPATH,"  SNAPSHOT FILENAME",
							for_reading_text,""))
						{
							if (load_snapshot(local_snapnam,&lseb))
							{
								/* Loaded snapshot okay - but only  */
								/* begin new fetch/exec if the 6809 */
								/* registers changed.               */
								if (lseb)
									longjmp(hard_reset,1);
								else
									got_snapname_ok = FALSE;
							}
							else
								selection_box(2,14,0,FALSE,FALSE,FALSE,99,
									"Failed to open",
									"snapshot file!");
						}
						else
							got_snapname_ok = FALSE;
					}
				}
				break;

		case 3:	/* save snapshot */
				{
					boolean			keep_trying = TRUE;
					unsigned char	local_snapnam[LFN_MAXPATH];

					if (!safe_to_exit())
					{
						if (exit_yes_no() != 'Y')
							break;
					}
					strcpy(local_snapnam,snapshot_name);
					while (keep_trying)
					{
						if (input_box(TRUE,local_snapnam,3,
							LFN_MAXPATH,"  SNAPSHOT FILENAME",
							for_writing_text,""))
						{
							if (save_snapshot(local_snapnam))
								keep_trying = FALSE;
						}
						else
							keep_trying = FALSE;
					}
				}
				break;

		case 4: /* load screen shot */
				{
					FILE			*ifp;
					unsigned int	count, s_end = new_screen_end();
					int				inch;

					if (generic_open(READING,screen_extension,
						" SCREENSHOT FILENAME",&ifp,
						" screen file!","screen",TRUE,
						NULL,TRUE))
					{
						count = new_screen_base;
							do
						{
							inch = fgetc(ifp);
							if (inch != EOF)
								far_set_mem8(count++, inch);
						} while ((inch != EOF) && (count <= s_end));

						/* Check file size is correct! */
						inch = fgetc(ifp);
						fclose(ifp);
						if ((inch != EOF) || (count != (s_end + 1)))
						{
							selection_box(3,22,0,FALSE,FALSE,FALSE,99,
								"       WARNING!","",
								"Incorrect file length!");
						}
					}
				}
				break;

		case 5: /* save screen shot */
				{
					FILE			*ofp;
					unsigned int	count, s_end = new_screen_end();

						if (generic_open(WRITING,screen_extension,
						" SCREENSHOT FILENAME",&ofp,
						" screen file!","screen",TRUE,
						NULL,TRUE))
					{
						for (count = new_screen_base;
								count <= s_end;
								count++)
							fputc(far_get_mem8(count),ofp);
							fclose(ofp);
					}
				}
				break;

		case 6: /* load cartridge */
				{
					FILE			*ifp;
					boolean			cart_done = FALSE;
					unsigned char	local_cart[LFN_MAXPATH+4];

					if (!safe_to_exit())
					{
						if (exit_yes_no() != 'Y')
							break;
					}
					strcpy(local_cart,cartfname);
					while (!cart_done)
					{
						if (generic_open(READING,"",
							"   CARTRIDGE IMAGE",&ifp,"",
							local_cart,FALSE,local_cart,FALSE))
						{
							cart_done = load_dgn_file(local_cart,2);
							if (cart_done)
								strcpy(cartfname,local_cart);
							else
								selection_box(2,14,0,FALSE,FALSE,FALSE,99,
									"Failed to load",
									"  cartridge!");
						}
						else
							cart_done = TRUE;
					}
				}
				break;

		case 7: /* load motorola */
				{
					FILE			*ifp;
					boolean			moto_done = FALSE;

					if (!safe_to_exit())
					{
						if (exit_yes_no() != 'Y')
							break;
					}
					while (!moto_done)
					{
						if (generic_open(READING,".s19",
							" MOTOROLA IMAGE FILE",&ifp,
							" image file!",last_moto,FALSE,
							last_moto,TRUE))
						{
							unsigned char mn[LFN_MAXPATH+4];

							fclose(ifp);
							strcpy(mn,last_moto);
							if (strext(mn) == NULL)
								strcat(mn,".s19");
							moto_done = load_motorola(mn);
						}
						else
							moto_done = TRUE;
					}
				}
				break;
	}
}

/* Spawn editor process. */
void call_editor(void)
{
	signed char spawn_return;

	if (new_int9_set)
		Set_Old_Int9();

	box_setup();
	temp_textmode();
	setcursortype(_NORMALCURSOR);

	if (swapping >= 2)
		swap_out();
	spawn_return = spawnlp(P_WAIT,path_editor,path_editor,name_editfile,NULL);
	if (swapping >= 2)
		swap_in();

	setup_screen();
	wx1 = 0;
	remove_box();

	if (spawn_return == -1)
		selection_box(1,22,0,FALSE,FALSE,FALSE,99,"Error invoking editor!");
}

/* Call for external editor/assembler (F9 and Ctrl-F9). */
void edit_key(boolean edit_first)
{
	boolean			exit_menu = FALSE;
	signed char		spawn_return;
	FILE			*op;

	if (edit_first)
		call_editor();

	while (!exit_menu)
	{
		switch (selection_box(7,19,5,FALSE,FALSE,FALSE,9,
			"   ASSEMBLER MENU","",
			"1 - Edit Source",
			"2 - Assemble & Load",
			"3 - Reload Image",
			"4 - View Errors",
			"5 - Change Filename"))
		{
			case 1:	call_editor();
					break;

			case 2:	if ((op = freopen("as9.err","wt",stdout)) == NULL)
					{
						selection_box(2,17,0,FALSE,FALSE,FALSE,99,
							"Error redirecting",
							"assembler output!");
						break;
					}
					if (swapping >= 3)
						swap_out();
					spawn_return = spawnlp(P_WAIT,path_assembler,
						path_assembler,name_editfile,NULL);
					if (swapping >= 3)
						swap_in();
					fclose(op);
					if (spawn_return == -1)
					{
						if (access("as9.err",0) == 0)
							lfn_remove("as9.err");
						selection_box(2,14,0,FALSE,FALSE,FALSE,99,
							"Error invoking",
							"  assembler!");
						break;
					}
					else if (spawn_return != 0)
			case 4:	{
						if (access("as9.err",0) == 0)
						{
							box_setup();
							textview("as9.err"," ASSEMBLER ERRORS ",BROWN);
							setup_screen();
							wx1 = 0;
							remove_box();
						}
						else
							selection_box(1,10,0,FALSE,FALSE,FALSE,99,"No errors!");
						break;
					}
					else
					{
						lfn_remove("as9.err");
						selection_box(2,14,0,FALSE,FALSE,FALSE,99,
							"Assembled with",
							"  no errors!");
					}
			case 3:	{
						unsigned char mname[LFN_MAXPATH+4];

						strcpy(mname,name_editfile);
						if (strext(mname) != NULL)
							*strext(mname) = '\0';
						strcat(mname,".s19");
						if (!load_motorola(mname))
							break;
					}
			case 0:
			case 9:	exit_menu = TRUE;
					break;

			case 5:	generic_open(EDITING,".a09","   SOURCE FILENAME",&op,"",
						name_editfile,FALSE,name_editfile,FALSE);
					if (strext(name_editfile) == NULL)
						strcat(name_editfile,".a09");
					break;
		}
	}
}

/* Asks if user really wants to exit. */ /* DOSEMU */
unsigned char exit_yes_no(void)
{
	unsigned char	return_value;

	/* Local copies for stacking. */
	unsigned char	local_saved_ff22;
	signed char		local_saved_new_vmode;
	unsigned int  	local_saved_new_screen_base;
	unsigned char	local_saved_new_screen_size;
	int				local_wx1, local_wy1, local_wx2, local_wy2;
	unsigned char	local_norm_screen[2000];
	struct text_info tinfo;
	int				last_cursor;
	char			int9flag;

	local_saved_ff22 = saved_ff22;
	local_saved_new_vmode = saved_new_vmode;
	local_saved_new_screen_base = saved_new_screen_base;
	local_saved_new_screen_size = local_saved_new_screen_size;
	local_wx1 = wx1;
	local_wy1 = wy1;
	local_wx2 = wx2;
	local_wy2 = wy2;
	memcpy(local_norm_screen, norm_screen, 2000);
	gettextinfo(&tinfo);
	last_cursor = getcursortype();
	int9flag = issetflag;

	if ((return_value = selection_box(6,20,0,TRUE,FALSE,FALSE,99,
						"      WARNING!","",
						"There may be unsaved",
						" virtual disk data.","",
						"  Continue (y/n)?")) != 'Y')
	{
		/* Need to restore stacked data. */
		saved_ff22 = local_saved_ff22;
		saved_new_vmode = local_saved_new_vmode;
		saved_new_screen_base = local_saved_new_screen_base;
		local_saved_new_screen_size = local_saved_new_screen_size;
		wx1 = local_wx1;
		wy1 = local_wy1;
		wx2 = local_wx2;
		wy2 = local_wy2;
		memcpy(norm_screen, local_norm_screen, 2000);
		window( tinfo.winleft,  tinfo.wintop,
				tinfo.winright, tinfo.winbottom);
		gotoxy(tinfo.curx, tinfo.cury);
		textattr(tinfo.attribute);
		setcursortype(last_cursor);
		if ((int9flag==0) && new_int9_set)
			Set_Old_Int9();
		else if ((int9flag==1) && !new_int9_set)
			Set_New_Int9();
	}

	return(return_value);
}
